"use strict";
const assert_1 = require("@aws-cdk/assert");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/cdk");
const sources = require("../lib");
const test_function_1 = require("./test-function");
module.exports = {
    'sufficiently complex example'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.String
            },
            streamSpecification: dynamodb.StreamViewType.NewImage
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            startingPosition: lambda.StartingPosition.TrimHorizon
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            "PolicyDocument": {
                "Statement": [
                    {
                        "Action": [
                            "dynamodb:DescribeStream",
                            "dynamodb:GetRecords",
                            "dynamodb:GetShardIterator",
                        ],
                        "Effect": "Allow",
                        "Resource": {
                            "Fn::GetAtt": [
                                "TD925BC7E",
                                "StreamArn"
                            ]
                        }
                    },
                    {
                        "Action": "dynamodb:ListStreams",
                        "Effect": "Allow",
                        "Resource": "*"
                    }
                ],
                "Version": "2012-10-17"
            },
            "PolicyName": "FnServiceRoleDefaultPolicyC6A839BF",
            "Roles": [{
                    "Ref": "FnServiceRoleB9001A96"
                }]
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 100,
            "StartingPosition": "TRIM_HORIZON"
        }));
        test.done();
    },
    'specific batch size'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.String
            },
            streamSpecification: dynamodb.StreamViewType.NewImage
        });
        // WHEN
        fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 50,
            startingPosition: lambda.StartingPosition.Latest
        }));
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventSourceMapping', {
            "EventSourceArn": {
                "Fn::GetAtt": [
                    "TD925BC7E",
                    "StreamArn"
                ]
            },
            "FunctionName": {
                "Ref": "Fn9270CBC0"
            },
            "BatchSize": 50,
            "StartingPosition": "LATEST"
        }));
        test.done();
    },
    'fails if batch size < 1'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.String
            },
            streamSpecification: dynamodb.StreamViewType.NewImage
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 0,
            startingPosition: lambda.StartingPosition.Latest
        })), /Maximum batch size must be between 1 and 1000 inclusive \(given 0\)/);
        test.done();
    },
    'fails if batch size > 1000'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const fn = new test_function_1.TestFunction(stack, 'Fn');
        const table = new dynamodb.Table(stack, 'T', {
            partitionKey: {
                name: 'id',
                type: dynamodb.AttributeType.String
            },
            streamSpecification: dynamodb.StreamViewType.NewImage
        });
        // WHEN
        test.throws(() => fn.addEventSource(new sources.DynamoEventSource(table, {
            batchSize: 1001,
            startingPosition: lambda.StartingPosition.Latest
        })), /Maximum batch size must be between 1 and 1000 inclusive \(given 1001\)/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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