"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const record_set_1 = require("./record-set");
const route53_generated_1 = require("./route53.generated");
const util_1 = require("./util");
class HostedZone extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * VPCs to which this hosted zone will be added
         */
        this.vpcs = new Array();
        util_1.validateZoneName(props.zoneName);
        const resource = new route53_generated_1.CfnHostedZone(this, 'Resource', {
            name: props.zoneName + '.',
            hostedZoneConfig: props.comment ? { comment: props.comment } : undefined,
            queryLoggingConfig: props.queryLogsLogGroupArn ? { cloudWatchLogsLogGroupArn: props.queryLogsLogGroupArn } : undefined,
            vpcs: core_1.Lazy.anyValue({ produce: () => this.vpcs.length === 0 ? undefined : this.vpcs })
        });
        this.hostedZoneId = resource.ref;
        this.hostedZoneNameServers = resource.attrNameServers;
        this.zoneName = props.zoneName;
        for (const vpc of props.vpcs || []) {
            this.addVpc(vpc);
        }
    }
    static fromHostedZoneId(scope, id, hostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = hostedZoneId;
            }
            get zoneName() {
                throw new Error(`HostedZone.fromHostedZoneId doesn't support "zoneName"`);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports a hosted zone from another stack.
     */
    static fromHostedZoneAttributes(scope, id, attrs) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = attrs.hostedZoneId;
                this.zoneName = attrs.zoneName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Lookup a hosted zone in the current account/region based on query parameters.
     */
    static fromLookup(scope, id, query) {
        const DEFAULT_HOSTED_ZONE = {
            Id: '/hostedzone/DUMMY',
            Name: query.domainName,
        };
        const response = core_1.ContextProvider.getValue(scope, {
            provider: cxapi.HOSTED_ZONE_PROVIDER,
            dummyValue: DEFAULT_HOSTED_ZONE,
            props: query
        }).value;
        // CDK handles the '.' at the end, so remove it here
        if (response.Name.endsWith('.')) {
            response.Name = response.Name.substring(0, response.Name.length - 1);
        }
        return this.fromHostedZoneAttributes(scope, id, {
            hostedZoneId: response.Id,
            zoneName: response.Name,
        });
    }
    /**
     * Add another VPC to this private hosted zone.
     *
     * @param vpc the other VPC to add.
     */
    addVpc(vpc) {
        this.vpcs.push({ vpcId: vpc.vpcId, vpcRegion: core_1.Stack.of(vpc).region });
    }
}
exports.HostedZone = HostedZone;
/**
 * Create a Route53 public hosted zone.
 *
 * @resource AWS::Route53::HostedZone
 */
class PublicHostedZone extends HostedZone {
    static fromPublicHostedZoneId(scope, id, publicHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = publicHostedZoneId;
            }
            get zoneName() { throw new Error(`cannot retrieve "zoneName" from an an imported hosted zone`); }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        if (props.caaAmazon) {
            new record_set_1.CaaAmazonRecord(this, 'CaaAmazon', {
                zone: this
            });
        }
    }
    addVpc(_vpc) {
        throw new Error('Cannot associate public hosted zones with a VPC');
    }
    /**
     * Adds a delegation from this zone to a designated zone.
     *
     * @param delegate the zone being delegated to.
     * @param opts     options for creating the DNS record, if any.
     */
    addDelegation(delegate, opts = {}) {
        new record_set_1.ZoneDelegationRecord(this, `${this.zoneName} -> ${delegate.zoneName}`, {
            zone: this,
            recordName: delegate.zoneName,
            nameServers: delegate.hostedZoneNameServers,
            comment: opts.comment,
            ttl: opts.ttl,
        });
    }
}
exports.PublicHostedZone = PublicHostedZone;
/**
 * Create a Route53 private hosted zone for use in one or more VPCs.
 *
 * Note that `enableDnsHostnames` and `enableDnsSupport` must have been enabled
 * for the VPC you're configuring for private hosted zones.
 *
 * @resource AWS::Route53::HostedZone
 */
class PrivateHostedZone extends HostedZone {
    static fromPrivateHostedZoneId(scope, id, privateHostedZoneId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.hostedZoneId = privateHostedZoneId;
            }
            get zoneName() { throw new Error(`cannot retrieve "zoneName" from an an imported hosted zone`); }
        }
        return new Import(scope, id);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
        this.addVpc(props.vpc);
    }
}
exports.PrivateHostedZone = PrivateHostedZone;
//# sourceMappingURL=data:application/json;base64,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