"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepFunctionsIntegration = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const integration_1 = require("../integration");
const aws_1 = require("./aws");
/**
 * Options to integrate with various StepFunction API
 */
class StepFunctionsIntegration {
    /**
     * Integrates a Synchronous Express State Machine from AWS Step Functions to an API Gateway method.
     *
     * @example
     *
     *    const stateMachine = new stepfunctions.StateMachine(this, 'MyStateMachine', {
     *       definition: stepfunctions.Chain.start(new stepfunctions.Pass(this, 'Pass')),
     *    });
     *
     *    const api = new apigateway.RestApi(this, 'Api', {
     *       restApiName: 'MyApi',
     *    });
     *    api.root.addMethod('GET', apigateway.StepFunctionsIntegration.startExecution(stateMachine));
     */
    static startExecution(stateMachine, options) {
        jsiiDeprecationWarnings._aws_cdk_aws_apigateway_StepFunctionsExecutionIntegrationOptions(options);
        return new StepFunctionsExecutionIntegration(stateMachine, options);
    }
}
exports.StepFunctionsIntegration = StepFunctionsIntegration;
_a = JSII_RTTI_SYMBOL_1;
StepFunctionsIntegration[_a] = { fqn: "@aws-cdk/aws-apigateway.StepFunctionsIntegration", version: "1.145.0" };
class StepFunctionsExecutionIntegration extends aws_1.AwsIntegration {
    constructor(stateMachine, options = {}) {
        super({
            service: 'states',
            action: 'StartSyncExecution',
            options: {
                credentialsRole: options.credentialsRole,
                integrationResponses: integrationResponse(),
                passthroughBehavior: integration_1.PassthroughBehavior.NEVER,
                requestTemplates: requestTemplates(stateMachine, options),
                ...options,
            },
        });
        this.stateMachine = stateMachine;
    }
    bind(method) {
        const bindResult = super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        this.stateMachine.grantExecution(principal, 'states:StartSyncExecution');
        let stateMachineName;
        if (this.stateMachine instanceof sfn.StateMachine) {
            const stateMachineType = this.stateMachine.stateMachineType;
            if (stateMachineType !== sfn.StateMachineType.EXPRESS) {
                throw new Error('State Machine must be of type "EXPRESS". Please use StateMachineType.EXPRESS as the stateMachineType');
            }
            //if not imported, extract the name from the CFN layer to reach the
            //literal value if it is given (rather than a token)
            stateMachineName = this.stateMachine.node.defaultChild.stateMachineName;
        }
        else {
            //imported state machine
            stateMachineName = `StateMachine-${this.stateMachine.stack.node.addr}`;
        }
        let deploymentToken;
        if (stateMachineName !== undefined && !core_1.Token.isUnresolved(stateMachineName)) {
            deploymentToken = JSON.stringify({ stateMachineName });
        }
        return {
            ...bindResult,
            deploymentToken,
        };
    }
}
/**
 * Defines the integration response that passes the result on success,
 * or the error on failure, from the synchronous execution to the caller.
 *
 * @returns integrationResponse mapping
 */
function integrationResponse() {
    const errorResponse = [
        {
            /**
             * Specifies the regular expression (regex) pattern used to choose
             * an integration response based on the response from the back end.
             * In this case it will match all '4XX' HTTP Errors
             */
            selectionPattern: '4\\d{2}',
            statusCode: '400',
            responseTemplates: {
                'application/json': `{
            "error": "Bad request!"
          }`,
            },
        },
        {
            /**
             * Match all '5XX' HTTP Errors
             */
            selectionPattern: '5\\d{2}',
            statusCode: '500',
            responseTemplates: {
                'application/json': '"error": $input.path(\'$.error\')',
            },
        },
    ];
    const integResponse = [
        {
            statusCode: '200',
            responseTemplates: {
                /* eslint-disable */
                'application/json': [
                    '#set($inputRoot = $input.path(\'$\'))',
                    '#if($input.path(\'$.status\').toString().equals("FAILED"))',
                    '#set($context.responseOverride.status = 500)',
                    '{',
                    '"error": "$input.path(\'$.error\')"',
                    '"cause": "$input.path(\'$.cause\')"',
                    '}',
                    '#else',
                    '$input.path(\'$.output\')',
                    '#end',
                ].join('\n'),
            },
        },
        ...errorResponse,
    ];
    return integResponse;
}
/**
 * Defines the request template that will be used for the integration
 * @param stateMachine
 * @param options
 * @returns requestTemplate
 */
function requestTemplates(stateMachine, options) {
    const templateStr = templateString(stateMachine, options);
    const requestTemplate = {
        'application/json': templateStr,
    };
    return requestTemplate;
}
/**
 * Reads the VTL template and returns the template string to be used
 * for the request template.
 *
 * @param stateMachine
 * @param includeRequestContext
 * @param options
 * @reutrns templateString
 */
function templateString(stateMachine, options) {
    var _b, _c, _d;
    let templateStr;
    let requestContextStr = '';
    const includeHeader = (_b = options.headers) !== null && _b !== void 0 ? _b : false;
    const includeQueryString = (_c = options.querystring) !== null && _c !== void 0 ? _c : true;
    const includePath = (_d = options.path) !== null && _d !== void 0 ? _d : true;
    if (options.requestContext && Object.keys(options.requestContext).length > 0) {
        requestContextStr = requestContext(options.requestContext);
    }
    templateStr = fs.readFileSync(path.join(__dirname, 'stepfunctions.vtl'), { encoding: 'utf-8' });
    templateStr = templateStr.replace('%STATEMACHINE%', stateMachine.stateMachineArn);
    templateStr = templateStr.replace('%INCLUDE_HEADERS%', String(includeHeader));
    templateStr = templateStr.replace('%INCLUDE_QUERYSTRING%', String(includeQueryString));
    templateStr = templateStr.replace('%INCLUDE_PATH%', String(includePath));
    templateStr = templateStr.replace('%REQUESTCONTEXT%', requestContextStr);
    return templateStr;
}
function requestContext(requestContextObj) {
    const context = {
        accountId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.accountId) ? '$context.identity.accountId' : undefined,
        apiId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.apiId) ? '$context.apiId' : undefined,
        apiKey: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.apiKey) ? '$context.identity.apiKey' : undefined,
        authorizerPrincipalId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.authorizerPrincipalId) ? '$context.authorizer.principalId' : undefined,
        caller: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.caller) ? '$context.identity.caller' : undefined,
        cognitoAuthenticationProvider: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoAuthenticationProvider) ? '$context.identity.cognitoAuthenticationProvider' : undefined,
        cognitoAuthenticationType: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoAuthenticationType) ? '$context.identity.cognitoAuthenticationType' : undefined,
        cognitoIdentityId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoIdentityId) ? '$context.identity.cognitoIdentityId' : undefined,
        cognitoIdentityPoolId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.cognitoIdentityPoolId) ? '$context.identity.cognitoIdentityPoolId' : undefined,
        httpMethod: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.httpMethod) ? '$context.httpMethod' : undefined,
        stage: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.stage) ? '$context.stage' : undefined,
        sourceIp: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.sourceIp) ? '$context.identity.sourceIp' : undefined,
        user: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.user) ? '$context.identity.user' : undefined,
        userAgent: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.userAgent) ? '$context.identity.userAgent' : undefined,
        userArn: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.userArn) ? '$context.identity.userArn' : undefined,
        requestId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.requestId) ? '$context.requestId' : undefined,
        resourceId: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.resourceId) ? '$context.resourceId' : undefined,
        resourcePath: (requestContextObj === null || requestContextObj === void 0 ? void 0 : requestContextObj.resourcePath) ? '$context.resourcePath' : undefined,
    };
    const contextAsString = JSON.stringify(context);
    // The VTL Template conflicts with double-quotes (") for strings.
    // Before sending to the template, we replace double-quotes (") with @@ and replace it back inside the .vtl file
    const doublequotes = '"';
    const replaceWith = '@@';
    return contextAsString.split(doublequotes).join(replaceWith);
}
//# sourceMappingURL=data:application/json;base64,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