import os
import sys
import importlib
import functools
import inspect
import asyncio
import nest_asyncio

# 应用 nest_asyncio 补丁，以允许在已经运行的事件循环中再次运行事件循环
nest_asyncio.apply() 

def call_func(func, args=None, kwargs=None):
    if args is None:
        args = ()
    if kwargs is None:
        kwargs = {}

    if inspect.isgeneratorfunction(func):
        return list(func(*args, **kwargs))
    elif inspect.iscoroutinefunction(func):
        coro = func(*args, **kwargs) if inspect.iscoroutinefunction(func) else func
        try:
            # 尝试获取当前运行的事件循环
            loop = asyncio.get_running_loop()
        except RuntimeError:
            # 如果没有运行的事件循环，使用 asyncio.run()
            return asyncio.run(coro)
        else:
            # 如果事件循环已经在运行，直接调度协程并等待结果
            async def run_coro():
                return await coro
            # 在当前事件循环中调度任务并等待完成
            future = asyncio.ensure_future(run_coro())
            while not future.done():
                loop.run_until_complete(asyncio.sleep(0.1))  # 避免阻塞
            return future.result()
    else:
        return func(*args, **kwargs)
    
class ChangeDirectoryAndPath:
    """Context manager to change the current working directory and sys.path."""

    def __init__(self, module_path):
        self.module_path = module_path
        self.old_path = None

    def __enter__(self):
        self.old_path = os.getcwd()
        sys.path.insert(0, self.module_path)
        os.chdir(self.module_path)
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        if self.module_path in sys.path:
            sys.path.remove(self.module_path)
        os.chdir(self.old_path)
        

class ModuleManager:
    """Context manager for handling module imports and sys.modules state."""

    def __init__(self, module_path):
        self.module_path = module_path
        self.original_modules = sys.modules.copy()

    def __enter__(self):
        """Enter the runtime context related to this object."""
        self.change_dir = ChangeDirectoryAndPath(self.module_path)
        self.change_dir.__enter__()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """Exit the runtime context related to this object."""
        self.change_dir.__exit__(exc_type, exc_value, traceback)
        self.cleanup_modules()

    def cleanup_modules(self):
        """Restore the original sys.modules state."""
        importlib.invalidate_caches()
        for key in list(sys.modules.keys()):
            if key not in self.original_modules:
                del sys.modules[key]
                

def wrapped_func(func, module_path):

    def _wrapped(*args, **kwargs):
        try:
            print(f"调用了 {func.__name__}")
        except:
            print(f"called unknown")
        with ChangeDirectoryAndPath(module_path):
            try:
                result = call_func(func, args, kwargs)
            except Exception as e:
                print(f"调用 {func.__name__} 时发生错误: {str(e)}")
                raise
        
        print(f"{func.__name__} 调用完毕,结果为:", result)

        return result
    
    wrapped_function = functools.wraps(func)(_wrapped)
    return wrapped_function

