# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['starsessions', 'starsessions.stores']

package_data = \
{'': ['*']}

install_requires = \
['itsdangerous>=2.0.1,<3.0.0', 'starlette>=0,<1']

extras_require = \
{'redis': ['redis>=4.2.0rc1']}

setup_kwargs = {
    'name': 'starsessions',
    'version': '2.1.1',
    'description': 'Advanced sessions for Starlette and FastAPI frameworks',
    'long_description': '## Starsessions\n\nAdvanced sessions for Starlette and FastAPI frameworks\n\n![PyPI](https://img.shields.io/pypi/v/starsessions)\n![GitHub Workflow Status](https://img.shields.io/github/workflow/status/alex-oleshkevich/starsessions/Lint%20and%20test)\n![GitHub](https://img.shields.io/github/license/alex-oleshkevich/starsessions)\n![Libraries.io dependency status for latest release](https://img.shields.io/librariesio/release/pypi/starsessions)\n![PyPI - Downloads](https://img.shields.io/pypi/dm/starsessions)\n![GitHub Release Date](https://img.shields.io/github/release-date/alex-oleshkevich/starsessions)\n\n## Installation\n\nInstall `starsessions` using PIP or poetry:\n\n```bash\npip install starsessions\n# or\npoetry add starsessions\n```\n\nUse `redis` extra for [Redis support](#redis).\n\n## Quick start\n\nSee example application in [`examples/`](examples) directory of this repository.\n\n## Usage\n\n1. Add `starsessions.SessionMiddleware` to your application to enable session support,\n2. Configure session store and pass it to the middleware,\n3. Load session in your view/middleware by calling `load_session(connection)` utility.\n\n```python\nfrom starlette.applications import Starlette\nfrom starlette.middleware import Middleware\nfrom starlette.responses import JSONResponse\nfrom starlette.routing import Route\n\nfrom starsessions import CookieStore, load_session, SessionMiddleware\n\n\nasync def index_view(request):\n    await load_session(request)\n\n    session_data = request.session\n    return JSONResponse(session_data)\n\n\nsession_store = CookieStore(secret_key=\'TOP SECRET\')\n\napp = Starlette(\n    middleware=[\n        Middleware(SessionMiddleware, store=session_store, lifetime=3600 * 24 * 14),\n    ],\n    routes=[\n        Route(\'/\', index_view),\n    ]\n)\n```\n\n### Cookie security\n\nBy default, the middleware uses strict defaults.\nThe cookie lifetime is limited to the browser session and sent via HTTPS protocol only.\nYou can change these defaults by changing `cookie_https_only` and `lifetime` arguments:\n\n```python\nfrom starlette.middleware import Middleware\n\nfrom starsessions import CookieStore, SessionMiddleware\n\nsession_store = CookieStore(secret_key=\'TOP SECRET\')\n\nmiddleware = [\n    Middleware(SessionMiddleware, store=session_store, cookie_https_only=False, lifetime=3600 * 24 * 14),\n]\n```\n\nThe example above will let session usage over insecure HTTP transport and the session lifetime will be set to 14 days.\n\n### Loading session\n\nThe session data is not loaded by default. Call `load_session` to load data from the store.\n\n```python\nasync def index_view(request):\n    await load_session(request)\n    request.session[\'key\'] = \'value\'\n```\n\nHowever, if you try to access uninitialized session, `SessionNotLoaded` exception will be raised.\n\n```python\nasync def index_view(request):\n    request.session[\'key\'] = \'value\'  # raises SessionNotLoaded\n```\n\nYou can automatically load session by using `SessionAutoloadMiddleware` middleware.\n\n### Session autoload\n\nFor performance reasons session is not autoloaded by default. Sometimes it is annoying to call `load_session` too often.\nWe provide `SessionAutoloadMiddleware` to reduce amount of boilerplate code by autoloading session for you.\n\nThere are two options: always autoload or autoload for specific paths only.\nHere are examples:\n\n```python\nfrom starlette.middleware import Middleware\n\nfrom starsessions import CookieStore, SessionAutoloadMiddleware, SessionMiddleware\n\nsession_store = CookieStore(secret_key=\'TOP SECRET\')\n\n# Always autoload\n\nmiddleware = [\n    Middleware(SessionMiddleware, store=session_store),\n    Middleware(SessionAutoloadMiddleware),\n]\n\n# Autoload session for selected paths\n\nmiddleware = [\n    Middleware(SessionMiddleware, store=session_store),\n    Middleware(SessionAutoloadMiddleware, paths=[\'/admin\', \'/app\']),\n]\n\n# regex patterns also supported\nimport re\n\nadmin_rx = re.compile(\'/admin*\')\n\nmiddleware = [\n    Middleware(SessionMiddleware, store=session_store),\n    Middleware(SessionAutoloadMiddleware, paths=[admin_rx]),\n]\n```\n\n### Rolling sessions\n\nThe default behavior of `SessionMiddleware` is to expire cookie after `lifetime` seconds after it was set.\nFor example, if you create a session with `lifetime=3600` then the session will be terminated exactly in 3600 seconds.\nSometimes this may not be what you need, so we provide alternate expiration strategy - rolling sessions.\n\nWhen rolling sessions in use, the cookie expiration time will be extended by `lifetime` value on every response.\nLet\'s see how it works on example. First, on the first response you create a new session with `lifetime=3600`,\nthen user does another request and session gets extended by another 3600 seconds and so on.\nThis approach is useful when you want to have short-timed sessions but don\'t want them to interrupt in the middle of\nuser\'s operation. With rolling strategy, session cookie will be expired only after some period of user\'s inactivity.\n\nTo enable rolling strategy set `rolling=True`.\n\n```python\nfrom starlette.middleware import Middleware\nfrom starsessions import SessionMiddleware\n\nmiddleware = [\n    Middleware(SessionMiddleware, lifetime=300, rolling=True),\n]\n```\n\nThe snippet above demonstrates an example setup where session will be dropped after 300 seconds (5 minutes) of\ninactivity, but will be automatically extended by another 5 minutes while the user is online.\n\n### Cookie path\n\nYou can pass `cookie_path` argument to bind session cookie to specific URLs. For example, to activate session cookie\nonly for admin area, use `cookie_path="/admin"` middleware argument.\n\n```python\nfrom starlette.middleware import Middleware\nfrom starsessions import SessionMiddleware\n\nmiddleware = [\n    Middleware(SessionMiddleware, cookie_path=\'/admin\'),\n]\n```\n\nAll other URLs not matching value of `cookie_path` will not receive cookie thus session will be unavailable.\n\n### Cookie domain\n\nYou can also specify which hosts can receive a cookie by passing `cookie_domain` argument to the middleware.\n\n```python\nfrom starlette.middleware import Middleware\nfrom starsessions import SessionMiddleware\n\nmiddleware = [\n    Middleware(SessionMiddleware, cookie_domain=\'example.com\'),\n]\n```\n\n> Note, this makes session cookie available for subdomains too.\n> For example, when you set `cookie_domain=example.com` then session cookie will be available on subdomains\n> like `app.example.com`.\n\n### Session-only cookies\n\nIf you want session cookie to automatically remove from tbe browser when tab closes then set `lifetime` to `0`.\n> Note, this depends on browser implementation!\n\n```python\nfrom starlette.middleware import Middleware\nfrom starsessions import SessionMiddleware\n\nmiddleware = [\n    Middleware(SessionMiddleware, lifetime=0),\n]\n```\n\n## Built-in stores\n\n### Memory\n\nClass: `starsessions.InMemoryStore`\n\nSimply stores data in memory. The data is cleared after server restart. Mostly for use with unit tests.\n\n### CookieStore\n\nClass: `starsessions.CookieStore`\n\nStores session data in a signed cookie on the client.\n\n### Redis\n\nClass: `starsessions.stores.redis.RedisStore`\n\nStores session data in a Redis server. The store accepts either connection URL or an instance of `Redis`.\n\n> Requires [redis-py](https://github.com/redis/redis-py),\n> use `pip install starsessions[redis]` or `poetry add starsessions[redis]`\n\n```python\nfrom redis.asyncio.utils import from_url\n\nfrom starsessions.stores.redis import RedisStore\n\nstore = RedisStore(\'redis://localhost\')\n# or\nredis = from_url(\'redis://localhost\')\n\nstore = RedisStore(connection=redis)\n```\n\n#### Redis key prefix\n\nBy default, all keys in Redis prefixed with `starsessions.`. If you want to change this use `prefix` argument.\n\n```python\nfrom starsessions.stores.redis import RedisStore\n\nstore = RedisStore(url=\'redis://localhost\', prefix=\'my_sessions\')\n```\n\nPrefix can be a callable:\n\n```python\nfrom starsessions.stores.redis import RedisStore\n\n\ndef make_prefix(key: str) -> str:\n    return \'my_sessions_\' + key\n\n\nstore = RedisStore(url=\'redis://localhost\', prefix=make_prefix)\n```\n\n#### Key expiration\n\nThe library automatically manages key expiration, usually you have nothing to do with it.\nBut for cases when `lifetime=0` we don\'t know when the session will over, and we have to heuristically calculate TTL\notherwise the data will remain in Redis forever. At this moment, we just set 30 days TTL. You can change it by\nsetting `gc_ttl` value on the store.\n\n```python\nfrom starsessions.stores.redis import RedisStore\n\nstore = RedisStore(url=\'redis://localhost\', gc_ttl=3600)  # max 1 hour\n```\n\n## Custom store\n\nCreating new stores is quite simple. All you need is to extend `starsessions.SessionStore`\nclass and implement abstract methods.\n\nHere is an example of how we can create a memory-based session store. Note, it is important that `write` method\nreturns session ID as a string value.\n\n```python\nfrom typing import Dict\n\nfrom starsessions import SessionStore\n\n\n# instance of class which manages session persistence\n\nclass InMemoryStore(SessionStore):\n    def __init__(self):\n        self._storage = {}\n\n    async def read(self, session_id: str, lifetime: int) -> Dict:\n        """ Read session data from a data source using session_id. """\n        return self._storage.get(session_id, {})\n\n    async def write(self, session_id: str, data: Dict, lifetime: int, ttl: int) -> str:\n        """ Write session data into data source and return session id. """\n        self._storage[session_id] = data\n        return session_id\n\n    async def remove(self, session_id: str):\n        """ Remove session data. """\n        del self._storage[session_id]\n\n    async def exists(self, session_id: str) -> bool:\n        return session_id in self._storage\n```\n\n### lifetime and ttl\n\nThe `write` accepts two special arguments: `lifetime` and `ttl`.\nThe difference is that `lifetime` is a total session duration (set by the middleware)\nand `ttl` is a remaining session time. After `ttl` seconds the data can be safely deleted from the storage.\n\n> Your custom backend has to correctly handle setups when `lifetime = 0`.\nIn such cases you don\'t have exact expiration value, and you have to find a way how to extend session TTL at the storage\nside, if any.\n\n## Serializers\n\nThe library automatically serializes session data to string using JSON.\nBy default, we use `starsessions.JsonSerializer` but you can implement your own by extending `starsessions.Serializer`\nclass.\n\n```python\nimport json\nimport typing\n\nfrom starlette.middleware import Middleware\n\nfrom starsessions import Serializer, SessionMiddleware\n\n\nclass MySerializer(Serializer):\n    def serialize(self, data: typing.Any) -> bytes:\n        return json.dumps(data).encode(\'utf-8\')\n\n    def deserialize(self, data: bytes) -> typing.Dict[str, typing.Any]:\n        return json.loads(data)\n\n\nmiddleware = [\n    Middleware(SessionMiddleware, serializer=MySerializer()),\n]\n```\n\n## Session termination\n\nThe middleware will remove session data and cookie if session has no data. Use `request.session.clear` to empty data.\n\n## Regenerating session ID\n\nSometimes you need a new session ID to avoid session fixation attacks (for example, after successful signs in).\nFor that, use `starsessions.session.regenerate_session_id(connection)` utility.\n\n```python\nfrom starsessions.session import regenerate_session_id\nfrom starlette.responses import Response\n\n\ndef login(request):\n    regenerate_session_id(request)\n    return Response(\'successfully signed in\')\n```\n',
    'author': 'alex.oleshkevich',
    'author_email': 'alex.oleshkevich@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/alex-oleshkevich/starsessions',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8.0,<4.0.0',
}


setup(**setup_kwargs)
