import re
from pathlib import Path

import argparse
import numpy as np


def get_images_from_export_file(export_file_path: str) -> [str]:
    with open(export_file_path) as file:
        file_content = file.read()
        normal_images = re.findall('"__GHOST_URL__/(content/images/[0-9]{4}/[0-9]{2}/[-\w.]+.[a-zA-Z]{2,4})', file_content)
        resized_images = re.findall('"__GHOST_URL__/(content/images/size/w[a-zA-Z0-9]+/[0-9]{4}/[0-9]{2}/[-\w.]+.[a-zA-Z]{2,4}) ', file_content)
        return normal_images + resized_images


def get_images_from_folder(ghost_install_dir: str) -> [str]:
    images_dir = f"{ghost_install_dir}/content/images/"
    posix_paths = list(Path(images_dir).rglob("*"))
    return list(map(lambda posix_path: str(posix_path).replace(ghost_install_dir.rstrip('/') + "/", ''), posix_paths))


def get_unused_images(images_in_dir: [str], images_in_export: [str]) -> [str]:
    return np.setdiff1d(images_in_dir, images_in_export)


def get_absent_images(images_in_dir: [str], images_in_export: [str]) -> [str]:
    return np.setdiff1d(images_in_export, images_in_dir)


def execute():
    parser = argparse.ArgumentParser(
        prog="cleanghostimages",
        description="A tool to find un-used images stored by your Ghost blog installation."
    )
    parser.add_argument("--json-export-file", type=str, required=True, help="The json export generated by Ghost")
    parser.add_argument("--ghost-dir", type=str, required=True, help="Your Ghost installation directory")
    parser.add_argument("--print-missing-images", type=bool, action=argparse.BooleanOptionalAction)
    parser.add_argument("--statistics", type=bool, action=argparse.BooleanOptionalAction, help="Displays information about the number of images.")
    parser.add_argument("--print-unused-images", type=bool, action=argparse.BooleanOptionalAction)
    args = parser.parse_args()

    images_in_export = get_images_from_export_file(args.json_export_file)
    images_in_dir = get_images_from_folder(args.ghost_dir)

    unused_files = get_unused_images(images_in_dir, images_in_export)
    absent_files = get_absent_images(images_in_dir, images_in_export)

    if args.statistics:
        print(f"Found a total of {len(images_in_export)} images in the export json.")
        print(f"Found a total of {len(images_in_dir)} images in the ghost content directory.")
        print(f"{len(unused_files)} images are not used, and {len(absent_files)} are missing.")

    if args.print_missing_images:
        print("The following images are missing:")
        for image in absent_files:
            print(image)

    if args.print_unused_images:
        print("The following images are not used:")
        for image in unused_files:
            print(image)
    return 0


if __name__ == "__main__":
    execute()
