"""Trajectory interpolation utility module.

Provides two main interpolation methods:
- linear
- minimum jerk
"""

from enum import Enum
from typing import Callable, Optional

import numpy as np


InterpolationFunc = Callable[[float], np.ndarray]


def linear(
    starting_position: np.ndarray,
    goal_position: np.ndarray,
    duration: float,
) -> InterpolationFunc:
    """Compute the linear interpolation function from starting position to goal position."""
    def f(t: float) -> np.ndarray:
        return starting_position + (goal_position - starting_position) * t / duration
    return f


def minimum_jerk(
    starting_position: np.ndarray,
    goal_position: np.ndarray,
    duration: float,
    starting_velocity: Optional[np.ndarray] = None,
    starting_acceleration: Optional[np.ndarray] = None,
    final_velocity: Optional[np.ndarray] = None,
    final_acceleration: Optional[np.ndarray] = None,
) -> InterpolationFunc:
    """Compute the mimimum jerk interpolation function from starting position to goal position."""
    if starting_velocity is None:
        starting_velocity = np.zeros(starting_position.shape)
    if starting_acceleration is None:
        starting_acceleration = np.zeros(starting_position.shape)
    if final_velocity is None:
        final_velocity = np.zeros(goal_position.shape)
    if final_acceleration is None:
        final_acceleration = np.zeros(goal_position.shape)

    a0 = starting_position
    a1 = starting_velocity
    a2 = starting_acceleration / 2

    d1, d2, d3, d4, d5 = [duration ** i for i in range(1, 6)]

    A = np.array((
        (d3, d4, d5),
        (3 * d2, 4 * d3, 5 * d4),
        (6 * d1, 12 * d2, 20 * d3)
    ))
    B = np.array((
        goal_position - a0 - (a1 * d1) - (a2 * d2),
        final_velocity - a1 - (2 * a2 * d1),
        final_acceleration - (2 * a2)
    ))
    X = np.linalg.solve(A, B)

    coeffs = [a0, a1, a2, X[0], X[1], X[2]]

    def f(t: float) -> np.ndarray:
        return np.sum([
            c * t ** i
            for i, c in enumerate(coeffs)
        ], axis=0)

    return f


class InterpolationMode(Enum):
    """Inteprolation Mode enumeration."""

    LINEAR: Callable[[np.ndarray, np.ndarray, float], InterpolationFunc] = linear
    MINIMUM_JERK: Callable[[np.ndarray, np.ndarray, float], InterpolationFunc] = minimum_jerk
