# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['frelatage',
 'frelatage.config',
 'frelatage.dictionary',
 'frelatage.input',
 'frelatage.mutator',
 'frelatage.queue',
 'frelatage.report',
 'frelatage.tracer']

package_data = \
{'': ['*']}

install_requires = \
['timeout-decorator>=0.5.0,<0.6.0']

setup_kwargs = {
    'name': 'frelatage',
    'version': '0.0.3',
    'description': 'Frelatage is a coverage-based Python fuzzing library which can be used to fuzz python code.',
    'long_description': '<p align=center>\n  <img src="doc/frelatage_logo.gif" width="200" height="200" style="border-radius:4px"/>\n  <br>\n  <code>pip3 install frelatage</code></br>\n  <i>Current release : <a href="https://github.com/Rog3rSm1th/Frelatage/releases">0.0.3</a></i></br></br>\n  <a target="_blank" href="https://www.python.org/downloads/" title="Python version"><img src="https://img.shields.io/badge/Made%20with-Python-1f425f.svg"></a>\n  <a target="_blank" href="https://www.python.org/downloads/" title="Python version"><img src="https://img.shields.io/badge/python-%3E=_3.6-green.svg"></a>\n  <a target="_blank" href="LICENSE" title="License: MIT"><img src="https://img.shields.io/badge/License-MIT-blue.svg"></a>\n  <a target="_blank" title="Downloads"><img src="https://static.pepy.tech/badge/frelatage"></a>\n  <a target="_blank" href="https://twitter.com/Rog3rSm1th" title="Twitter"><img src="https://img.shields.io/badge/-@Rog3rSm1th-1ca0f1?style=flat-square&labelColor=1ca0f1&logo=twitter&logoColor=white&link=https://twitter.com/Rog3rSm1th"></a>\n  <br>\n  <span><i>The Python Fuzzer that the world deserves</i></span>\n</p>\n\n<p align="center">\n  <a href="#installation">Installation</a>\n  &nbsp;&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;\n  <a href="#how-it-works">How it works</a>\n  &nbsp;&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;\n  <a href="#features">Features</a>\n  &nbsp;&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;\n  <a href="#use-frelatage">Use Frelatage</a>\n  &nbsp;&nbsp;&nbsp;|&nbsp;&nbsp;&nbsp;\n  <a href="#configuration">Configuration</a>\n</p>\n\n<p align="center">\n  <img src="https://github.com/Rog3rSm1th/Frelatage/blob/main/doc/frelatage_demo.gif?raw=true" alt="Frelatage demonstration"/>\n</p>\n\nFrelatage is a coverage-based Python fuzzing library which can be used to fuzz python code. The development of Frelatage was inspired by various other fuzzers, including [AFL](https://github.com/google/AFL)/[AFL++](https://github.com/AFLplusplus/AFLplusplus), [Atheris](https://github.com/google/atheris) and [PyFuzzer](https://github.com/eerimoq/pyfuzzer).The main purpose of the project is to take advantage of the best features of these fuzzers and gather them together into a new tool in order to efficiently fuzz python applications.\n\n**DISCLAIMER** : This project is at the alpha stage and can still cause many unexpected behaviors. Frelatage should not be used in a production environment at this time.\n\n## Requirements\n[Python 3](https://www.python.org/)\n\n## Installation\n\n#### Install with pip (recommended)\n\n```bash\npip3 install frelatage\n```\n\n#### Or build from source\n\nRecommended for developers. It automatically clones the main branch from the frelatage repo, and installs from source.\n\n```bash\n# Automatically clone the Frelatage repository and install Frelatage from source\nbash <(wget -q https://raw.githubusercontent.com/Rog3rSm1th/Frelatage/main/scripts/autoinstall.sh -O -)\n```\n\n## How it works\n\nThe idea behind the design of Frelatage is the usage of a genetic algorithm to generate mutations that will cover as much code as possible. The functioning of a fuzzing cycle can be roughly summarized with this diagram : \n\n```mermaid\ngraph TB\n\n    m1(Mutation 1) --> |input| function(Fuzzed function)\n    m2(Mutation 2) --> |input| function(Fuzzed function)\n    mplus(Mutation ...) --> |input| function(Fuzzed function)\n    mn(Mutation n) --> |input| function(Fuzzed function)\n    \n    function --> generate_reports(Generate reports)\n    generate_reports --> rank_reports(Rank reports)  \n    rank_reports --> select(Select n best reports)\n    \n    select --> |mutate| nm1(Mutation 1) & nm2(Mutation 2) & nmplus(Mutation ...) & nmn(Mutation n)\n    \n    subgraph Cycle mutations\n    direction LR\n    m1\n    m2\n    mplus\n    mn\n    end\n    \n    subgraph Next cycle mutations\n    direction LR\n    nm1\n    nm2\n    nmplus\n    nmn\n    end\n     \n    style function fill:#5388e8,stroke:white,stroke-width:4px\n```\n## Features\n\n#### Fuzzing different argument types: \n  - String\n  - Int\n  - Float\n  - List\n  - Tuple\n  - Dictionary\n  \n#### File fuzzing\nFrelatage allows to fuzz a function by passing a file as input. \n\n#### Fuzzer efficiency\n- Corpus\n- Dictionnary\n\n## Use Frelatage\n\n#### Fuzz a classical parameter\n\n```python\nimport frelatage\nimport my_vulnerable_library\n\ndef MyFunctionFuzz(data):\n  my_vulnerable_library.parse(data)\n\ninput = frelatage.Input(value="initial_value")\nf = frelatage.Fuzzer(MyFunctionFuzz, [[input]])\nf.fuzz()\n```\n\n#### Fuzz a file parameter\n\nFrelatage gives you the possibility to fuzz file type input parameters. To initialize the value of these files, you must create files in the input folder (```./in``` by default).\n\nIf we want to initialize the value of a file used to fuzz, we can do it like this:\n```bash\necho "initial value" > ./in/input.txt\n```\n\nAnd then run the fuzzer: \n\n```python\nimport frelatage\nimport my_vulnerable_library\n\ndef MyFunctionFuzz(data):\n  my_vulnerable_library.load_file(data)\n\ninput = frelatage.Input(file=True, value="input.txt")\nf = frelatage.Fuzzer(MyFunctionFuzz, [[input]])\nf.fuzz()\n```\n\n#### Fuzz with a dictionary\n\nYou can copy one or more dictionaries located [here](https://github.com/Rog3rSm1th/Frelatage/tree/main/dictionaries) in the directory dedicated to dictionaries (`./dict` by default).\n\n#### Examples\n\nYou can find more examples of fuzzers and corpus in the [examples directory](https://github.com/Rog3rSm1th/Frelatage/tree/main/examples).\n\n## Reports\n\nEach crash is saved in the output folder (```./out``` by default), in a folder named : ```id<crash ID>,err<error type>```.\n\nThe report directory is in the following form: \n```\n    ├── out\n    │   ├── id<crash ID>,err<error type>\n    │       ├── input\n    │       ├── 0\n    │       └── ...\n    │   ├── ...\n```\n\n#### Read a crash report\n\nInputs passed to a function are serialized using the [pickle](https://docs.python.org/3/library/pickle.html) module before being saved in the ```<report_folder>/input file```. It is therefore necessary to deserialize it to be able to read the contents of the file. This action can be performed with [this script](https://github.com/Rog3rSm1th/Frelatage/blob/main/scripts/read_report.py). \n\n```bash\n./read_report.py input\n```\n  \n## Configuration\n\nThere are two ways to set up Frelatage:\n\n#### Using the environment variables\n\n| ENV Variable                   | Description | Possible Values | Default Value |\n| -------------------------------| ----------- |--------|-------|\n| FRELATAGE_DICTIONARY_ENABLE   | Enable the use of mutations based on dictionary elements| ```1``` to enable, ```0``` otherwise | ```1``` |\n| FRELATAGE_TIMEOUT_DELAY        | Delay in seconds after which a function will return a TimeoutError | ```1``` - ```20``` | ```2``` |\n| FRELATAGE_INPUT_FILE_TMP_DIR   | Temporary folder where input files are stored | absolute path to a folder, e.g. ```/tmp/custom_dir```| ```/tmp/frelatage```|\n| FRELATAGE_INPUT_MAX_LEN        | Maximum size of an input variable in bytes | ```4``` - ```1000000``` |  ```4094``` |\n| FRELATAGE_MAX_THREADS          | Maximum number of simultaneous threads | ```8``` - ```50``` | ```8``` |\n| FRELATAGE_DICTIONARY_DIR      | Default directory for dictionaries. It needs to be a relative path (to the path of the fuzzing file) | relative path to a folder, e.g. ```./dict```  | ```./dict``` |  \n| FRELATAGE_MAX_CYCLES_WITHOUT_NEW_PATHS      | Number of cycles without new paths found after which we go to the next stage | ```10``` - ```50000``` | ```5000``` | \n\n\nA configuration example :\n\n```bash\nexport FRELATAGE_DICTIONARY_ENABLE=1 &&\nexport FRELATAGE_TIMEOUT_DELAY=2 &&\nexport FRELATAGE_INPUT_FILE_TMP_DIR="/tmp/frelatage" &&\nexport FRELATAGE_INPUT_MAX_LEN=4096 &&\nexport FRELATAGE_MAX_THREADS=8 &&\nexport FRELATAGE_DICTIONARY_DIR="./dict" &&\nexport FRELATAGE_MAX_CYCLES_WITHOUT_NEW_PATHS=5000 &&\npython3 fuzzer.py\n```\n\n#### Passing arguments to the fuzzer \n\n```python\nimport frelatage \n\ndef myfunction(input1_string, input2_int):\n    pass\n\ninput1 = frelatage.Input(value="initial_value")\ninput2 = frelatage.Input(value=2)\n\nf = frelatage.Fuzzer(\n    # The method you want to fuzz\n    method=myfunction,\n    # Corpus\n    corpus=[[input1], [input2]],\n    # Number of threads\n    threads_count=8,\n    # Exceptions that will be taken into account\n    exceptions_whitelist=(OSError),\n    # Exceptions that will not be taken into account\n    exceptions_blacklist=(),\n    # Directory where the error reports will be stored\n    output_directory="./out",\n    # Directory containing the initial input files\n    input_directory="./in",\n    # Enable or disable silent mode\n    silent=False\n)\nf.fuzz()\n```\n\n## Risks \n\nPlease keep in mind that, similarly to many other computationally-intensive\ntasks, fuzzing may put strain on your hardware and on the OS. In particular:\n\n  - Your CPU will run hot and will need adequate cooling. In most cases, if\n    cooling is insufficient or stops working properly, CPU speeds will be\n    automatically throttled. That said, especially when fuzzing on less\n    suitable hardware (laptops, smartphones, etc), it\'s not entirely impossible\n    for something to blow up.\n\n  - Targeted programs may end up erratically grabbing gigabytes of memory or\n    filling up disk space with junk files. Frelatage tries to enforce basic memory\n    limits, but can\'t prevent each and every possible mishap. The bottom line\n    is that you shouldn\'t be fuzzing on systems where the prospect of data loss\n    is not an acceptable risk.\n\n  - Fuzzing involves billions of reads and writes to the filesystem. On modern\n    systems, this will be usually heavily cached, resulting in fairly modest\n    "physical" I/O - but there are many factors that may alter this equation.\n    It is your responsibility to monitor for potential trouble; with very heavy\n    I/O, the lifespan of many HDDs and SSDs may be reduced.\n\n    A good way to monitor disk I/O on Linux is the \'iostat\' command:\n\n```shell\n    $ iostat -d 3 -x -k [...optional disk ID...]\n```\n\n## Contact \n\nfor any remark, suggestion, bug report, or if you found a bug using Frelatage, you can contact me at r0g3r5@protonmail.com or on twitter [@Rog3rSm1th](https://twitter.com/Rog3rSm1th)\n',
    'author': 'Rog3rSm1th',
    'author_email': 'r0g3r5@protonmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/Rog3rSm1th/Frelatage',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
