"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SnsPublish = exports.MessageAttributeDataType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const core_1 = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * The data type set for the SNS message attributes.
 *
 * @see https://docs.aws.amazon.com/sns/latest/dg/sns-message-attributes.html#SNSMessageAttributes.DataTypes
 * @stability stable
 */
var MessageAttributeDataType;
(function (MessageAttributeDataType) {
    MessageAttributeDataType["STRING"] = "String";
    MessageAttributeDataType["STRING_ARRAY"] = "String.Array";
    MessageAttributeDataType["NUMBER"] = "Number";
    MessageAttributeDataType["BINARY"] = "Binary";
})(MessageAttributeDataType = exports.MessageAttributeDataType || (exports.MessageAttributeDataType = {}));
/**
 * A Step Functions Task to publish messages to SNS topic.
 *
 * @stability stable
 */
class SnsPublish extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SnsPublish.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN) {
            if (!sfn.FieldUtils.containsTaskToken(props.message)) {
                throw new Error('Task Token is required in `message` Use JsonPath.taskToken to set the token.');
            }
        }
        this.taskPolicies = [
            new iam.PolicyStatement({
                actions: ['sns:Publish'],
                resources: [this.props.topic.topicArn],
            }),
        ];
    }
    /**
     * Provides the SNS Publish service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sns', 'publish', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                TopicArn: this.props.topic.topicArn,
                Message: this.props.message.value,
                MessageStructure: this.props.messagePerSubscriptionType ? 'json' : undefined,
                MessageAttributes: renderMessageAttributes(this.props.messageAttributes),
                Subject: this.props.subject,
            }),
        };
    }
}
exports.SnsPublish = SnsPublish;
_a = JSII_RTTI_SYMBOL_1;
SnsPublish[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.SnsPublish", version: "1.117.0" };
SnsPublish.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
function renderMessageAttributes(attributes) {
    if (attributes === undefined) {
        return undefined;
    }
    const renderedAttributes = {};
    Object.entries(attributes).map(([key, val]) => {
        renderedAttributes[key] = renderMessageAttributeValue(val);
    });
    return sfn.TaskInput.fromObject(renderedAttributes).value;
}
function renderMessageAttributeValue(attribute) {
    const dataType = attribute.dataType;
    if (attribute.value instanceof sfn.TaskInput) {
        return {
            DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING,
            StringValue: dataType !== MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
            BinaryValue: dataType === MessageAttributeDataType.BINARY ? attribute.value.value : undefined,
        };
    }
    if (dataType === MessageAttributeDataType.BINARY) {
        return { DataType: dataType, BinaryValue: `${attribute.value}` };
    }
    if (core_1.Token.isUnresolved(attribute.value)) {
        return { DataType: dataType !== null && dataType !== void 0 ? dataType : MessageAttributeDataType.STRING, StringValue: attribute.value };
    }
    validateMessageAttribute(attribute);
    if (Array.isArray(attribute.value)) {
        return { DataType: MessageAttributeDataType.STRING_ARRAY, StringValue: JSON.stringify(attribute.value) };
    }
    const value = attribute.value;
    if (typeof value === 'number') {
        return { DataType: MessageAttributeDataType.NUMBER, StringValue: `${value}` };
    }
    else {
        return { DataType: MessageAttributeDataType.STRING, StringValue: `${value}` };
    }
}
function validateMessageAttribute(attribute) {
    const dataType = attribute.dataType;
    const value = attribute.value;
    if (dataType === undefined) {
        return;
    }
    if (Array.isArray(value)) {
        if (dataType !== MessageAttributeDataType.STRING_ARRAY) {
            throw new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type Array`);
        }
        const validArrayTypes = ['string', 'boolean', 'number'];
        value.forEach((v) => {
            if (v !== null || !validArrayTypes.includes(typeof v)) {
                throw new Error(`Requested SNS message attribute type was ${typeof value} but Array values must be one of ${validArrayTypes}`);
            }
        });
        return;
    }
    const error = new Error(`Requested SNS message attribute type was ${dataType} but ${value} was of type ${typeof value}`);
    switch (typeof value) {
        case 'string':
            // trust the user or will default to string
            if (sfn.JsonPath.isEncodedJsonPath(attribute.value)) {
                return;
            }
            if (dataType === MessageAttributeDataType.STRING ||
                dataType === MessageAttributeDataType.BINARY) {
                return;
            }
            throw error;
        case 'number':
            if (dataType === MessageAttributeDataType.NUMBER) {
                return;
            }
            throw error;
        case 'boolean':
            if (dataType === MessageAttributeDataType.STRING) {
                return;
            }
            throw error;
        default:
            throw error;
    }
}
//# sourceMappingURL=data:application/json;base64,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