# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['scar_add_metadata_toolbox', 'scar_add_metadata_toolbox.hazmat']

package_data = \
{'': ['*'],
 'scar_add_metadata_toolbox': ['static/css/*',
                               'static/js/*',
                               'static/txt/*',
                               'static/xsl/iso-html/*',
                               'static/xsl/iso-rubric/*',
                               'templates/_includes/collection/*',
                               'templates/_includes/common/*',
                               'templates/_includes/items/*',
                               'templates/_layouts/*',
                               'templates/_views/*',
                               'templates/_views/errors/*',
                               'templates/_views/legal/*']}

install_requires = \
['Faker>=4.1.1,<5.0.0',
 'Flask>=1.1.1,<2.0.0',
 'Frozen-Flask>=0.15.0,<0.16.0',
 'Markdown>=3.2.2,<4.0.0',
 'OWSLib>=0.19.2,<0.20.0',
 'SQLAlchemy>=1.3.18,<2.0.0',
 'awscli>=1.18.93,<2.0.0',
 'backports-datetime-fromisoformat>=1.0.0,<2.0.0',
 'bas-metadata-library>=0.4.0,<0.5.0',
 'bas-style-kit-jinja-templates>=0.5.0,<0.6.0',
 'click-spinner>=0.1.10,<0.2.0',
 'flask-azure-oauth>=0.4.1,<0.5.0',
 'importlib_metadata>=1.7.0,<2.0.0',
 'importlib_resources>=3.0.0,<4.0.0',
 'inquirer>=2.7.0,<3.0.0',
 'lxml==4.5.2',
 'msal>=1.4.1,<2.0.0',
 'psycopg2-binary>=2.8.5,<3.0.0',
 'pycsw>=2.6.0,<3.0.0',
 'pyproj==2.6.0',
 'python-dateutil>=2.8.1,<3.0.0',
 'python-dotenv>=0.10.3,<0.11.0',
 'requests>=2.24.0,<3.0.0',
 'sentry-sdk[flask]>=0.14.3,<0.15.0',
 'simplejson>=3.17.0,<4.0.0',
 'str2bool>=1.1,<2.0',
 'tabulate>=0.8.7,<0.9.0']

setup_kwargs = {
    'name': 'scar-add-metadata-toolbox',
    'version': '0.3.0',
    'description': 'Editor, repository and data catalogue for SCAR Antarctic Digital Database (ADD) discovery metadata',
    'long_description': '# SCAR Antarctic Digital Database (ADD) Metadata Toolbox\n\nEditor, repository and data catalogue for\n[SCAR Antarctic Digital Database (ADD) discovery metadata](http://data.bas.ac.uk/collections/e74543c0-4c4e-4b41-aa33-5bb2f67df389/).\n\n## Status\n\nThis project is a mature alpha.\n\nThis means the core components needed have now been implemented but are subject to considerable change and refactoring.\n\nBetween releases major parts of this project may be replaced whilst the project evolves. As major non-core features are\nyet to be implemented the shape and scope of this project may change significantly. It is still expected that this\nproject will grow to cover other MAGIC datasets and products in future, and more widely to act as the seed for a new\nBAS wide data catalogue.\n\nThe *0.2.0* release has effectively been a complete rewrite of the project to reorganise and reimplement prototype code\nin a more structured way. Automated integration tests have been added and the project is now open-sourced.\n\nSome undesirable code from the *0.1.0* release still remains, to workaround issues in other packages until they can be\nproperly addressed. This code has been moved to a \'Hazardous Materials\' (`scar_add_metadata_toolbox.hazmat`) module.\n\nFurther information on upcoming changes to this project can be found in the issues and milestones in\n[GitLab (internal)](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/issues).\n\n**Note:** This project is designed to meet an internal need within the\n[Mapping and Geographic Information Centre (MAGIC)](https://www.bas.ac.uk/team/magic) at BAS. It has been open-sourced\nin case it\'s of use to others with similar needs.\n\n## Overview\n\nThis project is comprised of several components:\n\n1. metadata editor - for maintaining metadata records written as JSON files via the\n   [BAS Metadata Library](https://github.com/antarctica/metadata-library)\n2. unpublished (working) metadata repository - using an embedded, authenticated PyCSW catalogue\n3. published metadata repository - using an embedded, partially-authenticated PyCSW catalogue\n4. data catalogue - using a static website\n\nThese components map to components 2, 4 and 6 in the draft ADD data workflow\n([#139 (internal)](https://gitlab.data.bas.ac.uk/MAGIC/add/issues/139)).\n\nMetadata records are persisted within the the *metadata repositories*, using the ISO 19115-2 (geographic information)\nmetadata standard.\n\nAccess to unpublished records, and the ability to publish/retract records is restricted to restricted to relevant ADD\nproject members. Once published, records can viewed through the *data catalogue*, which presents them as human readable\nitems, (such as visualising geographic extents on a map). Manually curated collections provide a means to group items.\n\nThe *data catalogue* is rendered as a static website for improved performance, reliability and ease of hosting.\nCurrently this content is accessed within the existing [BAS data catalogue](https://data.bas.ac.uk).\n\n## Usage\n\n### Metadata editor\n\nThe *metadata editor* component of this project is ran on the BAS central workstations using the shared MAGIC user:\n\n```shell\n$ ssh geoweb@bslws01.nerc-bas.ac.uk\n$ scar-add-metadata-toolbox [command]\n```\n\nThe editor is configured using a settings file: `/users/geoweb/.config/scar-add-metadata-toolbox/.env`.\n\n### Metadata repositories and data catalogue\n\nThe *unpublished repository*, *published repository* and *data catalogue* run as a\n[service](http://bsl-nomad-magic-dev-s1.nerc-bas.ac.uk:4646/ui/jobs/scar-add-metadata-toolbox) in the experimental\n[MAGIC Nomad cluster](https://gitlab.data.bas.ac.uk/MAGIC/infrastructure/nomad).\n\nAny errors will be automatically reported to [Sentry](#sentry-error-tracking) and relevant individuals alerted by email.\n\n### Workflows\n\n* [adding new records](docs/workflow-adding-records.md)\n* [updating existing records](docs/workflow-updating-records.md)\n* [adding new collections](docs/workflow-adding-collections.md)\n* [updating existing collections](docs/workflow-updating-collections.md)\n\n### Available commands\n\n[Command line reference](docs/command-reference.md)\n\n## Implementation\n\nFlask application using [CSW](#csw) to store [Metadata records](#metadata-records) and display them as [Items](#items)\nin [Collections](#collections) rendered as [Jinja templates](#jinja-templates) served as a\n[static website](#s3-static-website) within the [BAS data catalogue](https://data.bas.ac.uk).\n\nCSW catalogues are backed by PostGIS databases, secured using [OAuth](#oauth). Contact forms for feedback and items in\nthe static catalogue use [Microsoft Power Automate](#feedback-and-contact-forms). Legal policies use templates from the\n[Legal Policies](https://gitlab.data.bas.ac.uk/web-apps/legal-policies-templates) project.\n\n### Architecture\n\nThis diagram shows the main concepts in this project and how they relate:\n\n![concepts overview](docs/assets/diagrams/concepts.png)\n\n### Metadata records\n\nMetadata records are the content and data within project. Records describe resources, which are typically datasets\nwithin the ADD, e.g. a record might describe the Antarctic Coastline dataset. Records are based on the ISO 19115\nmetadata standard (specifically 19115-2:2009), which defines an information model for geographic data.\n\nRecords are stored/persisted in a records repository (implemented using [CSW](#csw)) or in files for import and export.\n\nA metadata record includes information to answer questions such as:\n\n* what is this dataset?\n* what formats is this dataset available in?\n* what projection does this dataset use?\n* what keywords describe the themes, places, etc. related to this dataset?\n* why is this dataset useful?\n* who is this dataset for?\n* who made this dataset?\n* who can I contact with any questions about this dataset?\n* when was this dataset created?\n* when was it last updated?\n* what time period does it cover?\n* where does this dataset cover?\n* how was this dataset created?\n* how can trust the quality of this dataset?\n* how can I download or access this dataset?\n\nThis metadata is termed \'discovery metadata\' (to separate it from metadata for calibration or analysis for example). It\nhelps users find metadata in catalogues or search engines, and then to help them decide if the data is useful to them.\n\nThe information in a metadata record is encoded in a different formats at different stages:\n\n* during editing, records are encoded as JSON, using the\n  [BAS Metadata Library](https://github.com/antarctica/metadata-library)\'s record configuration\n* when stored in a repository, records are encoded as XML using the ISO 19139 standard\n* when viewed in the data catalogue, records are encoded in HTML or as (styled) XML\n\nThese different formats are used for different reasons:\n\n* JSON is easily understood by machines and is concise to understand and edit by humans\n* XML is also machine readable but more suited/mature for complex standards such ISO 19139\n* HTML is designed for presenting information to humans, with very flexible formatting options\n\n### Items\n\nItems represent [Metadata records](#metadata-records) but in a form intended for human consumption. They are derived\nfrom [Records](#metadata-records) and are specific to the data catalogue, allowing greater flexibility compared to the\nstrict rigidity and formality enforced by metadata records.\n\nFor example, a resource\'s coordinate reference system may be defined as `urn:ogc:def:crs:EPSG::3031` in a metadata\nrecord but will be shown as `WGS 84 / Antarctic Polar Stereographic (EPSG:3031)` in items. Both are technically correct\nbut the descriptive version is easier for a human to understand, at the sake of being less precise and harder for\nmachines to parse.\n\nAs items are derived from records, they are not persisted themselves, except as rendered pages within the data catalogue\nstatic site.\n\n### Collections\n\nCollections are a simple way to group [Items](#items) together based on a shared purpose, theme or topic. They are\nspecific to the data catalogue and are not based on metadata records.\n\nCollections are stored/persisted in a collections repository (implemented as a JSON file) or in files for import and\nexport.\n\nThey support a limited set of properties compared to records/items:\n\n| Property           | Data Type | Required | Description                                                  |\n| ------------------ | --------- | -------- | ------------------------------------------------------------ |\n| identifier         | String    | Yes      | UUID                                                         |\n| title              | String    | Yes      | Descriptive title                                            |\n| topics             | Array     | Yes      | BAS Research Topics associated with collection               |\n| topics.*           | String    | Yes      | BAS Research Topic                                           |\n| publishers         | Array     | Yes      | Data catalogue publishers associated with collection         |\n| publishers.*       | String    | Yes      | Data catalogue publisher                                     |\n| summary            | String    | Yes      | Descriptive summary, supports markdown formatting            |\n| item_identifiers   | Array     | Yes      | Items associated with collection, specified by their Item ID |\n| item_identifiers.* | String    | Yes      | Item ID                                                      |\n\n**Note:** Items in collections will be shown in the order they are listed in the `item_identifiers` property.\n\nFor example:\n\n```json\n{\n    "identifier": "1790c9d5-af77-4a03-9a08-6ba8e83ce748",\n    "title": "Operation Tabarin",\n    "topics": [\n        "Living and Working in Antarctica"\n    ],\n    "publishers": [\n        "BAS Archives Service"\n    ],\n    "summary": "A secret British Antarctic expedition launched in 1943 during the course of World War II ...",\n    "item_identifiers": [\n        "82f3fe32-6d6b-4e7a-8256-690ce99fc653",\n        "88a22198-36e0-4aff-9099-aae1dfd7baa9",\n        "35c6d732-3acc-4044-9c8f-680eed39268a"\n    ]\n}\n```\n\n### OAuth\n\nThis project uses OAuth to protect access to the *Unpublished* and *Published* repositories via the\n[Microsoft (Azure) identity platform](https://docs.microsoft.com/en-us/azure/active-directory/develop/).\n\nThe *Unpublished* and *Published* repositories are registered together as the resource to be protected with different\nscopes and roles to authorise users to read, write and publish records. The\n[Flask Azure AD OAuth Provider](https://pypi.org/project/flask-azure-oauth/) is used to verify access tokens when CSW\nrequests are made and enforce these permissions as needed.\n\nThe *Metadata editor* is registered as a separate application as a client that will interact with protected resource\n(i.e. to read, write and publish records). The\n[Microsoft Authentication Library (MSAL) for Python](https://github.com/AzureAD/microsoft-authentication-library-for-python)\nlibrary is used to request access tokens using the OAuth device code grant type.\n\nBoth applications are registered in the NERC Azure tenancy administered by the\n[UKRI/NERC DDaT](https://infohub.ukri.org/corporate-hub/digital-data-and-technology-ddat/) team, currently via the old\n[RTS Helpdesk](mailto:rtsservicedesk@nerc.ac.uk).\n\nThe [Azure Portal](https://portal.azure.com) is used to assign permissions to applications and users as needed:\n\n* [assigning permissions to users](docs/workflow-permissions-users.md)\n\n### CSW\n\nThe *Unpublished* and *Published* repositories are implemented as embedded [PyCSW](http://pycsw.org) servers. The\nembedded mode allowing integration with Flask for authentication and authorisation of requests via [OAuth](#oauth).\n\nThe CSW transactional profile is used extensively for clients (such as the *Metadata editor* and *Data catalogue*) to\ninsert, update and delete records programmatically.\n\nThe CSW version is fixed to *2.0.2* because it\'s the latest version supported by\n[OWSLib](https://geopython.github.io/OWSLib/), the CSW client used by the *Metadata editor*.\n\n**Note:** The CSW repositories are considered to be APIs, and so ran as services through the\n[BAS API Load Balancer](https://gitlab.data.bas.ac.uk/WSF/api-load-balancer) (internal) with documentation in the\n[BAS API Documentation](https://gitlab.data.bas.ac.uk/WSF/api-docs) project (internal).\n\n**Note:** Some elements of both the PyCSW server and the OWSLib client have been extended by this project to incorporate\nOAuth support. These modifications will be formalised, ideally as upstream contributions, but currently reside within\nthe \'Hazardous Materials\' module as a number of unsightly workarounds are currently needed.\n\n#### Max records limit\n\nBoth PyCSW (CSW servers) and OWSLib (CSW clients) have a maximum record of 100 per request.\n\n#### CSW backing databases\n\nCSW servers are backed using PostGIS (PostgreSQL) databases provided by BAS IT (via the central Postgres database\n`bsldb`). As PyCSW uses a single table for all records, all servers share the same database and schema, configured\nthrough SQLAlchemy connection strings.\n\nSeparate databases are used for each environment (development, staging and production). Credentials are stored in the\nMAGIC 1Password shared vault. In local development, a local PostGIS database configured in `docker-compose.yml` can be\nused:\n\n```\npostgresql://postgres:password@csw-db/postgres`\n```\n\nTo test against real data in a non-production environment, use the staging environment database, which is synced from\nthe production database automatically by BAS IT every Tuesday at 02:00.\n\n### Jinja templates\n\nA series of [Jinja2](https://jinja.palletsprojects.com/) templates are used for rendering pages in the *Data catalogue*.\n\nTemplates use the [BAS Style Kit Jinja Templates](https://pypi.org/project/bas-style-kit-jinja-templates/) and styled\nusing the [BAS Style Kit](https://style-kit.web.bas.ac.uk).\n\n### S3 static website\n\nRendered pages and other assets are hosted through an AWS S3 bucket with static website hosting enabled.\n\nReverse proxy rules are used to expose content from this static site within the existing/legacy BAS Data Catalogue, DMS.\n\n### Feedback and contact forms\n\nA Microsoft\n[Power Automate](https://emea.flow.microsoft.com/manage/environments/Default-b311db95-32ad-438f-a101-7ba061712a4e/flows/97d95c3b-5d40-4358-86a6-979a679a4b7c/details)\nFlow is used to process feedback and contact form submissions. Messages support Markdown formatting, converted to HTML\nprior to submission. On submitted, Power Automate creates an issue for the message in a relevant GitLab project.\n\n### Sentry error tracking\n\nErrors in this service are tracked with Sentry:\n\n* [Sentry dashboard](https://sentry.io/organizations/antarctica/issues/?project=5197036)\n* [GitLab dashboard](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/error_tracking)\n\nError tracking will be enabled or disabled depending on the environment. It can be manually controlled by setting the\n`APP_ENABLE_SENTRY` [Configuration option](#configuration).\n\n### Application logging\n\nLogs for this service are written to *stdout/stderr* as appropriate.\n\n## Configuration\n\nApplication configuration options are set in per-environment classes extending a base `Config` class in\n`scar_add_metadata_toolbox/config.py`. The active environment is set using the `FLASK_ENV` environment variable.\n\nConfiguration options are defined, and documented, using class properties. Some configuration options may optionally be\nset at runtime using environment variables. If not set, default values will be used.\n\n| Configuration Option                                | Description                                                      | Allowed Values                     | Example Value                                                   |\n| --------------------------------------------------- | ---------------------------------------------------------------- | ---------------------------------- | --------------------------------------------------------------- |\n| `APP_ENABLE_SENTRY`                                 | Feature flag to enable/disable Sentry error tracking             | true/false                         | `true`                                                          |\n| `APP_LOGGING_LEVEL`                                 | Minimum logging level to include in application logs             | debug/info/warning/error/critical  | `warning`                                                       |\n| `APP_AUTH_SESSION_FILE_PATH`                        | Path to file used for authentication information                 | valid file path                    | `/home/user/.config/scar_add_metadata_toolbox/auth.json`        |\n| `APP_COLLECTIONS_PATH`                              | Path to file used for data catalogue collections                 | valid file path                    | `/home/user/.config/scar_add_metadata_toolbox/collections.json` |\n| `APP_SITE_PATH`                                     | Path to directory used for rendered static site content          | valid directory path               | `/home/user/.config/scar_add_metadata_toolbox/_site`            |\n| `CSW_ENDPOINT_UNPUBLISHED`                          | CSW endpoint for accessing unpublished catalogue                 | valid URL                          | `http://example.com/csw/unpublished`                            |\n| `CSW_ENDPOINT_PUBLISHED`                            | CSW endpoint for accessing published catalogue                   | valid URL                          | `http://example.com/csw/published`                              |\n| `CSW_SERVER_CONFIG_UNPUBLISHED_ENDPOINT`            | Endpoint at which to run unpublished CSW catalogue               | valid URL                          | `http://example.com/csw/unpublished`                            |\n| `CSW_SERVER_CONFIG_PUBLISHED_ENDPOINT`              | Endpoint at which to run published CSW catalogue                 | Valid URL                          | `http://example.com/csw/published`                              |\n| `CSW_SERVER_CONFIG_UNPUBLISHED_DATABASE_CONNECTION` | Connection string for unpublished CSW catalogue backing database | Valid SQLAlchemy connection string | `postgresql://postgres:password@db.example.com/postgres`        |\n| `CSW_SERVER_CONFIG_PUBLISHED_DATABASE_CONNECTION`   | Connection string for published CSW catalogue backing database   | Valid SQLAlchemy connection string | `postgresql://postgres:password@db.example.com/postgres`        |\n| `APP_S3_BUCKET`                                     | AWS S3 bucket name used for hosting static website content       | Valid AWS S3 bucket name           | `add-catalogue.data.bas.ac.uk`                                  |\n\nThese options are typically set when running this application as a client (metadata editor and data catalogue):\n\n* `APP_LOGGING_LEVEL`\n* `APP_AUTH_SESSION_FILE_PATH`\n* `APP_COLLECTIONS_PATH`\n* `APP_SITE_PATH`\n* `CSW_ENDPOINT_UNPUBLISHED`\n* `CSW_ENDPOINT_PUBLISHED`\n* `APP_S3_BUCKET`\n\nThese options are typically set when running this application as a server (metadata repositories):\n\n* `APP_LOGGING_LEVEL`\n* `CSW_SERVER_CONFIG_UNPUBLISHED_ENDPOINT`\n* `CSW_SERVER_CONFIG_PUBLISHED_ENDPOINT`\n* `CSW_SERVER_CONFIG_UNPUBLISHED_DATABASE_CONNECTION`\n* `CSW_SERVER_CONFIG_PUBLISHED_DATABASE_CONNECTION`\n\n## Setup\n\n[Continuous deployment](#continuous-deployment) will configure this application to run on the BAS central workstations\nas a Podman container, using an automatically generated launch script and environment variables.\n\n[Continuous deployment](#continuous-deployment) will configure this application to run in the experimental\n[MAGIC Nomad cluster](https://gitlab.data.bas.ac.uk/MAGIC/infrastructure/nomad), using an automatically\ngenerated job definition.\n\nSee the [Usage](#usage) section for how to use the application.\n\n### PyCSW backing database setup\n\nBacking databases for PyCSW servers require initialisation using the `csw setup` application\n[CLI command](docs/command-reference.md#csw-setup) for both the *published* and *unpublished* repositories.\n\n**Note:** Backing databases must use the PostgreSQL engine with the PostGIS extension enabled.\n\nNormally this command will create the required database table, geometry column and relevant indexes. As catalogues only\nrequire a single table, multiple can be stored in the same database/schema. However, two of the indexes used\n(`fts_gin_idx` [full text search] and `wkb_geometry_idx` [binary geometry]) are named non-uniquely, preventing multiple\ncatalogues being co-located in the same schema.\n\nThis appears to be an oversight, as all other indexes are made unique by prefixing them with the name of the records\ntable, and doing this manually for these indexes appears to work without issue. To workaround this issue, you will need\nto manually modify the indexes of catalogue tables once they\'ve been setup.\n\nAssuming the *Unpublished catalogue* is setup first, perform these steps *before* setting up the *Published catalogue*:\n\n1. verify that the `records_unpublished` table was created successfully (contains `fts_gin_idx` and `wkb_geometry_idx`\n   indexes)\n2. alter the affected indexes in the `records_unpublished` table [1]\n3. setup the *Published catalogue* `flask csw setup published`\n4. alter the affected indexes in the second table [2]\n\n[1]\n\n```sql\nALTER INDEX fts_gin_idx RENAME TO ix_records_unpublished_fts_gin_indx;\nALTER INDEX wkb_geometry_idx RENAME TO ix_unpublished_wkb_geometry_idx;\n```\n\n[2]\n\n```sql\nALTER INDEX fts_gin_idx RENAME TO ix_records_published_fts_gin_indx;\nALTER INDEX wkb_geometry_idx RENAME TO ix_published_wkb_geometry_idx;\n```\n\n### Azure permissions\n\n[Terraform](#terraform) will create and configure the relevant Azure application registrations required for using\n[OAuth](#oauth) to protect the CSW catalogues. However manual approval by a Tenancy Administrator is needed to grant\nthe registration representing the *client* role of the application access to the registration for the *server* role.\n\nThis has been approved by NERC RTS in [#3](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/issues/3).\n\n### Terraform\n\nTerraform is used for:\n\n* resources required for hosting the *Data catalogue* component as a static website\n* resources required for protecting and accessing the *unpublished repository*, *published repository*  components\n* a templated job file for Nomad during [Continuous deployment](#continuous-deployment)\n* a templated launch script for Podman during [Continuous deployment](#continuous-deployment)\n\nAccess to the [BAS AWS account](https://gitlab.data.bas.ac.uk/WSF/bas-aws),\n[Terraform remote state](#terraform-remote-state) and NERC Azure tenancy are required to provision these resources.\n\n**Note:** The templated Podman and Nomad runtime files are not included in Terraform state.\n\n```shell\n$ cd provisioning/terraform\n$ docker compose run terraform\n\n$ az login --allow-no-subscriptions\n\n$ terraform init\n$ terraform validate\n$ terraform fmt\n$ terraform apply\n\n$ exit\n$ docker compose down\n```\n\nOnce provisioned the following steps need to be taken manually:\n\n1. set branding icons (if desired)\n2. set [Azure permissions](#azure-permissions)\n3. [assign roles](docs/workflow-permissions-users.md) to users and/or groups\n4. set `accessTokenAcceptedVersion: 2` in both application registration manifests\n\n**Note:** Assignments are 1:1 between users/groups and roles but there can be multiple assignments. I.e. roles `Foo`\nand `Bar` can be assigned to the same user/group by creating two role assignments.\n\n#### Terraform remote state\n\nState information for this project is stored remotely using a\n[Backend](https://www.terraform.io/docs/backends/index.html).\n\nSpecifically the [AWS S3](https://www.terraform.io/docs/backends/types/s3.html) backend as part of the\n[BAS Terraform Remote State](https://gitlab.data.bas.ac.uk/WSF/terraform-remote-state) project.\n\nRemote state storage will be automatically initialised when running `terraform init`. Any changes to remote state will\nbe automatically saved to the remote backend, there is no need to push or pull changes.\n\n##### Remote state authentication\n\nPermission to read and/or write remote state information for this project is restricted to authorised users. Contact\nthe [BAS Web & Applications Team](mailto:servicedesk@bas.ac.uk) to request access.\n\nSee the [BAS Terraform Remote State](https://gitlab.data.bas.ac.uk/WSF/terraform-remote-state) project for how these\npermissions to remote state are enforced.\n\n### Docker image tag expiration policy\n\nThe Docker image for this project uses a [Tag expiration policy](#docker-image-expiration-policy) which needs to be\nconfigured manually in [GitLab](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/settings/ci_cd).\n\n* Expiration policy: *enabled*\n* Expiration interval: *90 days*\n* Expiration schedule: *Every week*\n* Number of tags to retain: *10 tag per image name*\n* Tags with names matching this regex pattern will expire: `(review.+|build.+)`\n* Tags with names matching this regex pattern will be preserved: `release.+`\n\n### BAS IT\n\nManually request a new PostGIS database for the CSW catalogue backing databases from the BAS IT ServiceDesk.\n\nManually request a new application to be deployed from the BAS IT ServiceDesk using the\n[request template](http://ictdocs.nerc-bas.ac.uk/wiki/index.php/Provisioning_Process#Template_ServiceDesk_request).\n\nSee [#44](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/issues/44) for an example.\n\n### BAS API Load Balancer\n\nManually [add a new service](https://gitlab.data.bas.ac.uk/WSF/api-load-balancer#adding-a-new-service) and related\n[documentation](https://gitlab.data.bas.ac.uk/WSF/api-docs#adding-a-new-service-service-version).\n\nSee [#60](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/issues/60) for an example.\n\n## Development\n\n```shell\n$ git clone https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox\n$ cd add-metadata-toolbox\n```\n\n### Development environment\n\nA flexible development environment is available for developing this application locally. It can be used in a variety of\nways depending on what is being developed:\n\n* all components, the Flask application, CSW database and static website can be ran locally\n    * useful for end-to-end testing\n    * useful for testing changes to how data is loaded into the CSW catalogues\n* the Flask application can be ran directly, without needing to convert it into a static site\n    * useful for iterating on changes to the data catalogue website\n* the Flask application can use the production CSW database\n    * useful for testing with real-world data\n\nThe local development environment is defined using Docker Compose in `./docker-compose.yml`. It consists of:\n\n* an `app` service for running the Flask application as a web application\n* an `app-cli` service for running [application commands](docs/command-reference.md)\n* a `csw-db` service for storing data added to local CSW catalogues (if used)\n* a `web` service for serving a local version of the data catalogue static site (if used)\n\nTo create a local development environment:\n\n1. pull docker images: `docker compose pull` [1]\n3. run the Docker Compose stack: `docker compose up`\n    * the Flask application will be available directly at: [http://localhost:9000](http://localhost:9000)\n    * the static site will be available at: [http://localhost:9001](http://localhost:9001)\n4. run application [Commands](docs/command-reference.md) [2]\n\nTo destroy a local development environment:\n\n1. run `docker compose down`\n\n[1] This requires access to the BAS Docker Registry (part of [gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)):\n\n```shell\n$ docker login docker-registry.data.bas.ac.uk\n```\n\n**Note:** You will need to sign-in using your GitLab credentials (your password is set through your GitLab profile) the\nfirst time this is used.\n\n[2] In a new terminal:\n\n```shell\n$ docker compose run app-cli flask [task]\n```\n\n#### Development container\n\nA development container image, defined by `./Dockerfile`, is built manually, tagged as `:latest` and hosted in the\nprivate BAS Docker Registry (part of [gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)):\n\n[docker-registry.data.bas.ac.uk/magic/add-metadata-toolbox](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/container_registry)\n\nIt is separate to the [deployment container](#docker-image) and installs both runtime and development\n[dependencies](#dependencies) (deployment containers only install runtime dependencies).\n\nIf you don\'t have access to the BAS Docker Register, you can build this image locally using `docker compose build app`.\n\n### Python version\n\nWhen upgrading to a new version of Python, ensure the following are also checked and updated where needed:\n\n* `Dockerfile`:\n    * base stage image (e.g. `FROM python:3.X-alpine as base` to `FROM python:3.Y-alpine as base`)\n    * pre-compiled wheels (e.g. `https://.../linux_x86_64/cp3Xm/lxml-4.5.0-cp3X-cp3X-linux_x86_64.whl` to\n     `http://.../linux_x86_64/cp3Ym/lxml-4.5.0-cp3Y-cp3Y-linux_x86_64.whl`)\n* `support/docker-packaging/Dockerfile`:\n    * base stage image (e.g. `FROM python:3.X-alpine as base` to `FROM python:3.Y-alpine as base`)\n    * pre-compiled wheels (e.g. `http://.../linux_x86_64/cp3Xm/lxml-4.5.0-cp3X-cp3X-linux_x86_64.whl` to\n     `http://.../linux_x86_64/cp3Ym/lxml-4.5.0-cp3Y-cp3Y-linux_x86_64.whl`)\n* `pyproject.toml`\n    * `[tool.poetry.dependencies]`\n        * `python` (e.g. `python = "^3.X"` to `python = "^3.Y"`)\n    * `[tool.black]`\n        * `target-version` (e.g. `target-version = [\'py3X\']` to `target-version = [\'py3Y\']`)\n\n### Package structure\n\nAll code for this project should be defined in the [`scar_add_metadata_toolbox`](scar_add_metadata_toolbox) package,\nwith the exception of tests.\n\nIn brief, this package is comprised of these modules:\n\n* `scar_add_metadata_toolbox` - contains [Flask application](#flask-application)\n* `scar_add_metadata_toolbox.classes` - contains classes for concepts (Repositories, Records, Items, Collections)\n* `scar_add_metadata_toolbox.commands` - contains Flask blueprints used for CLI commands\n* `scar_add_metadata_toolbox.config` - contains [Flask configuration](#flask-configuration)\n* `scar_add_metadata_toolbox.csw` - contains classes for [CSW](#csw) servers and clients\n* `scar_add_metadata_toolbox.hazmat` - contains [\'Hazardous Material\' code](#hazardous-materials-module)\n* `scar_add_metadata_toolbox.static` - contains static site assets (CSS, JS, etc.)\n* `scar_add_metadata_toolbox.templates` - contains [Application templates](#templates)\n* `scar_add_metadata_toolbox.utils` - contains various utility/helper methods and classes\n\n#### Hazardous Materials module\n\nWhilst this application has been developed, extended/modified versions of class from 3rd party packages have been\ncreated to address unresolved bugs or add new required functionality. As this code often requires workarounds and hacks\nit is ugly, non-standard and against established best practices (such as not to use mocks outside of tests).\n\nIn time, these changes and additions are expected to be either incorporated into upstream packages, or if not possible,\ninto forked packages that we maintain. Until then, this code is kept in a \'Hazardous Materials\' (Hazmat) module,\n`scar_add_metadata_toolbox.hazmat`, to indicate that it shouldn\'t be treated like other modules in this project.\n\nIdeally no additional code will be added to this module, however if other changes/extensions need to be made in a\nnon-clean way then they should be added here, rather than \'polluting\' the main package.\n\n### Dependencies\n\nPython dependencies for this project are managed with [Poetry](https://python-poetry.org) in `pyproject.toml`.\n\nNon-code files, such as static files, can also be included in the [Python package](#python-package) using the\n`include` key in `pyproject.toml`.\n\n#### Adding new dependencies\n\nTo add a new (development) dependency:\n\n```shell\n$ docker compose run app ash\n$ poetry add [dependency] (--dev)\n```\n\nThen rebuild the [Development container](#development-container) and push to GitLab (GitLab will rebuild other images\nautomatically as needed):\n\n```shell\n$ docker compose build app\n$ docker compose push app\n```\n\n#### Updating dependencies\n\n```shell\n$ docker compose run app ash\n$ poetry update\n```\n\nThen rebuild the [Development container](#development-container) and push to GitLab (GitLab will rebuild other images\nautomatically as needed):\n\n```shell\n$ docker compose build app\n$ docker compose push app\n```\n\n### Static security scanning\n\nTo ensure the security of this API, source code is checked against [Bandit](https://github.com/PyCQA/bandit) for issues\nsuch as not sanitising user inputs or using weak cryptography. Bandit is configured in `.bandit`.\n\n**Warning:** Bandit is a static analysis tool and can\'t check for issues that are only be detectable when running the\napplication. As with all security tools, Bandit is an aid for spotting common mistakes, not a guarantee of secure code.\n\nTo run checks manually:\n\n```shell\n$ docker compose run app bandit -r .\n```\n\nChecks are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Code Style\n\nPEP-8 style and formatting guidelines must be used for this project, with the exception of the 80 character line limit.\n\n[Black](https://github.com/psf/black) is used to ensure compliance, configured in `pyproject.toml`.\n\nBlack can be [integrated](https://black.readthedocs.io/en/stable/editor_integration.html) with a range of editors, such\nas PyCharm, to perform formatting automatically.\n\nTo apply formatting manually:\n\n```shell\n$ docker compose run app black scar_add_metadata_toolbox/\n```\n\nTo check compliance manually:\n\n```shell\n$ docker compose run app black --check scar_add_metadata_toolbox/\n```\n\nChecks are ran automatically in [Continuous Integration](#continuous-integration).\n\n### Flask application\n\nThe Flask application representing this project is defined in the\n[`scar_add_metadata_toolbox`](/scar_add_metadata_toolbox) package. The application uses the\n[application factory](https://flask.palletsprojects.com/en/1.1.x/patterns/appfactories/) pattern.\n\nFlask Blueprints are used to logically organise application commands, currently all within the\n`scar_add_metadata_toolbox.commands` module. Until this is refactored, additional commands should be registered in the\nsame module.\n\n### Flask configuration\n\nThe Flask application\'s configuration (`app.config`) is populated from an environment specific class in the\n`scar_add_metadata_toolbox.config` module.\n\nNew configuration options should be added to the base config class as properties, overridden as needed in environment\nsub-classes. Where a configuration should be configurable at runtime it should be read as an environment variable and\ndocumented in the [Configuration](#configuration) section.\n\n### Logging\n\nUse the Flask application\'s logger, for example:\n\n```python\nfrom flask import current_app\n\ncurrent_app.logger.info(\'Log message\')\n```\n\n### File paths\n\nUse Python\'s [`pathlib`](https://docs.python.org/3.8/library/pathlib.html) library for file paths.\n\nWhere displaying a file path to the user, use the absolute form to aid in debugging:\n\n```python\nfrom pathlib import Path\n\nfoo_path = Path("foo.txt")\nprint(f"foo_path is: {str(foo_path.absolute())}")\n```\n\n### Templates\n\nApplication templates use the Flask application\'s Jinja environment configured to use general templates from the\n[BAS Style Kit Jinja Templates](https://pypi.org/project/bas-style-kit-jinja-templates) package (for layouts, etc.) and\napplication specific templates from the `scar_add_metadata_toolbox.templates` module.\n\nStyles, components and patterns from the [BAS Style Kit](https://style-kit.web.bas.ac.uk) should be used where possible.\nConfiguration options for Style Kit Jinja Templates are set in the `scar_add_metadata_toolbox.config` module, including\nloading local styles and scripts defined in [`scar_add_metadata_toolbox/static`](scar_add_metadata_toolbox/static).\n\nApplication views should inherit from the application layout,\n[`app.j2`](scar_add_metadata_toolbox/templates/_layouts/app.j2), and using\n[includes](https://jinja.palletsprojects.com/en/2.11.x/templates/#include) and\n[macros](https://jinja.palletsprojects.com/en/2.11.x/templates/#macros) to breakdown and reuse content within views is\nstrongly encouraged.\n\n### Editor support\n\n#### PyCharm\n\nMultiple run/debug configurations are included in the project for debugging and testing:\n\n* *App* runs the Flask application and is useful for debug the server role of this application (e.g. CSW requests)\n* *App CLI* runs the Flask application and is useful for debug the client role of this application (e.g. static site\n   commands requests) - change the *parameters* option to set which command to run\n\n### Testing\n\nAll code in the `scar_add_metadata_toolbox` package must be covered by tests, defined in `tests/`. This project uses\n[PyTest](https://docs.pytest.org/en/latest/) which should be ran in a random order using\n[pytest-random-order](https://pypi.org/project/pytest-random-order/).\n\nTo run tests manually from the command line:\n\n```shell\n$ docker compose run app -e FLASK_ENV=testing app pytest --random-order\n```\n\nTo run/debug tests using PyCharm, use the included *App (Tests)* run/debug configuration.\n\nTests are ran automatically in [Continuous Integration](#continuous-integration).\n\n#### Test coverage\n\n[pytest-cov](https://pypi.org/project/pytest-cov/) is used to measure test coverage.\n\nA `.coveragerc` file is used to omit code from the `scar_add_metadata_toolbox.hazmat` module.\n\nTo measure coverage manually:\n\n```shell\n$ docker compose run -e FLASK_ENV=testing app pytest --cov=scar_add_metadata_toolbox --cov-fail-under=100 --cov-report=html .\n```\n\n[Continuous Integration](#continuous-integration) will check coverage automatically and fail if less than 100%.\n\n#### Continuous Integration\n\nAll commits will trigger a Continuous Integration process using GitLab\'s CI/CD platform, configured in `.gitlab-ci.yml`.\n\n## Review apps\n\nTo review changes to functionality, commits made in branches will trigger review apps to be created using GitLab\'s\nCI/CD platform, configured in `.gitlab-ci.yml`.\n\nReview apps run as [Nomad services](#nomad-service) only, not as [Command line applications](#command-line-application).\n\nContainers for review apps are built using the [deployment Docker image](#docker-image) but tagged as `review:[slug]`,\nwhere `[slug]` is a reference to the merge request the review app is related to. Images are hosted in the private BAS\nDocker Registry (part of [gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)):\n\n[docker-registry.data.bas.ac.uk/magic/add-metadata-toolbox/deploy](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/container_registry)\n\n### Limitations\n\n* the URL for review apps point to the Nomad job via it\'s UI, rather than the managed application, as the port number\n  for the application is set dynamically and not stored in the Terraform state for the Nomad job\n* the application will currently use the production CSW database, therefore records **MUST NOT** be changed by review\n  apps, this is currently unenforced but will be when ServiceDesk ticket #42232 is resolved\n\n## Deployment\n\n### Python package\n\nA project Python package is built by [Continuous Delivery](#continuous-deployment), hosted through the private BAS Repo\nServer:\n\n[bsl-repoa.nerc-bas.ac.uk/magic/v1/projects/scar-add-metadata-toolbox/latest/dist/](http://bsl-repoa.nerc-bas.ac.uk/magic/v1/projects/scar-add-metadata-toolbox/latest/dist/)\n\n### Docker image\n\nA deployment container image, defined by `./support/docker-packaging/Dockerfile`, is built by\n[Continuous Delivery](#continuous-deployment) for releases (Git tags). Images are tagged as `/release:[tag]`, where\n`[tag]` is the name of the Git tag a release is related to. Images are hosted in the private BAS Docker Registry (part\nof [gitlab.data.bas.ac.uk](https://gitlab.data.bas.ac.uk)):\n\n[docker-registry.data.bas.ac.uk/magic/add-metadata-toolbox/deploy](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/container_registry)\n\n**Note:** All container builds (including those from [Review apps](#review-apps)) are also tagged as `/build:[commit]`,\nwhere `[commit]` is a reference to the Git commit that triggered the image to be built.\n\n#### Docker image expiration policy\n\nAn image [expiration policy](https://docs.gitlab.com/ee/user/packages/container_registry/#cleanup-policy) is used to\nlimit the number of non-release container images that are kept. This policy is set within, and enforced automatically\nby, GitLab. See the [Setup section](#docker-image-tag-expiration-policy) for how this is configured.\n\n### Nomad service\n\nThe deployment [Docker image](#docker-image) is deployed as a service job in the experimental\n[MAGIC Nomad cluster](https://gitlab.data.bas.ac.uk/MAGIC/infrastructure/nomad) (internal).\n\n### BAS IT service\n\nThe deployment [Python package](#python-package) is deployed as a WSGI application via BAS IT using an Ansible playbook:\n[`/playbooks/magic/add-metadata-toolbox.yml`](https://gitlab.data.bas.ac.uk/station-data-management/ansible/-/blob/master/playbooks/magic/add-metadata-toolbox.yml) (internal)\n\nVariables for this application are set in:\n[`/group_vars/magic/add-metadata-toolbox.yml`](https://gitlab.data.bas.ac.uk/station-data-management/ansible/-/blob/master/group_vars/magic/add-metadata-toolbox.yml) (internal)\n\nEnvironment variables used by this application are set in:\n[`/playbooks/magic/add-metadata-toolbox.yml`](https://gitlab.data.bas.ac.uk/station-data-management/ansible/-/blob/master/playbooks/magic/add-metadata-toolbox.yml) (internal)\n\nThis application is deployed to a development, staging and production environment. Hosts for each environment are listed\nin the relevant Ansible inventory in:\n[`/inventory/magic/`](https://gitlab.data.bas.ac.uk/station-data-management/ansible/-/tree/master/inventory/magic) (internal)\n\n**Note:** The process to run/update this playbook/variables is still under development (see\n[#44](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/issues/44) (internal) for background). Currently\neither needs to be requested through the [IT ServiceDesk](mailto:servicedesk@bas.ac.uk).\n\n#### Key paths\n\nKey files/directories within this deployed application are:\n\n* `/etc/httpd/sites/10-add-metadata-toolbox.conf`: Apache virtual host\n* `/var/opt/wsgi/.virtualenvs/add-metadata-toolbox`: Python virtual environment\n* `/var/www/add-metadata-toolbox/app.py`: Application entrypoint script\n* `/var/log/httpd/access_log.add_metadata_toolbox`: Apache virtual host access log\n* `/var/log/httpd/error_log.add_metadata_toolbox`: Apache/Application error/log file\n\n#### SSH access\n\n| Environment | SSH Access     | Sudo | Access   |\n| ----------- | -------------- | ---- | -------- |\n| Development | Yes            | Yes  | `felnne` |\n| Staging     | Yes (for logs) | No   | `felnne` |\n| Production  | Yes (for logs) | No   | `felnne` |\n\nCurrently access to the servers for each environment is bespoke but should be standardised in future, see\n[#100](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/-/issues/100) for more information.\n\n#### Flask CLI\n\nTo use the Flask CLI:\n\n```shell\n$ ssh [server]\n$ sudo su\n$ . [path to virtual environment]/bin/activate\n$ export FLASK_APP=scar_add_metadata_toolbox\n$ export FLASK_ENV=production\n$ flask [command]\n$ deactivate\n$ exit\n$ exit\n```\n\n### API Service\n\nThe CSW Catalogues are deployed as a service within the BAS API Load Balancer, backed by the production\n[BAS IT service](#bas-it-service).\n\n#### API Documentation\n\nUsage documentation for this API service is held in `docs/api/` and currently\n[manually](https://gitlab.data.bas.ac.uk/WSF/api-docs#adding-a-service-manually) published using these service paths:\n\n* `s3://bas-api-docs-content-testing/services/data/metadata/add/csw/`\n* `s3://bas-api-docs-content/services/data/metadata/add/csw/`\n\n### Command line application\n\nThe deployment [Docker image](#docker-image) is made available as a command line application on the BAS central\nworkstations using Podman. A wrapper shell script is used to mask the `podman run` run command for ease of use.\n\n### Continuous Deployment\n\nAll commits will trigger a Continuous Deployment process using GitLab\'s CI/CD platform, configured in `.gitlab-ci.yml`.\n\n## Release procedure\n\nFor all releases:\n\n1. create a release branch\n2. close release in `CHANGELOG.md`\n3. push changes, merge the release branch into `master` and tag with version\n4. create a ServiceDesk request to deploy the new package version (and change/add environment variables if needed)\n5. re-deploy API documentation if needed\n\n## Feedback\n\nThe maintainer of this project is the BAS Mapping and Geographic Information Centre (MAGIC), they can be contacted at:\n[magic@bas.ac.uk](mailto:magic@bas.ac.uk).\n\n## Issue tracking\n\nThis project uses issue tracking, see the\n[Issue tracker](https://gitlab.data.bas.ac.uk/MAGIC/add-metadata-toolbox/issues) for more information.\n\n**Note:** Read & write access to this issue tracker is restricted. Contact the project maintainer to request access.\n\n## License\n\n© UK Research and Innovation (UKRI), 2020 - 2021, British Antarctic Survey.\n\nYou may use and re-use this software and associated documentation files free of charge in any format or medium, under\nthe terms of the Open Government Licence v3.0.\n\nYou may obtain a copy of the Open Government Licence at http://www.nationalarchives.gov.uk/doc/open-government-licence/\n',
    'author': 'Felix Fennell',
    'author_email': 'felnne@bas.ac.uk',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/antarctica/scar-add-metadata-toolbox',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
