// Generated SaaS Dashboard Workflow Test
// Framework: Playwright TypeScript
// Generated by Browse-to-Test with enhanced TypeScript patterns

import { test, expect, Page, BrowserContext } from '@playwright/test';
import { chromium, Browser } from '@playwright/test';

interface DashboardStats {
  users: string;
  revenue: string;
  conversion: string;
}

interface UserData {
  fullName: string;
  email: string;
  role: 'admin' | 'manager' | 'user';
}

class SaaSDashboardPage {
  constructor(private page: Page) {}

  // Authentication methods
  async login(email: string, password: string): Promise<void> {
    await this.page.fill('[data-testid="email-input"]', email);
    await this.page.fill('[data-testid="password-input"]', password);
    await this.page.click('[data-testid="login-button"]');
    
    // Wait for authentication to complete
    await this.page.waitForURL('**/dashboard', { timeout: 10000 });
  }

  // Dashboard interaction methods
  async waitForDashboardLoad(): Promise<void> {
    await this.page.waitForSelector('[data-testid="dashboard-stats"]', { timeout: 8000 });
    
    // Verify dashboard is fully loaded
    const statsContainer = this.page.locator('[data-testid="dashboard-stats"]');
    await expect(statsContainer).toHaveClass(/loaded/);
  }

  async getDashboardStats(): Promise<DashboardStats> {
    const statsContainer = this.page.locator('[data-testid="dashboard-stats"]');
    
    const users = await statsContainer.locator('.stat-card[data-metric="users"]').textContent() || '';
    const revenue = await statsContainer.locator('.stat-card[data-metric="revenue"]').textContent() || '';
    const conversion = await statsContainer.locator('.stat-card[data-metric="conversion"]').textContent() || '';
    
    return { users, revenue, conversion };
  }

  async navigateToUsers(): Promise<void> {
    await this.page.click('[data-nav="users"]');
    await this.page.waitForURL('**/users', { timeout: 5000 });
    await this.page.waitForSelector('.users-table', { timeout: 5000 });
  }

  // User management methods
  async createUser(userData: UserData): Promise<void> {
    // Open add user modal
    await this.page.click('[data-testid="add-user-btn"]');
    await this.page.waitForSelector('.modal.add-user-modal', { timeout: 3000 });
    
    // Verify modal is accessible
    const modal = this.page.locator('.modal.add-user-modal');
    await expect(modal).toHaveAttribute('aria-modal', 'true');
    
    // Fill user form
    await this.page.fill('[data-testid="user-name-input"]', userData.fullName);
    await this.page.fill('[data-testid="user-email-input"]', userData.email);
    await this.page.selectOption('[data-testid="user-role-select"]', userData.role);
    
    // Submit form
    await this.page.click('.save-user-btn');
    
    // Wait for success notification
    await this.page.waitForSelector('.success-notification', { timeout: 5000 });
    
    const notification = this.page.locator('.success-notification');
    await expect(notification).toContainText(`User '${userData.fullName}' created successfully`);
  }

  async verifyUserInTable(userData: UserData): Promise<void> {
    const userRow = this.page.locator('.users-table tbody tr').filter({
      hasText: userData.fullName
    });
    
    await expect(userRow).toBeVisible();
    await expect(userRow).toContainText(userData.email);
    await expect(userRow).toContainText(userData.role === 'manager' ? 'Manager' : userData.role === 'admin' ? 'Administrator' : 'Standard User');
    await expect(userRow).toContainText('Active');
  }
}

test.describe('SaaS Dashboard Workflow', () => {
  let browser: Browser;
  let context: BrowserContext;
  let page: Page;
  let dashboardPage: SaaSDashboardPage;

  test.beforeAll(async () => {
    browser = await chromium.launch({
      headless: process.env.HEADLESS !== 'false'
    });
  });

  test.afterAll(async () => {
    await browser.close();
  });

  test.beforeEach(async () => {
    context = await browser.newContext({
      viewport: { width: 1440, height: 900 },
      userAgent: 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36'
    });
    
    page = await context.newPage();
    dashboardPage = new SaaSDashboardPage(page);
    
    // Set up request/response monitoring
    page.on('request', request => {
      if (request.url().includes('/api/')) {
        console.log(`API Request: ${request.method()} ${request.url()}`);
      }
    });
    
    page.on('response', response => {
      if (response.url().includes('/api/') && !response.ok()) {
        console.error(`API Error: ${response.status()} ${response.url()}`);
      }
    });
  });

  test.afterEach(async () => {
    await context.close();
  });

  test('complete dashboard workflow with user management', async () => {
    try {
      // Step 1: Navigate to login page
      await page.goto('https://app.saas-platform.com/login', {
        timeout: 30000,
        waitUntil: 'networkidle'
      });

      // Verify login page loaded
      await expect(page).toHaveTitle('Login - SaaS Platform');
      await expect(page.locator('[data-testid="email-input"]')).toBeVisible();

      // Step 2-4: Perform login
      await dashboardPage.login('admin@company.com', 'SecurePass123!');

      // Verify successful authentication
      await expect(page).toHaveURL(/.*\/dashboard/);
      
      // Step 5-6: Wait for dashboard to load and verify stats
      await dashboardPage.waitForDashboardLoad();
      
      const stats = await dashboardPage.getDashboardStats();
      expect(stats.users).toContain('1,245 Active Users');
      expect(stats.revenue).toContain('$24,567 Revenue');
      expect(stats.conversion).toContain('3.2% Conversion');

      // Step 7-8: Navigate to users section
      await dashboardPage.navigateToUsers();
      
      // Verify users table is loaded
      const usersTable = page.locator('.users-table');
      await expect(usersTable).toBeVisible();
      await expect(usersTable).toHaveAttribute('role', 'table');

      // Step 9-15: Create new user
      const newUser: UserData = {
        fullName: 'Jane Smith',
        email: 'jane.smith@company.com',
        role: 'manager'
      };

      await dashboardPage.createUser(newUser);

      // Step 16: Verify user appears in table
      await dashboardPage.verifyUserInTable(newUser);

      // Additional validations for real-time features
      await test.step('Verify real-time updates', async () => {
        // Check if WebSocket connection is established for real-time updates
        const wsConnections = await page.evaluate(() => {
          // @ts-ignore - Access WebSocket connections for testing
          return window.WebSocket ? 'WebSocket available' : 'No WebSocket';
        });
        expect(wsConnections).toBe('WebSocket available');
      });

      // Performance validation
      await test.step('Performance checks', async () => {
        const performanceMetrics = await page.evaluate(() => {
          return {
            loadTime: performance.now(),
            domContentLoaded: performance.getEntriesByType('navigation')[0]
          };
        });
        
        // Ensure reasonable load times
        expect(performanceMetrics.loadTime).toBeLessThan(30000); // 30 seconds max
      });

      console.log('✓ SaaS dashboard workflow completed successfully');

    } catch (error) {
      // Enhanced error handling with context
      const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
      
      // Take screenshot
      await page.screenshot({
        path: `test-failure-dashboard-${timestamp}.png`,
        fullPage: true
      });
      
      // Capture page state
      const pageState = {
        url: page.url(),
        title: await page.title(),
        timestamp,
        error: error.message
      };
      
      console.error('Dashboard workflow test failed:', pageState);
      
      // Log network failures if any
      const failedRequests = await page.evaluate(() => {
        // @ts-ignore - Access failed requests for debugging
        return window.__failedRequests || [];
      });
      
      if (failedRequests.length > 0) {
        console.error('Failed network requests:', failedRequests);
      }
      
      throw new Error(`Dashboard workflow test failed: ${error.message}`);
    }
  });

  // Additional test for responsive design
  test('dashboard responsive behavior', async () => {
    // Test mobile viewport
    await context.setViewportSize({ width: 375, height: 667 });
    
    await page.goto('https://app.saas-platform.com/login');
    await dashboardPage.login('admin@company.com', 'SecurePass123!');
    await dashboardPage.waitForDashboardLoad();
    
    // Verify mobile-friendly layout
    const statsContainer = page.locator('[data-testid="dashboard-stats"]');
    const containerBox = await statsContainer.boundingBox();
    
    expect(containerBox?.width).toBeLessThanOrEqual(375);
    
    // Test tablet viewport
    await context.setViewportSize({ width: 768, height: 1024 });
    await page.reload();
    await dashboardPage.waitForDashboardLoad();
    
    // Verify tablet layout adjustments
    const tabletStats = await dashboardPage.getDashboardStats();
    expect(tabletStats.users).toBeTruthy();
  });
});