import { AccessLevelList } from '../shared/access-level';
import { PolicyStatement, Operator } from '../shared';
/**
 * Statement provider for service [connect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnect.html).
 *
 * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
 */
export declare class Connect extends PolicyStatement {
    servicePrefix: string;
    /**
     * Statement provider for service [connect](https://docs.aws.amazon.com/service-authorization/latest/reference/list_amazonconnect.html).
     *
     * @param sid [SID](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_sid.html) of the statement
     */
    constructor(sid?: string);
    /**
     * Grants permission to associate approved origin for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateApprovedOrigin(): this;
    /**
     * Grants permission to associate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - lex:CreateResourcePolicy
     * - lex:DescribeBotAlias
     * - lex:GetBot
     * - lex:UpdateResourcePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateBot(): this;
    /**
     * Grants permission to associate a Customer Profiles domain for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - profile:GetDomain
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateCustomerProfilesDomain(): this;
    /**
     * Grants permission to default vocabulary for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     */
    toAssociateDefaultVocabulary(): this;
    /**
     * Grants permission to associate instance storage for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - firehose:DescribeDeliveryStream
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kinesis:DescribeStream
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateInstanceStorageConfig(): this;
    /**
     * Grants permission to associate a Lambda function for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - lambda:AddPermission
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateLambdaFunction(): this;
    /**
     * Grants permission to associate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - lex:GetBot
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateLexBot(): this;
    /**
     * Grants permission to associate quick connects with a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateQueueQuickConnects.html
     */
    toAssociateQueueQuickConnects(): this;
    /**
     * Grants permission to associate queues with a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_AssociateRoutingProfileQueues.html
     */
    toAssociateRoutingProfileQueues(): this;
    /**
     * Grants permission to associate a security key for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toAssociateSecurityKey(): this;
    /**
     * Grants permission to grant access and to associate the datasets with the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/optimization-apis.html
     */
    toBatchAssociateAnalyticsDataSet(): this;
    /**
     * Grants permission to revoke access and to disassociate the datasets with the specified AWS account
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/optimization-apis.html
     */
    toBatchDisassociateAnalyticsDataSet(): this;
    /**
     * Grants permission to create agent status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateAgentStatus.html
     */
    toCreateAgentStatus(): this;
    /**
     * Grants permission to create a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContactFlow.html
     */
    toCreateContactFlow(): this;
    /**
     * Grants permission to create a contact flow module in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateContactFlowModule.html
     */
    toCreateContactFlowModule(): this;
    /**
     * Grants permission to create hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateHoursOfOperation.html
     */
    toCreateHoursOfOperation(): this;
    /**
     * Grants permission to create a new Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - ds:AuthorizeApplication
     * - ds:CheckAlias
     * - ds:CreateAlias
     * - ds:CreateDirectory
     * - ds:CreateIdentityPoolDirectory
     * - ds:DeleteDirectory
     * - ds:DescribeDirectories
     * - ds:UnauthorizeApplication
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toCreateInstance(): this;
    /**
     * Grants permission to create an integration association with an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - app-integrations:CreateEventIntegrationAssociation
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     * - events:PutRule
     * - events:PutTargets
     * - mobiletargeting:GetApp
     * - voiceid:DescribeDomain
     * - wisdom:GetAssistant
     * - wisdom:GetKnowledgeBase
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateIntegrationAssociation.html
     */
    toCreateIntegrationAssociation(): this;
    /**
     * Grants permission to create a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateQueue.html
     */
    toCreateQueue(): this;
    /**
     * Grants permission to create a quick connect in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateQuickConnect.html
     */
    toCreateQuickConnect(): this;
    /**
     * Grants permission to create a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateRoutingProfile.html
     */
    toCreateRoutingProfile(): this;
    /**
     * Grants permission to create a security profile for the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateSecurityProfile.html
     */
    toCreateSecurityProfile(): this;
    /**
     * Grants permission to create a use case for an integration association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUseCase.html
     */
    toCreateUseCase(): this;
    /**
     * Grants permission to create a user for the specified Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUser.html
     */
    toCreateUser(): this;
    /**
     * Grants permission to create a user hierarchy group in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateUserHierarchyGroup.html
     */
    toCreateUserHierarchyGroup(): this;
    /**
     * Grants permission to create a vocabulary in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsRequestTag()
     * - .ifAwsTagKeys()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_CreateVocabulary.html
     */
    toCreateVocabulary(): this;
    /**
     * Grants permission to delete a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteContactFlow.html
     */
    toDeleteContactFlow(): this;
    /**
     * Grants permission to delete a contact flow module in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteContactFlowModule.html
     */
    toDeleteContactFlowModule(): this;
    /**
     * Grants permission to delete hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteHoursOfOperation.html
     */
    toDeleteHoursOfOperation(): this;
    /**
     * Grants permission to delete an Amazon Connect instance. When you remove an instance, the link to an existing AWS directory is also removed
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ds:DeleteDirectory
     * - ds:DescribeDirectories
     * - ds:UnauthorizeApplication
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDeleteInstance(): this;
    /**
     * Grants permission to delete an integration association from an Amazon Connect instance. The association must not have any use cases associated with it
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - app-integrations:DeleteEventIntegrationAssociation
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     * - events:DeleteRule
     * - events:ListTargetsByRule
     * - events:RemoveTargets
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteIntegrationAssociation.html
     */
    toDeleteIntegrationAssociation(): this;
    /**
     * Grants permission to delete a quick connect in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteQuickConnect.html
     */
    toDeleteQuickConnect(): this;
    /**
     * Grants permission to delete a security profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteSecurityProfile.html
     */
    toDeleteSecurityProfile(): this;
    /**
     * Grants permission to delete a use case from an integration association
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUseCase.html
     */
    toDeleteUseCase(): this;
    /**
     * Grants permission to delete a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUser.html
     */
    toDeleteUser(): this;
    /**
     * Grants permission to delete a user hierarchy group in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteUserHierarchyGroup.html
     */
    toDeleteUserHierarchyGroup(): this;
    /**
     * Grants permission to delete a vocabulary in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DeleteVocabulary.html
     */
    toDeleteVocabulary(): this;
    /**
     * Grants permission to describe agent status in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeAgentStatus.html
     */
    toDescribeAgentStatus(): this;
    /**
     * Grants permission to describe a contact in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContact.html
     */
    toDescribeContact(): this;
    /**
     * Grants permission to describe a contact flow in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContactFlow.html
     */
    toDescribeContactFlow(): this;
    /**
     * Grants permission to describe a contact flow module in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeContactFlowModule.html
     */
    toDescribeContactFlowModule(): this;
    /**
     * Grants permission to describe hours of operation in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeHoursOfOperation.html
     */
    toDescribeHoursOfOperation(): this;
    /**
     * Grants permission to view details of an Amazon Connect instance and is also required to create an instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     * - .ifAwsResourceTag()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDescribeInstance(): this;
    /**
     * Grants permission to view the attribute details of an existing Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAttributeType()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDescribeInstanceAttribute(): this;
    /**
     * Grants permission to view the instance storage configuration for an existing Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDescribeInstanceStorageConfig(): this;
    /**
     * Grants permission to describe a queue in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeQueue.html
     */
    toDescribeQueue(): this;
    /**
     * Grants permission to describe a quick connect in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeQuickConnect.html
     */
    toDescribeQuickConnect(): this;
    /**
     * Grants permission to describe a routing profile in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeRoutingProfile.html
     */
    toDescribeRoutingProfile(): this;
    /**
     * Grants permission to describe a security profile in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeSecurityProfile.html
     */
    toDescribeSecurityProfile(): this;
    /**
     * Grants permission to describe a user in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUser.html
     */
    toDescribeUser(): this;
    /**
     * Grants permission to describe a hierarchy group for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUserHierarchyGroup.html
     */
    toDescribeUserHierarchyGroup(): this;
    /**
     * Grants permission to describe the hierarchy structure for an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeUserHierarchyStructure.html
     */
    toDescribeUserHierarchyStructure(): this;
    /**
     * Grants permission to describe a vocabulary in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DescribeVocabulary.html
     */
    toDescribeVocabulary(): this;
    /**
     * Grants permission to disassociate approved origin for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateApprovedOrigin(): this;
    /**
     * Grants permission to disassociate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - lex:DeleteResourcePolicy
     * - lex:UpdateResourcePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateBot(): this;
    /**
     * Grants permission to disassociate a Customer Profiles domain for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:DeleteRolePolicy
     * - iam:DetachRolePolicy
     * - iam:GetPolicy
     * - iam:GetPolicyVersion
     * - iam:GetRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateCustomerProfilesDomain(): this;
    /**
     * Grants permission to disassociate instance storage for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateInstanceStorageConfig(): this;
    /**
     * Grants permission to disassociate a Lambda function for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - lambda:RemovePermission
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateLambdaFunction(): this;
    /**
     * Grants permission to disassociate a Lex bot for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateLexBot(): this;
    /**
     * Grants permission to disassociate quick connects from a queue in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateQueueQuickConnects.html
     */
    toDisassociateQueueQuickConnects(): this;
    /**
     * Grants permission to disassociate queues from a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_DisassociateRoutingProfileQueues.html
     */
    toDisassociateRoutingProfileQueues(): this;
    /**
     * Grants permission to disassociate the security key for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toDisassociateSecurityKey(): this;
    /**
     * Grants permission to retrieve the contact attributes for the specified contact
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetContactAttributes.html
     */
    toGetContactAttributes(): this;
    /**
     * Grants permission to retrieve current metric data for the queues in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetCurrentMetricData.html
     */
    toGetCurrentMetricData(): this;
    /**
     * Grants permission to federate into an Amazon Connect instance when using SAML-based authentication for identity management
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetFederationToken.html
     */
    toGetFederationToken(): this;
    /**
     * Grants permission to federate into an Amazon Connect instance (Log in for emergency access functionality in the Amazon Connect console)
     *
     * Access Level: Write
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - connect:ListInstances
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/emergency-admin-login.html
     */
    toGetFederationTokens(): this;
    /**
     * Grants permission to retrieve historical metric data for queues in an Amazon Connect instance
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_GetMetricData.html
     */
    toGetMetricData(): this;
    /**
     * Grants permission to list agent statuses in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListAgentStatuses.html
     */
    toListAgentStatuses(): this;
    /**
     * Grants permission to view approved origins of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListApprovedOrigins(): this;
    /**
     * Grants permission to view the Lex bots of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListBots(): this;
    /**
     * Grants permission to list contact flow module resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactFlowModules.html
     */
    toListContactFlowModules(): this;
    /**
     * Grants permission to list contact flow resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactFlows.html
     */
    toListContactFlows(): this;
    /**
     * Grants permission to list references associated with a contact in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListContactReferences.html
     */
    toListContactReferences(): this;
    /**
     * Grants permission to list default vocabularies associated with a Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListDefaultVocabularies.html
     */
    toListDefaultVocabularies(): this;
    /**
     * Grants permission to list hours of operation resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListHoursOfOperations.html
     */
    toListHoursOfOperations(): this;
    /**
     * Grants permission to view the attributes of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListInstanceAttributes(): this;
    /**
     * Grants permission to view storage configurations of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListInstanceStorageConfigs(): this;
    /**
     * Grants permission to view the Amazon Connect instances associated with an AWS account
     *
     * Access Level: List
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListInstances(): this;
    /**
     * Grants permission to list summary information about the integration associations for the specified Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListIntegrationAssociations.html
     */
    toListIntegrationAssociations(): this;
    /**
     * Grants permission to view the Lambda functions of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListLambdaFunctions(): this;
    /**
     * Grants permission to view the Lex bots of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListLexBots(): this;
    /**
     * Grants permission to list phone number resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPhoneNumbers.html
     */
    toListPhoneNumbers(): this;
    /**
     * Grants permission to list prompt resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListPrompts.html
     */
    toListPrompts(): this;
    /**
     * Grants permission to list quick connect resources in a queue in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQueueQuickConnects.html
     */
    toListQueueQuickConnects(): this;
    /**
     * Grants permission to list queue resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQueues.html
     */
    toListQueues(): this;
    /**
     * Grants permission to list quick connect resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListQuickConnects.html
     */
    toListQuickConnects(): this;
    /**
     * Grants permission to list the analysis segments for a real-time analysis session
     *
     * Access Level: Read
     *
     * https://docs.aws.amazon.com/contact-lens/latest/APIReference/API_ListRealtimeContactAnalysisSegments.html
     */
    toListRealtimeContactAnalysisSegments(): this;
    /**
     * Grants permission to list queue resources in a routing profile in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRoutingProfileQueues.html
     */
    toListRoutingProfileQueues(): this;
    /**
     * Grants permission to list routing profile resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListRoutingProfiles.html
     */
    toListRoutingProfiles(): this;
    /**
     * Grants permission to view the security keys of an existing Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toListSecurityKeys(): this;
    /**
     * Grants permission to list permissions associated with security profile in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListSecurityProfilePermissions.html
     */
    toListSecurityProfilePermissions(): this;
    /**
     * Grants permission to list security profile resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListSecurityProfiles.html
     */
    toListSecurityProfiles(): this;
    /**
     * Grants permission to list tags for an Amazon Connect resource
     *
     * Access Level: Read
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListTagsForResource.html
     */
    toListTagsForResource(): this;
    /**
     * Grants permission to list the use cases of an integration association
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - connect:DescribeInstance
     * - ds:DescribeDirectories
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUseCases.html
     */
    toListUseCases(): this;
    /**
     * Grants permission to list the hierarchy group resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUserHierarchyGroups.html
     */
    toListUserHierarchyGroups(): this;
    /**
     * Grants permission to list user resources in an Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ListUsers.html
     */
    toListUsers(): this;
    /**
     * Grants permission to resume recording for the specified contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_ResumeContactRecording.html
     */
    toResumeContactRecording(): this;
    /**
     * Grants permission to search vocabularies in a Amazon Connect instance
     *
     * Access Level: List
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SearchVocabularies.html
     */
    toSearchVocabularies(): this;
    /**
     * Grants permission to initiate a chat using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartChatContact.html
     */
    toStartChatContact(): this;
    /**
     * Grants permission to start recording for the specified contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactRecording.html
     */
    toStartContactRecording(): this;
    /**
     * Grants permission to start chat streaming using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartContactStreaming.html
     */
    toStartContactStreaming(): this;
    /**
     * Grants permission to initiate outbound calls using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartOutboundVoiceContact.html
     */
    toStartOutboundVoiceContact(): this;
    /**
     * Grants permission to initiate a task using the Amazon Connect API
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StartTaskContact.html
     */
    toStartTaskContact(): this;
    /**
     * Grants permission to stop contacts that were initiated using the Amazon Connect API. If you use this operation on an active contact the contact ends, even if the agent is active on a call with a customer
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContact.html
     */
    toStopContact(): this;
    /**
     * Grants permission to stop recording for the specified contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContactRecording.html
     */
    toStopContactRecording(): this;
    /**
     * Grants permission to stop chat streaming using the Amazon Connect API
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_StopContactStreaming.html
     */
    toStopContactStreaming(): this;
    /**
     * Grants permission to suspend recording for the specified contact
     *
     * Access Level: Write
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_SuspendContactRecording.html
     */
    toSuspendContactRecording(): this;
    /**
     * Grants permission to tag an Amazon Connect resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_TagResource.html
     */
    toTagResource(): this;
    /**
     * Grants permission to untag an Amazon Connect resource
     *
     * Access Level: Tagging
     *
     * Possible conditions:
     * - .ifAwsTagKeys()
     * - .ifAwsRequestTag()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UntagResource.html
     */
    toUntagResource(): this;
    /**
     * Grants permission to update agent status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateAgentStatus.html
     */
    toUpdateAgentStatus(): this;
    /**
     * Grants permission to update a contact in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContact.html
     */
    toUpdateContact(): this;
    /**
     * Grants permission to create or update the contact attributes associated with the specified contact
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactAttributes.html
     */
    toUpdateContactAttributes(): this;
    /**
     * Grants permission to update contact flow content in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowContent.html
     */
    toUpdateContactFlowContent(): this;
    /**
     * Grants permission to update the metadata of a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowMetadata.html
     */
    toUpdateContactFlowMetadata(): this;
    /**
     * Grants permission to update contact flow module content in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowModuleContent.html
     */
    toUpdateContactFlowModuleContent(): this;
    /**
     * Grants permission to update the metadata of a contact flow module in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowModuleMetadata.html
     */
    toUpdateContactFlowModuleMetadata(): this;
    /**
     * Grants permission to update the name and description of a contact flow in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactFlowName.html
     */
    toUpdateContactFlowName(): this;
    /**
     * Grants permission to update the schedule of a contact that is already scheduled in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateContactSchedule.html
     */
    toUpdateContactSchedule(): this;
    /**
     * Grants permission to update hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateHoursOfOperation.html
     */
    toUpdateHoursOfOperation(): this;
    /**
     * Grants permission to update the attribute for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAttributeType()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - logs:CreateLogGroup
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toUpdateInstanceAttribute(): this;
    /**
     * Grants permission to update the storage configuration for an existing Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifStorageResourceType()
     * - .ifInstanceId()
     *
     * Dependent actions:
     * - ds:DescribeDirectories
     * - firehose:DescribeDeliveryStream
     * - iam:AttachRolePolicy
     * - iam:CreateServiceLinkedRole
     * - iam:PutRolePolicy
     * - kinesis:DescribeStream
     * - kms:CreateGrant
     * - kms:DescribeKey
     * - s3:GetBucketAcl
     * - s3:GetBucketLocation
     *
     * https://docs.aws.amazon.com/console/connect/amazon-connect-console/grant-instance-permissions
     */
    toUpdateInstanceStorageConfig(): this;
    /**
     * Grants permission to update queue hours of operation in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueHoursOfOperation.html
     */
    toUpdateQueueHoursOfOperation(): this;
    /**
     * Grants permission to update queue capacity in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueMaxContacts.html
     */
    toUpdateQueueMaxContacts(): this;
    /**
     * Grants permission to update a queue name and description in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueName.html
     */
    toUpdateQueueName(): this;
    /**
     * Grants permission to update queue outbound caller config in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueOutboundCallerConfig.html
     */
    toUpdateQueueOutboundCallerConfig(): this;
    /**
     * Grants permission to update queue status in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQueueStatus.html
     */
    toUpdateQueueStatus(): this;
    /**
     * Grants permission to update the configuration of a quick connect in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQuickConnectConfig.html
     */
    toUpdateQuickConnectConfig(): this;
    /**
     * Grants permission to update a quick connect name and description in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateQuickConnectName.html
     */
    toUpdateQuickConnectName(): this;
    /**
     * Grants permission to update the concurrency in a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileConcurrency.html
     */
    toUpdateRoutingProfileConcurrency(): this;
    /**
     * Grants permission to update the outbound queue in a routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileDefaultOutboundQueue.html
     */
    toUpdateRoutingProfileDefaultOutboundQueue(): this;
    /**
     * Grants permission to update a routing profile name and description in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileName.html
     */
    toUpdateRoutingProfileName(): this;
    /**
     * Grants permission to update the queues in routing profile in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateRoutingProfileQueues.html
     */
    toUpdateRoutingProfileQueues(): this;
    /**
     * Grants permission to update a security profile group for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateSecurityProfile.html
     */
    toUpdateSecurityProfile(): this;
    /**
     * Grants permission to update a hierarchy group for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchy.html
     */
    toUpdateUserHierarchy(): this;
    /**
     * Grants permission to update a user hierarchy group name in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchyGroupName.html
     */
    toUpdateUserHierarchyGroupName(): this;
    /**
     * Grants permission to update user hierarchy structure in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserHierarchyStructure.html
     */
    toUpdateUserHierarchyStructure(): this;
    /**
     * Grants permission to update identity information for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserIdentityInfo.html
     */
    toUpdateUserIdentityInfo(): this;
    /**
     * Grants permission to update phone configuration settings for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserPhoneConfig.html
     */
    toUpdateUserPhoneConfig(): this;
    /**
     * Grants permission to update a routing profile for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserRoutingProfile.html
     */
    toUpdateUserRoutingProfile(): this;
    /**
     * Grants permission to update security profiles for a user in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdateUserSecurityProfiles.html
     */
    toUpdateUserSecurityProfiles(): this;
    /**
     * Grants permission to update contact flow module content in an Amazon Connect instance
     *
     * Access Level: Write
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     * - .ifInstanceId()
     *
     * https://docs.aws.amazon.com/connect/latest/APIReference/API_UpdatedescribeContent.html
     */
    toUpdatedescribeContent(): this;
    protected accessLevelList: AccessLevelList;
    /**
     * Adds a resource of type instance to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/amazon-connect-instances.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onInstance(instanceId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-attributes.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactId - Identifier for the contactId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onContact(instanceId: string, contactId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type user to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-agents.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param userId - Identifier for the userId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUser(instanceId: string, userId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type routing-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/routing-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param routingProfileId - Identifier for the routingProfileId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onRoutingProfile(instanceId: string, routingProfileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type security-profile to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-security-profiles.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param securityProfileId - Identifier for the securityProfileId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onSecurityProfile(instanceId: string, securityProfileId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type hierarchy-group to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/agent-hierarchy.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param hierarchyGroupId - Identifier for the hierarchyGroupId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHierarchyGroup(instanceId: string, hierarchyGroupId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type queue to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/API_Queue.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param queueId - Identifier for the queueId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQueue(instanceId: string, queueId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-queue to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/API_Queue.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWildcardQueue(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type quick-connect to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/quick-connects.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param quickConnectId - Identifier for the quickConnectId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onQuickConnect(instanceId: string, quickConnectId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-quick-connect to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/quick-connects.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWildcardQuickConnect(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact-flow to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-flows.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactFlowId - Identifier for the contactFlowId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactFlow(instanceId: string, contactFlowId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type contact-flow-module to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/contact-flow-modules.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param contactFlowModuleId - Identifier for the contactFlowModuleId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onContactFlowModule(instanceId: string, contactFlowModuleId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-contact-flow to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/connect-contact-flows.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWildcardContactFlow(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type hours-of-operation to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/API_HoursOfOperation.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param hoursOfOperationId - Identifier for the hoursOfOperationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onHoursOfOperation(instanceId: string, hoursOfOperationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type agent-status to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/agent-status.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param agentStatusId - Identifier for the agentStatusId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onAgentStatus(instanceId: string, agentStatusId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-agent-status to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/agent-status.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWildcardAgentStatus(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/contact-center-phone-number.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param phoneNumberId - Identifier for the phoneNumberId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onPhoneNumber(instanceId: string, phoneNumberId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type wildcard-phone-number to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/contact-center-phone-number.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param resourceName - Identifier for the resourceName.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     */
    onWildcardPhoneNumber(instanceId: string, resourceName: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type integration-association to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/API_IntegrationAssociationSummary.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param integrationAssociationId - Identifier for the integrationAssociationId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onIntegrationAssociation(instanceId: string, integrationAssociationId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type use-case to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/API_UseCase.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param useCaseId - Identifier for the useCaseId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onUseCase(instanceId: string, useCaseId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Adds a resource of type vocabulary to the statement
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/API_Vocabulary.html
     *
     * @param instanceId - Identifier for the instanceId.
     * @param vocabularyId - Identifier for the vocabularyId.
     * @param account - Account of the resource; defaults to empty string: all accounts.
     * @param region - Region of the resource; defaults to empty string: all regions.
     * @param partition - Partition of the AWS account [aws, aws-cn, aws-us-gov]; defaults to `aws`.
     *
     * Possible conditions:
     * - .ifAwsResourceTag()
     */
    onVocabulary(instanceId: string, vocabularyId: string, account?: string, region?: string, partition?: string): this;
    /**
     * Filters access by the attribute type of the Amazon Connect instance
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toDescribeInstanceAttribute()
     * - .toUpdateInstanceAttribute()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifAttributeType(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting federation into specified Amazon Connect instances
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateApprovedOrigin()
     * - .toAssociateBot()
     * - .toAssociateDefaultVocabulary()
     * - .toAssociateInstanceStorageConfig()
     * - .toAssociateLambdaFunction()
     * - .toAssociateLexBot()
     * - .toAssociateQueueQuickConnects()
     * - .toAssociateRoutingProfileQueues()
     * - .toAssociateSecurityKey()
     * - .toBatchAssociateAnalyticsDataSet()
     * - .toBatchDisassociateAnalyticsDataSet()
     * - .toCreateAgentStatus()
     * - .toCreateContactFlow()
     * - .toCreateContactFlowModule()
     * - .toCreateHoursOfOperation()
     * - .toCreateIntegrationAssociation()
     * - .toCreateQueue()
     * - .toCreateQuickConnect()
     * - .toCreateRoutingProfile()
     * - .toCreateSecurityProfile()
     * - .toCreateUseCase()
     * - .toCreateUser()
     * - .toCreateUserHierarchyGroup()
     * - .toCreateVocabulary()
     * - .toDeleteContactFlow()
     * - .toDeleteContactFlowModule()
     * - .toDeleteHoursOfOperation()
     * - .toDeleteInstance()
     * - .toDeleteIntegrationAssociation()
     * - .toDeleteQuickConnect()
     * - .toDeleteSecurityProfile()
     * - .toDeleteUseCase()
     * - .toDeleteUser()
     * - .toDeleteUserHierarchyGroup()
     * - .toDeleteVocabulary()
     * - .toDescribeAgentStatus()
     * - .toDescribeContact()
     * - .toDescribeContactFlow()
     * - .toDescribeContactFlowModule()
     * - .toDescribeHoursOfOperation()
     * - .toDescribeInstance()
     * - .toDescribeInstanceAttribute()
     * - .toDescribeInstanceStorageConfig()
     * - .toDescribeQueue()
     * - .toDescribeQuickConnect()
     * - .toDescribeRoutingProfile()
     * - .toDescribeSecurityProfile()
     * - .toDescribeUser()
     * - .toDescribeUserHierarchyGroup()
     * - .toDescribeUserHierarchyStructure()
     * - .toDescribeVocabulary()
     * - .toDisassociateApprovedOrigin()
     * - .toDisassociateBot()
     * - .toDisassociateInstanceStorageConfig()
     * - .toDisassociateLambdaFunction()
     * - .toDisassociateLexBot()
     * - .toDisassociateQueueQuickConnects()
     * - .toDisassociateRoutingProfileQueues()
     * - .toDisassociateSecurityKey()
     * - .toGetContactAttributes()
     * - .toGetCurrentMetricData()
     * - .toGetFederationToken()
     * - .toGetMetricData()
     * - .toListApprovedOrigins()
     * - .toListBots()
     * - .toListContactReferences()
     * - .toListDefaultVocabularies()
     * - .toListHoursOfOperations()
     * - .toListInstanceAttributes()
     * - .toListInstanceStorageConfigs()
     * - .toListIntegrationAssociations()
     * - .toListLambdaFunctions()
     * - .toListLexBots()
     * - .toListPrompts()
     * - .toListQueueQuickConnects()
     * - .toListRoutingProfileQueues()
     * - .toListRoutingProfiles()
     * - .toListSecurityKeys()
     * - .toListSecurityProfilePermissions()
     * - .toListSecurityProfiles()
     * - .toListUseCases()
     * - .toListUserHierarchyGroups()
     * - .toListUsers()
     * - .toSearchVocabularies()
     * - .toStartTaskContact()
     * - .toStopContact()
     * - .toUpdateAgentStatus()
     * - .toUpdateContact()
     * - .toUpdateContactAttributes()
     * - .toUpdateContactFlowContent()
     * - .toUpdateContactFlowMetadata()
     * - .toUpdateContactFlowModuleContent()
     * - .toUpdateContactFlowModuleMetadata()
     * - .toUpdateContactFlowName()
     * - .toUpdateContactSchedule()
     * - .toUpdateHoursOfOperation()
     * - .toUpdateInstanceAttribute()
     * - .toUpdateInstanceStorageConfig()
     * - .toUpdateQueueHoursOfOperation()
     * - .toUpdateQueueMaxContacts()
     * - .toUpdateQueueName()
     * - .toUpdateQueueOutboundCallerConfig()
     * - .toUpdateQueueStatus()
     * - .toUpdateQuickConnectConfig()
     * - .toUpdateQuickConnectName()
     * - .toUpdateRoutingProfileConcurrency()
     * - .toUpdateRoutingProfileDefaultOutboundQueue()
     * - .toUpdateRoutingProfileName()
     * - .toUpdateRoutingProfileQueues()
     * - .toUpdateSecurityProfile()
     * - .toUpdateUserHierarchy()
     * - .toUpdateUserHierarchyGroupName()
     * - .toUpdateUserHierarchyStructure()
     * - .toUpdateUserIdentityInfo()
     * - .toUpdateUserPhoneConfig()
     * - .toUpdateUserRoutingProfile()
     * - .toUpdateUserSecurityProfiles()
     * - .toUpdatedescribeContent()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifInstanceId(value: string | string[], operator?: Operator | string): this;
    /**
     * Filters access by restricting the storage resource type of the Amazon Connect instance storage configuration
     *
     * https://docs.aws.amazon.com/connect/latest/adminguide/security_iam_service-with-iam.html
     *
     * Applies to actions:
     * - .toAssociateInstanceStorageConfig()
     * - .toDescribeInstanceStorageConfig()
     * - .toDisassociateInstanceStorageConfig()
     * - .toUpdateInstanceStorageConfig()
     *
     * @param value The value(s) to check
     * @param operator Works with [string operators](https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_elements_condition_operators.html#Conditions_String). **Default:** `StringLike`
     */
    ifStorageResourceType(value: string | string[], operator?: Operator | string): this;
}
