import itertools
import pathlib
import jinja2

from gosling.examples import iter_examples


GALLERY_TEMPLATE = jinja2.Template(
    """
.. This document is auto-generated by the gosling-gallery extension. Do not modify directly.

.. _{{ gallery_ref }}:

{{ title }}
{% for char in title %}-{% endfor %}

This gallery contains a selection of examples of the visualizations **gos** can create.

.. toctree::

{% for example in examples %}
    {{ example.name }}
{%- endfor %}
"""
)

EXAMPLE_TEMPLATE = jinja2.Template(
    """
.. This document is auto-generated by the gosling-gallery extension. Do not modify directly.

.. _gallery_{{ name }}:

{{ title }}
{% for char in title %}-{% endfor %}

{{ docstring }}

.. gosling-plot::
    {% if code_below %}:code-below:{% endif %}

    {{ code | indent(4) }}
"""
)


def prev_this_next(it, sentinel=None):
    """Utility to return (prev, this, next) tuples from an iterator"""
    i1, i2, i3 = itertools.tee(it, 3)
    next(i3, None)
    return zip(itertools.chain([sentinel], i1), i2, itertools.chain(i3, [sentinel]))


def populate_examples():
    return [
        dict(
            docstring="",
            name=example.stem,
            title=" ".join(list(map(lambda s: s.capitalize(), example.stem.split("_")))),
            code=example.read_text(),
            code_below=True,
        )
        for example in sorted(iter_examples())
    ]


def main(app):
    srcdir = pathlib.Path(app.builder.srcdir)
    title = "Example Gallery"
    gallery_dir = "gallery"
    gallery_ref = "example-gallery"
    target_dir = srcdir / gallery_dir

    if not target_dir.is_dir():
        target_dir.mkdir(parents=True)

    examples = populate_examples()

    # Write the gallery index file
    with open(target_dir / "index.rst", "w") as f:
        f.write(
            GALLERY_TEMPLATE.render(
                title=title,
                examples=examples,
                gallery_ref=gallery_ref,
            )
        )

    for prev_ex, example, next_ex in prev_this_next(examples):
        if prev_ex:
            example["prev_ref"] = f"gallery_{prev_ex['name']}"
        if next_ex:
            example["next_ref"] = f"gallery_{next_ex['name']}"
        with open(target_dir / (example["name"] + ".rst"), "w") as f:
            f.write(EXAMPLE_TEMPLATE.render(example))


def setup(app):
    app.connect("builder-inited", main)
