"""
Choice model representing a single candidate response from an AI model.

This module defines the Choice model, which represents a single candidate response
from an AI generation. When models provide multiple alternative responses to the
same input (controlled by the 'n' parameter), each alternative is represented as a
Choice with its own index and message content.

The Choice model uses a generic type parameter to support structured parsing of
responses, allowing for type-safe access to parsed data when available.
"""

from agentle.generations.models.messages.generated_assistant_message import (
    GeneratedAssistantMessage,
)
from rsb.decorators.value_objects import valueobject
from rsb.models.base_model import BaseModel
from rsb.models.field import Field


@valueobject
class Choice[T](BaseModel):
    """
    A single candidate response from an AI generation.

    Choice objects represent individual responses when a model provides multiple
    alternative completions for the same input. Each Choice contains an index
    indicating its position in the array of choices and a message containing
    the actual content generated by the model.

    The generic type parameter T allows for typed access to parsed structured
    data when the model response has been parsed into a specific schema.

    Attributes:
        index: Zero-based position of this choice in the array of choices.
        message: The response content generated by the model, potentially
            containing both text and structured data of type T.
    """

    index: int = Field(
        description="The zero-based index position of this choice in the sequence of responses. Used for identification and ordering when multiple alternative choices are generated.",
        ge=0,  # Must be greater than or equal to 0
        examples=[0, 1, 2],
    )

    message: GeneratedAssistantMessage[T] = Field(
        description="The complete message content produced by the AI model for this choice. Contains all generated text, tool calls, and structured data (if parsed). This represents the actual output content for this alternative response.",
        kw_only=False,
    )
