""" Unit Test Query
"""
import unittest
from unittest.case import TestCase

from mock import patch
from core_main_app.commons import exceptions
from core_main_app.utils.tests_tools.MockUser import create_mock_user
from core_explore_common_app.components.abstract_query.models import (
    Authentication,
    DataSource,
)
from core_explore_common_app.components.query import api as query_api
from core_explore_common_app.components.query.models import Query


class TestQueryUpsert(TestCase):
    """Test Query Upsert"""

    @patch.object(Query, "save")
    def test_upsert_query_returns_query(self, mock_save):
        """test_upsert_query_returns_query

        Returns:

        """
        query = _create_query()
        mock_save.return_value = query
        mock_user = create_mock_user("1")
        self.assertTrue(isinstance(query_api.upsert(query, mock_user), Query))


class TestQueryGetById(TestCase):
    """Test Query Get By Id"""

    @patch.object(Query, "get_by_id")
    def test_saved_query_get_by_id_raises_api_error_if_not_found(self, mock_get):
        """test_saved_query_get_by_id_raises_api_error_if_not_found

        Returns:

        """
        # Arrange
        mock_get.side_effect = exceptions.DoesNotExist("")
        mock_user = create_mock_user("1")
        # Act # Assert
        with self.assertRaises(exceptions.DoesNotExist):
            query_api.get_by_id("1", mock_user)

    @patch.object(Query, "get_by_id")
    def test_query_get_by_id_return_data_if_found(self, mock_get):
        """test_query_get_by_id_return_data_if_found

        Returns:

        """
        # Arrange
        query = _create_query()
        mock_get.return_value = query
        mock_user = create_mock_user("1")
        # Act
        result = query_api.get_by_id(query.id, mock_user)
        # Assert
        self.assertIsInstance(result, Query)


class TestGetDataSourceByNameAndUrlQuery(TestCase):
    """Test Get Data Source By Name And Url Query"""

    @patch.object(Query, "get_data_source_by_name_and_url_query")
    def test_saved_query_get_by_name_and_url_raises_api_error_if_not_found(
        self, mock_get
    ):
        """test_saved_query_get_by_name_and_url_raises_api_error_if_not_found

        Returns:

        """
        # create query
        query = _create_query()
        # Arrange
        mock_get.side_effect = exceptions.DoesNotExist("")
        mock_user = create_mock_user("1")
        # Act # Assert
        with self.assertRaises(exceptions.DoesNotExist):
            query_api.get_data_source_by_name_and_url_query(
                query, "name", "url", mock_user
            )

    @patch.object(Query, "get_data_source_by_name_and_url_query")
    def test_query_get_by_id_return_data_if_found(self, mock_get):
        """test_query_get_by_id_return_data_if_found

        Returns:

        """
        # create query
        query = _create_query()
        # Arrange
        mock_get.return_value = query
        mock_user = create_mock_user("1")
        # Act
        result = query_api.get_data_source_by_name_and_url_query(
            query, "Data Source", "/url", mock_user
        )
        # Assert
        self.assertIsInstance(result, Query)


class TestQueryAddDataSource(TestCase):
    """Test Query Add Data Source"""

    @patch.object(Query, "save")
    @patch.object(Query, "get_data_source_by_name_and_url_query")
    def test_add_data_source_adds_data_source_if_not_found(self, mock_get, mock_save):
        """test_add_data_source_adds_data_source_if_not_found

        Returns:

        """
        # create query
        query = _create_query()
        # Arrange
        mock_get.side_effect = exceptions.DoesNotExist("")
        mock_save.return_value = query
        mock_user = create_mock_user("1")

        origin_data_sources = len(query.data_sources)
        data_source = _create_data_source("Remote", "/remote/url")
        query_api.add_data_source(query, data_source, mock_user)
        self.assertTrue(len(query.data_sources) == origin_data_sources + 1)

    @unittest.skip("Need database to set data sources")
    @patch.object(Query, "save")
    @patch.object(Query, "get_data_source_by_name_and_url_query")
    def test_add_data_source_does_not_add_data_source_if_found(
        self, mock_get, mock_save
    ):
        """test_add_data_source_does_not_add_data_source_if_found

        Returns:

        """
        # create query
        query = _create_query()
        # Arrange
        mock_get.return_value = query.data_sources[0]
        mock_save.return_value = query
        mock_user = create_mock_user("1")

        origin_data_sources = len(query.data_sources)
        data_source = _create_data_source("Remote", "/remote/url")
        query_api.add_data_source(query, data_source, mock_user)
        self.assertTrue(len(query.data_sources) == origin_data_sources)


class TestQueryRemoveDataSource(TestCase):
    """Test Query Remove Data Source"""

    @unittest.skip("Need database to set data sources")
    @patch.object(Query, "save")
    def test_remove_data_source(self, mock_save):
        """test_remove_data_source

        Returns:

        """
        # create query
        query = _create_query()
        mock_save.return_value = query
        mock_user = create_mock_user("1")
        origin_data_sources = len(query.data_sources)
        data_source = _create_data_source()
        query_api.remove_data_source(query, data_source, mock_user)
        self.assertTrue(len(query.data_sources) == origin_data_sources - 1)


def _create_data_source(name="Local", url="/url"):
    authentication = Authentication(auth_type="session")
    data_source = DataSource(name=name, url_query=url, authentication=authentication)
    return data_source


def _create_query():
    query = Query(
        user_id="1",
        content="{'root.value': 'test'}",
    )
    return query
