"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.deserialize = exports.serialize = void 0;
const yaml = require("yaml");
const yaml_types = require("yaml/types");
/**
 * Serializes the given data structure into valid YAML.
 *
 * @param obj the data structure to serialize
 * @returns a string containing the YAML representation of {@param obj}
 */
function serialize(obj) {
    const oldFold = yaml_types.strOptions.fold.lineWidth;
    try {
        yaml_types.strOptions.fold.lineWidth = 0;
        return yaml.stringify(obj, { schema: 'yaml-1.1' });
    }
    finally {
        yaml_types.strOptions.fold.lineWidth = oldFold;
    }
}
exports.serialize = serialize;
/**
 * Deserialize the YAML into the appropriate data structure.
 *
 * @param str the string containing YAML
 * @returns the data structure the YAML represents
 *   (most often in case of CloudFormation, an object)
 */
function deserialize(str) {
    return parseYamlStrWithCfnTags(str);
}
exports.deserialize = deserialize;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: (_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        },
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or', 'GetAtt',
].map(name => makeTagForCfnIntrinsic(name, true)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'core',
    });
}
//# sourceMappingURL=data:application/json;base64,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