# Copyright (c) 2019-2021 - for information on the respective copyright owner
# see the NOTICE file and/or the repository
# https://github.com/boschresearch/pylife
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

__author__ = "Simone Schreijäg"
__maintainer__ = "Johannes Mueller"

import numpy as np


def true_strain(tech_strain):
    """
    Calculation of true strain data (from experimental data generated by tensile experiments)

    Parameters
    ----------
    tech_strain: array-like float

    Returns
    -------
    true_strain: array-like float

   """
    return np.log(1. + tech_strain)


def true_stress(tech_stress, tech_strain):
    """
     Calculate the true stress data from technical data

    Parameters
    ----------
    tech_stress : array-like float
        stress data from tensile experiments
    tech_strain : list of float
        strain data from tensile experiments

    Returns
    -------
    true_stress : array-like float

    """
    return tech_stress * (1. + tech_strain)


def true_fracture_strain(reduction_area_fracture):
    """
    Calculation of the true frature strain (in the FKM Non Linear (static assessment))
    Parameters
    ----------
    reduction_area_fracture : float
        directly measured on the fractures sample

    Returns
    -------
    true_fracture_strain: float
        descrivbes the calculated true fracture strain.

    """
    return np.log(1./(1. - reduction_area_fracture))


def true_fracture_stress(fracture_force, initial_cross_section, reduction_area_fracture):
    """
    Calculation of the true fracture stress (euqation FKM Non-linear (static assessment))

    Parameters
    ----------
    fracture_force : float
        from experimental results
    initial_cross_section : float
        cross section of initial tensile sample.
    reduction_area_fracture : float
         directly measured on the fractures sample.

    Returns
    -------
    true_fracture_stress: float
        calculated true fracture stress of the sample.

    """
    return fracture_force/(initial_cross_section * (1. - reduction_area_fracture))
