"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Canary = exports.Runtime = exports.Test = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const crypto = require("crypto");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const synthetics_canned_metrics_generated_1 = require("./synthetics-canned-metrics.generated");
const synthetics_generated_1 = require("./synthetics.generated");
/**
 * (experimental) Specify a test that the canary should run.
 *
 * @experimental
 */
class Test {
    /**
     * Construct a Test property
     *
     * @param code The code that the canary should run
     * @param handler The handler of the canary
     */
    constructor(code, handler) {
        this.code = code;
        this.handler = handler;
    }
    /**
     * (experimental) Specify a custom test with your own code.
     *
     * @param options The configuration options.
     * @returns `Test` associated with the specified Code object
     * @experimental
     */
    static custom(options) {
        Test.validateHandler(options.handler);
        return new Test(options.code, options.handler);
    }
    /**
     * Verifies that the given handler ends in '.handler'. Returns the handler if successful and
     * throws an error if not.
     *
     * @param handler - the handler given by the user
     */
    static validateHandler(handler) {
        if (!handler.endsWith('.handler')) {
            throw new Error(`Canary Handler must end in '.handler' (${handler})`);
        }
        if (handler.length > 21) {
            throw new Error(`Canary Handler must be less than 21 characters (${handler})`);
        }
    }
}
exports.Test = Test;
_a = JSII_RTTI_SYMBOL_1;
Test[_a] = { fqn: "@aws-cdk/aws-synthetics.Test", version: "1.95.0" };
/**
 * (experimental) Runtime options for a canary.
 *
 * @experimental
 */
class Runtime {
    /**
     * @param name The name of the runtime version.
     * @experimental
     */
    constructor(name) {
        this.name = name;
    }
}
exports.Runtime = Runtime;
_b = JSII_RTTI_SYMBOL_1;
Runtime[_b] = { fqn: "@aws-cdk/aws-synthetics.Runtime", version: "1.95.0" };
/**
 * (experimental) `syn-1.0` includes the following:.
 *
 * - Synthetics library 1.0
 * - Synthetics handler code 1.0
 * - Lambda runtime Node.js 10.x
 * - Puppeteer-core version 1.14.0
 * - The Chromium version that matches Puppeteer-core 1.14.0
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-1.0
 * @experimental
 */
Runtime.SYNTHETICS_1_0 = new Runtime('syn-1.0');
/**
 * (experimental) `syn-nodejs-2.0` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 83.0.4103.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-2.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_0 = new Runtime('syn-nodejs-2.0');
/**
 * (experimental) `syn-nodejs-2.1` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 83.0.4103.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-2.1
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_1 = new Runtime('syn-nodejs-2.1');
/**
 * (experimental) `syn-nodejs-2.2` includes the following: - Lambda runtime Node.js 10.x - Puppeteer-core version 3.3.0 - Chromium version 83.0.4103.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-2.2
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_2_2 = new Runtime('syn-nodejs-2.2');
/**
 * (experimental) `syn-nodejs-puppeteer-3.0` includes the following: - Lambda runtime Node.js 12.x - Puppeteer-core version 5.5.0 - Chromium version 88.0.4298.0.
 *
 * @see https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Synthetics_Library_nodejs_puppeteer.html#CloudWatch_Synthetics_runtimeversion-nodejs-puppeteer-3.0
 * @experimental
 */
Runtime.SYNTHETICS_NODEJS_PUPPETEER_3_0 = new Runtime('syn-nodejs-puppeteer-3.0');
/**
 * (experimental) Define a new Canary.
 *
 * @experimental
 */
class Canary extends cdk.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _d, _e, _f, _g, _h, _j, _k, _l;
        if (props.canaryName && !cdk.Token.isUnresolved(props.canaryName)) {
            validateName(props.canaryName);
        }
        super(scope, id, {
            physicalName: props.canaryName || cdk.Lazy.string({
                produce: () => this.generateUniqueName(),
            }),
        });
        this.artifactsBucket = (_e = (_d = props.artifactsBucketLocation) === null || _d === void 0 ? void 0 : _d.bucket) !== null && _e !== void 0 ? _e : new s3.Bucket(this, 'ArtifactsBucket', {
            encryption: s3.BucketEncryption.KMS_MANAGED,
        });
        this.role = (_f = props.role) !== null && _f !== void 0 ? _f : this.createDefaultRole((_g = props.artifactsBucketLocation) === null || _g === void 0 ? void 0 : _g.prefix);
        const resource = new synthetics_generated_1.CfnCanary(this, 'Resource', {
            artifactS3Location: this.artifactsBucket.s3UrlForObject((_h = props.artifactsBucketLocation) === null || _h === void 0 ? void 0 : _h.prefix),
            executionRoleArn: this.role.roleArn,
            startCanaryAfterCreation: (_j = props.startAfterCreation) !== null && _j !== void 0 ? _j : true,
            runtimeVersion: props.runtime.name,
            name: this.physicalName,
            schedule: this.createSchedule(props),
            failureRetentionPeriod: (_k = props.failureRetentionPeriod) === null || _k === void 0 ? void 0 : _k.toDays(),
            successRetentionPeriod: (_l = props.successRetentionPeriod) === null || _l === void 0 ? void 0 : _l.toDays(),
            code: this.createCode(props),
        });
        this.canaryId = resource.attrId;
        this.canaryState = resource.attrState;
        this.canaryName = this.getResourceNameAttribute(resource.ref);
    }
    /**
     * (experimental) Measure the Duration of a single canary run, in seconds.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricDuration(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.durationAverage, options);
    }
    /**
     * (experimental) Measure the percentage of successful canary runs.
     *
     * @param options - configuration options for the metric.
     * @default avg over 5 minutes
     * @experimental
     */
    metricSuccessPercent(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.successPercentAverage, options);
    }
    /**
     * (experimental) Measure the number of failed canary runs over a given time period.
     *
     * Default: sum over 5 minutes
     *
     * @param options - configuration options for the metric.
     * @experimental
     */
    metricFailed(options) {
        return this.cannedMetric(synthetics_canned_metrics_generated_1.CloudWatchSyntheticsMetrics.failedSum, options);
    }
    /**
     * Returns a default role for the canary
     */
    createDefaultRole(prefix) {
        const { partition } = cdk.Stack.of(this);
        // Created role will need these policies to run the Canary.
        // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-synthetics-canary.html#cfn-synthetics-canary-executionrolearn
        const policy = new iam.PolicyDocument({
            statements: [
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['s3:ListAllMyBuckets'],
                }),
                new iam.PolicyStatement({
                    resources: [this.artifactsBucket.arnForObjects(`${prefix ? prefix + '/*' : '*'}`)],
                    actions: ['s3:PutObject', 's3:GetBucketLocation'],
                }),
                new iam.PolicyStatement({
                    resources: ['*'],
                    actions: ['cloudwatch:PutMetricData'],
                    conditions: { StringEquals: { 'cloudwatch:namespace': 'CloudWatchSynthetics' } },
                }),
                new iam.PolicyStatement({
                    resources: [`arn:${partition}:logs:::*`],
                    actions: ['logs:CreateLogStream', 'logs:CreateLogGroup', 'logs:PutLogEvents'],
                }),
            ],
        });
        return new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            inlinePolicies: {
                canaryPolicy: policy,
            },
        });
    }
    /**
     * Returns the code object taken in by the canary resource.
     */
    createCode(props) {
        var _d, _e, _f;
        const codeConfig = {
            handler: props.test.handler,
            ...props.test.code.bind(this, props.test.handler),
        };
        return {
            handler: codeConfig.handler,
            script: codeConfig.inlineCode,
            s3Bucket: (_d = codeConfig.s3Location) === null || _d === void 0 ? void 0 : _d.bucketName,
            s3Key: (_e = codeConfig.s3Location) === null || _e === void 0 ? void 0 : _e.objectKey,
            s3ObjectVersion: (_f = codeConfig.s3Location) === null || _f === void 0 ? void 0 : _f.objectVersion,
        };
    }
    /**
     * Returns a canary schedule object
     */
    createSchedule(props) {
        var _d, _e, _f, _g;
        return {
            durationInSeconds: String(`${(_e = (_d = props.timeToLive) === null || _d === void 0 ? void 0 : _d.toSeconds()) !== null && _e !== void 0 ? _e : 0}`),
            expression: (_g = (_f = props.schedule) === null || _f === void 0 ? void 0 : _f.expressionString) !== null && _g !== void 0 ? _g : 'rate(5 minutes)',
        };
    }
    /**
     * Creates a unique name for the canary. The generated name is the physical ID of the canary.
     */
    generateUniqueName() {
        const name = cdk.Names.uniqueId(this).toLowerCase().replace(' ', '-');
        if (name.length <= 21) {
            return name;
        }
        else {
            return name.substring(0, 15) + nameHash(name);
        }
    }
    cannedMetric(fn, props) {
        return new aws_cloudwatch_1.Metric({
            ...fn({ CanaryName: this.canaryName }),
            ...props,
        }).attachTo(this);
    }
}
exports.Canary = Canary;
_c = JSII_RTTI_SYMBOL_1;
Canary[_c] = { fqn: "@aws-cdk/aws-synthetics.Canary", version: "1.95.0" };
/**
 * Take a hash of the given name.
 *
 * @param name the name to be hashed
 */
function nameHash(name) {
    const md5 = crypto.createHash('sha256').update(name).digest('hex');
    return md5.slice(0, 6);
}
const nameRegex = /^[0-9a-z_\-]+$/;
/**
 * Verifies that the name fits the regex expression: ^[0-9a-z_\-]+$.
 *
 * @param name - the given name of the canary
 */
function validateName(name) {
    if (name.length > 21) {
        throw new Error(`Canary name is too large, must be between 1 and 21 characters, but is ${name.length} (got "${name}")`);
    }
    if (!nameRegex.test(name)) {
        throw new Error(`Canary name must be lowercase, numbers, hyphens, or underscores (got "${name}")`);
    }
}
//# sourceMappingURL=data:application/json;base64,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