"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
        }
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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