"""Check if a host is in the Google Chrome HSTS Preload list"""

import functools
import os
import typing

__version__ = "2020.8.8"
__checksum__ = "3ebde2a97a5b3188c1fde14370fa62c425ad649a846d61a3dfc43e06807aaf23"
__all__ = ["in_hsts_preload"]

# fmt: off
_GTLD_INCLUDE_SUBDOMAINS = {b'android', b'app', b'bank', b'chrome', b'dev', b'foo', b'gle', b'gmail', b'google', b'hangout', b'insurance', b'meet', b'new', b'page', b'play', b'search', b'youtube'}  # noqa: E501
_JUMPTABLE = [[(0, 11), (11, 5), None, (16, 57), (73, 26), (99, 12), None, (111, 19), (130, 22), (152, 7), (159, 20), (179, 18), None, (197, 22), (219, 45), (264, 7), (271, 9), (280, 36), (316, 10), (326, 10), (336, 21), None, (357, 50), (407, 8), (415, 9), (424, 19), (443, 13), (456, 14), (470, 14), None, None, (484, 29), (513, 16), (529, 35), (564, 14), (578, 24), (602, 9), None, (611, 25), (636, 20), (656, 8), (664, 13), (677, 10), None, (687, 17), (704, 6), (710, 26), (736, 5), (741, 5), (746, 10), (756, 10), (766, 11), (777, 12), (789, 27), None, (816, 11), (827, 11), (838, 7), (845, 29), (874, 18), (892, 27), (919, 46), (965, 25), (990, 16), (1006, 8), (1014, 5), (1019, 22), (1041, 18), None, (1059, 36), (1095, 15), (1110, 8), (1118, 5), None, (1123, 5), (1128, 16), (1144, 14), (1158, 18), None, (1176, 14), (1190, 18), (1208, 48), (1256, 19), (1275, 5), (1280, 46), (1326, 14), (1340, 14), (1354, 20), None, (1374, 10), (1384, 13), (1397, 10), (1407, 19), None, (1426, 13), (1439, 19), (1458, 5), (1463, 4), (1467, 22), (1489, 10), (1499, 7), (1506, 14), (1520, 21), (1541, 11), (1552, 10), (1562, 12), (1574, 32), None, (1606, 10), (1616, 14), (1630, 12), (1642, 45), (1687, 15), None, (1702, 11), (1713, 23), (1736, 21), (1757, 26), (1783, 6), (1789, 6), (1795, 7), (1802, 5), (1807, 20), (1827, 23), (1850, 24), (1874, 13), (1887, 15), (1902, 19), (1921, 6), (1927, 61), (1988, 44), (2032, 12), (2044, 23), (2067, 16), (2083, 38), (2121, 6), (2127, 12), (2139, 44), (2183, 6), (2189, 41), (2230, 13), (2243, 23), (2266, 30), (2296, 16), (2312, 8), (2320, 15), (2335, 12), (2347, 19), (2366, 21), (2387, 15), None, (2402, 35), (2437, 21), (2458, 17), (2475, 19), (2494, 26), (2520, 5), (2525, 37), (2562, 30), (2592, 16), (2608, 10), (2618, 17), (2635, 23), (2658, 14), (2672, 17), (2689, 8), (2697, 4), (2701, 7), (2708, 29), (2737, 6), (2743, 18), (2761, 27), (2788, 20), (2808, 17), (2825, 19), (2844, 12), (2856, 40), (2896, 40), (2936, 12), (2948, 48), (2996, 25), (3021, 12), None, (3033, 8), (3041, 20), (3061, 19), (3080, 6), (3086, 23), None, (3109, 23), (3132, 33), (3165, 14), (3179, 12), (3191, 27), None, (3218, 26), (3244, 31), (3275, 50), (3325, 15), (3340, 20), (3360, 15), (3375, 21), (3396, 32), (3428, 24), (3452, 20), (3472, 13), (3485, 60), (3545, 19), (3564, 9), (3573, 12), (3585, 12), (3597, 11), (3608, 10), (3618, 48), (3666, 32), None, (3698, 25), (3723, 12), None, (3735, 8), (3743, 8), (3751, 7), None, (3758, 25), (3783, 17), None, (3800, 21), (3821, 35), (3856, 12), (3868, 10), (3878, 36), (3914, 20), (3934, 22), (3956, 23), (3979, 19), (3998, 12), (4010, 5), (4015, 30), (4045, 24), (4069, 14), (4083, 14), (4097, 47), (4144, 46), None, None, (4190, 51), (4241, 42), None, (4283, 14), None, (4297, 15), (4312, 8), (4320, 21), (4341, 6), (4347, 16), (4363, 17)], [(4380, 6151), (10531, 6552), (17083, 7025), (24108, 5927), (30035, 6251), (36286, 5971), (42257, 6895), (49152, 6207), (55359, 6743), (62102, 5969), (68071, 7034), (75105, 6331), (81436, 6724), (88160, 7010), (95170, 6467), (101637, 6443), (108080, 6983), (115063, 5941), (121004, 6233), (127237, 6491), (133728, 6839), (140567, 6501), (147068, 6774), (153842, 6084), (159926, 6231), (166157, 6588), (172745, 6484), (179229, 6786), (186015, 6233), (192248, 6540), (198788, 6824), (205612, 6455), (212067, 6384), (218451, 6976), (225427, 6076), (231503, 6760), (238263, 6159), (244422, 6941), (251363, 6683), (258046, 6919), (264965, 7421), (272386, 6350), (278736, 6270), (285006, 6113), (291119, 6319), (297438, 5959), (303397, 6253), (309650, 7005), (316655, 6264), (322919, 5674), (328593, 6359), (334952, 6544), (341496, 6466), (347962, 6682), (354644, 6787), (361431, 6730), (368161, 6759), (374920, 5875), (380795, 6794), (387589, 5774), (393363, 6596), (399959, 6319), (406278, 6312), (412590, 6778), (419368, 6643), (426011, 6504), (432515, 6111), (438626, 7039), (445665, 6671), (452336, 6785), (459121, 6473), (465594, 6385), (471979, 5683), (477662, 6916), (484578, 7016), (491594, 6916), (498510, 6112), (504622, 7148), (511770, 6952), (518722, 5971), (524693, 6697), (531390, 5689), (537079, 6320), (543399, 6512), (549911, 6436), (556347, 6402), (562749, 6502), (569251, 6651), (575902, 6743), (582645, 6409), (589054, 7112), (596166, 5974), (602140, 6156), (608296, 6470), (614766, 6492), (621258, 7089), (628347, 6810), (635157, 6357), (641514, 6209), (647723, 6075), (653798, 6191), (659989, 6687), (666676, 6103), (672779, 6395), (679174, 6092), (685266, 6785), (692051, 6614), (698665, 7008), (705673, 7969), (713642, 7054), (720696, 6823), (727519, 6453), (733972, 6252), (740224, 6614), (746838, 6845), (753683, 6473), (760156, 6217), (766373, 6349), (772722, 6304), (779026, 6865), (785891, 6700), (792591, 6803), (799394, 7009), (806403, 6791), (813194, 7685), (820879, 6337), (827216, 5716), (832932, 6895), (839827, 6562), (846389, 7891), (854280, 6916), (861196, 6099), (867295, 6722), (874017, 6803), (880820, 6290), (887110, 6660), (893770, 6125), (899895, 6713), (906608, 6397), (913005, 6476), (919481, 6509), (925990, 7153), (933143, 6188), (939331, 6216), (945547, 6549), (952096, 6552), (958648, 6383), (965031, 6775), (971806, 6117), (977923, 7125), (985048, 6592), (991640, 6657), (998297, 6823), (1005120, 6443), (1011563, 6540), (1018103, 6430), (1024533, 6273), (1030806, 6308), (1037114, 6152), (1043266, 5708), (1048974, 6099), (1055073, 6548), (1061621, 7147), (1068768, 6048), (1074816, 6592), (1081408, 6871), (1088279, 6370), (1094649, 6072), (1100721, 6787), (1107508, 6406), (1113914, 5902), (1119816, 6494), (1126310, 7618), (1133928, 5981), (1139909, 6174), (1146083, 6644), (1152727, 6151), (1158878, 6648), (1165526, 6316), (1171842, 5953), (1177795, 7358), (1185153, 6668), (1191821, 6370), (1198191, 6923), (1205114, 7281), (1212395, 7229), (1219624, 6139), (1225763, 6928), (1232691, 6263), (1238954, 6547), (1245501, 6687), (1252188, 6017), (1258205, 6777), (1264982, 6909), (1271891, 6505), (1278396, 6548), (1284944, 6357), (1291301, 6506), (1297807, 6747), (1304554, 6262), (1310816, 6569), (1317385, 5824), (1323209, 7092), (1330301, 6876), (1337177, 6568), (1343745, 6861), (1350606, 5653), (1356259, 6680), (1362939, 6386), (1369325, 6624), (1375949, 6760), (1382709, 7105), (1389814, 6503), (1396317, 6618), (1402935, 6726), (1409661, 6317), (1415978, 6461), (1422439, 6413), (1428852, 6467), (1435319, 6305), (1441624, 6425), (1448049, 5901), (1453950, 7479), (1461429, 6609), (1468038, 6264), (1474302, 6676), (1480978, 6593), (1487571, 5752), (1493323, 6639), (1499962, 6490), (1506452, 7561), (1514013, 6456), (1520469, 5916), (1526385, 7014), (1533399, 6329), (1539728, 7045), (1546773, 6042), (1552815, 6169), (1558984, 5825), (1564809, 6609), (1571418, 6492), (1577910, 6806), (1584716, 6171), (1590887, 6506), (1597393, 6482), (1603875, 6951), (1610826, 6299), (1617125, 5746), (1622871, 6450), (1629321, 6140), (1635461, 6611), (1642072, 6822), (1648894, 7069), (1655963, 6247), (1662210, 6244), (1668454, 6638)], [(1675092, 703), (1675795, 625), (1676420, 628), (1677048, 663), (1677711, 523), (1678234, 633), (1678867, 644), (1679511, 836), (1680347, 640), (1680987, 627), (1681614, 536), (1682150, 544), (1682694, 758), (1683452, 866), (1684318, 955), (1685273, 714), (1685987, 1207), (1687194, 606), (1687800, 875), (1688675, 673), (1689348, 733), (1690081, 708), (1690789, 802), (1691591, 731), (1692322, 684), (1693006, 631), (1693637, 940), (1694577, 1087), (1695664, 791), (1696455, 657), (1697112, 922), (1698034, 787), (1698821, 557), (1699378, 671), (1700049, 732), (1700781, 761), (1701542, 619), (1702161, 688), (1702849, 692), (1703541, 992), (1704533, 683), (1705216, 812), (1706028, 676), (1706704, 689), (1707393, 728), (1708121, 362), (1708483, 763), (1709246, 857), (1710103, 721), (1710824, 542), (1711366, 789), (1712155, 656), (1712811, 763), (1713574, 952), (1714526, 918), (1715444, 483), (1715927, 661), (1716588, 486), (1717074, 578), (1717652, 724), (1718376, 734), (1719110, 776), (1719886, 1008), (1720894, 861), (1721755, 628), (1722383, 692), (1723075, 767), (1723842, 444), (1724286, 561), (1724847, 518), (1725365, 692), (1726057, 821), (1726878, 523), (1727401, 725), (1728126, 634), (1728760, 671), (1729431, 552), (1729983, 696), (1730679, 768), (1731447, 428), (1731875, 672), (1732547, 629), (1733176, 828), (1734004, 623), (1734627, 607), (1735234, 346), (1735580, 597), (1736177, 725), (1736902, 756), (1737658, 676), (1738334, 816), (1739150, 1074), (1740224, 814), (1741038, 773), (1741811, 674), (1742485, 436), (1742921, 971), (1743892, 881), (1744773, 580), (1745353, 601), (1745954, 683), (1746637, 843), (1747480, 839), (1748319, 552), (1748871, 632), (1749503, 740), (1750243, 395), (1750638, 467), (1751105, 924), (1752029, 863), (1752892, 792), (1753684, 774), (1754458, 610), (1755068, 771), (1755839, 659), (1756498, 699), (1757197, 709), (1757906, 433), (1758339, 652), (1758991, 615), (1759606, 914), (1760520, 653), (1761173, 789), (1761962, 404), (1762366, 703), (1763069, 679), (1763748, 835), (1764583, 908), (1765491, 781), (1766272, 904), (1767176, 774), (1767950, 524), (1768474, 757), (1769231, 583), (1769814, 758), (1770572, 732), (1771304, 639), (1771943, 682), (1772625, 622), (1773247, 656), (1773903, 594), (1774497, 691), (1775188, 715), (1775903, 637), (1776540, 455), (1776995, 570), (1777565, 656), (1778221, 573), (1778794, 717), (1779511, 594), (1780105, 761), (1780866, 532), (1781398, 490), (1781888, 656), (1782544, 596), (1783140, 642), (1783782, 639), (1784421, 796), (1785217, 594), (1785811, 609), (1786420, 857), (1787277, 826), (1788103, 522), (1788625, 695), (1789320, 820), (1790140, 632), (1790772, 659), (1791431, 480), (1791911, 609), (1792520, 633), (1793153, 737), (1793890, 598), (1794488, 898), (1795386, 695), (1796081, 794), (1796875, 721), (1797596, 649), (1798245, 566), (1798811, 646), (1799457, 699), (1800156, 1277), (1801433, 514), (1801947, 624), (1802571, 608), (1803179, 962), (1804141, 754), (1804895, 745), (1805640, 546), (1806186, 565), (1806751, 808), (1807559, 553), (1808112, 569), (1808681, 847), (1809528, 650), (1810178, 871), (1811049, 791), (1811840, 668), (1812508, 686), (1813194, 822), (1814016, 620), (1814636, 889), (1815525, 631), (1816156, 762), (1816918, 558), (1817476, 742), (1818218, 449), (1818667, 775), (1819442, 779), (1820221, 648), (1820869, 916), (1821785, 775), (1822560, 790), (1823350, 903), (1824253, 1077), (1825330, 815), (1826145, 586), (1826731, 854), (1827585, 677), (1828262, 483), (1828745, 443), (1829188, 692), (1829880, 762), (1830642, 406), (1831048, 974), (1832022, 462), (1832484, 758), (1833242, 863), (1834105, 712), (1834817, 769), (1835586, 647), (1836233, 788), (1837021, 715), (1837736, 784), (1838520, 608), (1839128, 566), (1839694, 408), (1840102, 651), (1840753, 437), (1841190, 767), (1841957, 834), (1842791, 764), (1843555, 718), (1844273, 621), (1844894, 568), (1845462, 848), (1846310, 440), (1846750, 590), (1847340, 780), (1848120, 494), (1848614, 847), (1849461, 2086), (1851547, 548), (1852095, 627), (1852722, 875), (1853597, 851), (1854448, 510)], [(1854958, 48), None, (1855006, 35), (1855041, 42), None, None, None, None, None, None, None, None, None, None, None, None, None, (1855083, 42), None, (1855125, 25), (1855150, 16), None, (1855166, 18), None, None, None, None, (1855184, 26), None, None, None, None, (1855210, 21), (1855231, 25), None, None, (1855256, 26), None, None, None, None, (1855282, 44), (1855326, 21), (1855347, 23), None, None, None, None, (1855370, 48), None, None, None, None, None, (1855418, 31), None, None, None, None, (1855449, 42), None, (1855491, 22), None, (1855513, 21), None, (1855534, 26), (1855560, 42), None, None, (1855602, 77), None, None, None, None, None, (1855679, 21), (1855700, 21), None, None, (1855721, 34), (1855755, 42), None, None, None, (1855797, 25), None, None, (1855822, 21), None, None, None, None, None, (1855843, 24), (1855867, 21), None, None, (1855888, 26), None, (1855914, 18), None, (1855932, 54), None, None, None, None, None, None, (1855986, 26), None, (1856012, 19), None, (1856031, 20), None, None, (1856051, 42), (1856093, 42), (1856135, 17), None, (1856152, 26), None, (1856178, 26), None, None, None, (1856204, 26), (1856230, 20), (1856250, 26), None, (1856276, 42), (1856318, 63), None, None, None, (1856381, 40), (1856421, 48), None, None, None, (1856469, 47), None, None, None, None, None, None, None, (1856516, 42), None, (1856558, 55), None, (1856613, 9), None, (1856622, 21), (1856643, 42), None, None, (1856685, 42), (1856727, 82), None, None, (1856809, 42), None, None, None, None, None, None, None, None, None, (1856851, 42), (1856893, 21), (1856914, 21), None, (1856935, 42), (1856977, 25), None, None, (1857002, 21), (1857023, 42), None, None, (1857065, 21), (1857086, 19), (1857105, 26), None, None, None, (1857131, 21), None, None, (1857152, 38), None, (1857190, 22), (1857212, 21), (1857233, 21), None, None, (1857254, 63), None, (1857317, 21), (1857338, 42), None, (1857380, 17), None, None, None, None, (1857397, 21), (1857418, 21), None, None, (1857439, 21), None, None, (1857460, 21), None, (1857481, 26), None, (1857507, 50), None, None, None, (1857557, 50), (1857607, 26), (1857633, 21), (1857654, 21), (1857675, 19), None, (1857694, 35), (1857729, 26), (1857755, 23), (1857778, 21), (1857799, 42), None, None, None, None, None, None, (1857841, 21), None, None, None, (1857862, 21), None, None, (1857883, 90), None, (1857973, 239), (1858212, 38), None, None, None, None]]  # noqa: E501
_CRC8_TABLE = [
    0x00, 0x07, 0x0e, 0x09, 0x1c, 0x1b, 0x12, 0x15,
    0x38, 0x3f, 0x36, 0x31, 0x24, 0x23, 0x2a, 0x2d,
    0x70, 0x77, 0x7e, 0x79, 0x6c, 0x6b, 0x62, 0x65,
    0x48, 0x4f, 0x46, 0x41, 0x54, 0x53, 0x5a, 0x5d,
    0xe0, 0xe7, 0xee, 0xe9, 0xfc, 0xfb, 0xf2, 0xf5,
    0xd8, 0xdf, 0xd6, 0xd1, 0xc4, 0xc3, 0xca, 0xcd,
    0x90, 0x97, 0x9e, 0x99, 0x8c, 0x8b, 0x82, 0x85,
    0xa8, 0xaf, 0xa6, 0xa1, 0xb4, 0xb3, 0xba, 0xbd,
    0xc7, 0xc0, 0xc9, 0xce, 0xdb, 0xdc, 0xd5, 0xd2,
    0xff, 0xf8, 0xf1, 0xf6, 0xe3, 0xe4, 0xed, 0xea,
    0xb7, 0xb0, 0xb9, 0xbe, 0xab, 0xac, 0xa5, 0xa2,
    0x8f, 0x88, 0x81, 0x86, 0x93, 0x94, 0x9d, 0x9a,
    0x27, 0x20, 0x29, 0x2e, 0x3b, 0x3c, 0x35, 0x32,
    0x1f, 0x18, 0x11, 0x16, 0x03, 0x04, 0x0d, 0x0a,
    0x57, 0x50, 0x59, 0x5e, 0x4b, 0x4c, 0x45, 0x42,
    0x6f, 0x68, 0x61, 0x66, 0x73, 0x74, 0x7d, 0x7a,
    0x89, 0x8e, 0x87, 0x80, 0x95, 0x92, 0x9b, 0x9c,
    0xb1, 0xb6, 0xbf, 0xb8, 0xad, 0xaa, 0xa3, 0xa4,
    0xf9, 0xfe, 0xf7, 0xf0, 0xe5, 0xe2, 0xeb, 0xec,
    0xc1, 0xc6, 0xcf, 0xc8, 0xdd, 0xda, 0xd3, 0xd4,
    0x69, 0x6e, 0x67, 0x60, 0x75, 0x72, 0x7b, 0x7c,
    0x51, 0x56, 0x5f, 0x58, 0x4d, 0x4a, 0x43, 0x44,
    0x19, 0x1e, 0x17, 0x10, 0x05, 0x02, 0x0b, 0x0c,
    0x21, 0x26, 0x2f, 0x28, 0x3d, 0x3a, 0x33, 0x34,
    0x4e, 0x49, 0x40, 0x47, 0x52, 0x55, 0x5c, 0x5b,
    0x76, 0x71, 0x78, 0x7f, 0x6a, 0x6d, 0x64, 0x63,
    0x3e, 0x39, 0x30, 0x37, 0x22, 0x25, 0x2c, 0x2b,
    0x06, 0x01, 0x08, 0x0f, 0x1a, 0x1d, 0x14, 0x13,
    0xae, 0xa9, 0xa0, 0xa7, 0xb2, 0xb5, 0xbc, 0xbb,
    0x96, 0x91, 0x98, 0x9f, 0x8a, 0x8d, 0x84, 0x83,
    0xde, 0xd9, 0xd0, 0xd7, 0xc2, 0xc5, 0xcc, 0xcb,
    0xe6, 0xe1, 0xe8, 0xef, 0xfa, 0xfd, 0xf4, 0xf3
]
# fmt: on

_IS_LEAF = 0x80
_INCLUDE_SUBDOMAINS = 0x40


try:
    from importlib.resources import open_binary

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open_binary("hstspreload", path)


except ImportError:

    def open_pkg_binary(path: str) -> typing.BinaryIO:
        return open(
            os.path.join(os.path.dirname(os.path.abspath(__file__)), path), "rb",
        )


@functools.lru_cache(maxsize=1024)
def in_hsts_preload(host: typing.AnyStr) -> bool:
    """Determines if an IDNA-encoded host is on the HSTS preload list"""

    if isinstance(host, str):
        host = host.encode("ascii")
    labels = host.lower().split(b".")

    # Fast-branch for gTLDs that are registered to preload all sub-domains.
    if labels[-1] in _GTLD_INCLUDE_SUBDOMAINS:
        return True

    with open_pkg_binary("hstspreload.bin") as f:
        for layer, label in enumerate(labels[::-1]):
            # None of our layers are greater than 4 deep.
            if layer > 3:
                return False

            # Read the jump table for the layer and label
            jump_info = _JUMPTABLE[layer][_crc8(label)]
            if jump_info is None:
                # No entry: host is not preloaded
                return False

            # Read the set of entries for that layer and label
            f.seek(jump_info[0])
            data = bytearray(jump_info[1])
            f.readinto(data)

            for is_leaf, include_subdomains, ent_label in _iter_entries(data):
                # We found a potential leaf
                if is_leaf:
                    if ent_label == host:
                        return True
                    if include_subdomains and host.endswith(b"." + ent_label):
                        return True

                # Continue traversing as we're not at a leaf.
                elif label == ent_label:
                    break
            else:
                return False
    return False


def _iter_entries(data: bytes) -> typing.Iterable[typing.Tuple[int, int, bytes]]:
    while data:
        flags = data[0]
        size = data[1]
        label = bytes(data[2 : 2 + size])
        yield (flags & _IS_LEAF, flags & _INCLUDE_SUBDOMAINS, label)
        data = data[2 + size :]


def _crc8(value: bytes) -> int:
    # CRC8 reference implementation: https://github.com/niccokunzmann/crc8
    checksum = 0x00
    for byte in value:
        checksum = _CRC8_TABLE[checksum ^ byte]
    return checksum
