"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Rule = void 0;
const core_1 = require("@aws-cdk/core");
const constructs_1 = require("constructs");
const events_generated_1 = require("./events.generated");
const schedule_1 = require("./schedule");
const util_1 = require("./util");
/**
 * Defines an EventBridge Rule in this stack.
 *
 * @stability stable
 * @resource AWS::Events::Rule
 */
class Rule extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.ruleName,
        });
        this.targets = new Array();
        this.eventPattern = {};
        this.accountEventBusTargets = {};
        if (props.eventBus && props.schedule) {
            throw new Error('Cannot associate rule with \'eventBus\' when using \'schedule\'');
        }
        this.description = props.description;
        this.scheduleExpression = props.schedule && props.schedule.expressionString;
        const resource = new events_generated_1.CfnRule(this, 'Resource', {
            name: this.physicalName,
            description: this.description,
            state: props.enabled == null ? 'ENABLED' : (props.enabled ? 'ENABLED' : 'DISABLED'),
            scheduleExpression: this.scheduleExpression,
            eventPattern: core_1.Lazy.any({ produce: () => this._renderEventPattern() }),
            targets: core_1.Lazy.any({ produce: () => this.renderTargets() }),
            eventBusName: props.eventBus && props.eventBus.eventBusName,
        });
        this.ruleArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'events',
            resource: 'rule',
            resourceName: this.physicalName,
        });
        this.ruleName = this.getResourceNameAttribute(resource.ref);
        this.addEventPattern(props.eventPattern);
        for (const target of props.targets || []) {
            this.addTarget(target);
        }
    }
    /**
     * Import an existing EventBridge Rule provided an ARN.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param eventRuleArn Event Rule ARN (i.e. arn:aws:events:<region>:<account-id>:rule/MyScheduledRule).
     * @stability stable
     */
    static fromEventRuleArn(scope, id, eventRuleArn) {
        const parts = core_1.Stack.of(scope).parseArn(eventRuleArn);
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.ruleArn = eventRuleArn;
                this.ruleName = parts.resourceName || '';
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a target to the rule. The abstract class RuleTarget can be extended to define new targets.
     *
     * No-op if target is undefined.
     *
     * @stability stable
     */
    addTarget(target) {
        var _a;
        if (!target) {
            return;
        }
        // Simply increment id for each `addTarget` call. This is guaranteed to be unique.
        const autoGeneratedId = `Target${this.targets.length}`;
        const targetProps = target.bind(this, autoGeneratedId);
        const inputProps = targetProps.input && targetProps.input.bind(this);
        const roleArn = (_a = targetProps.role) === null || _a === void 0 ? void 0 : _a.roleArn;
        const id = targetProps.id || autoGeneratedId;
        if (targetProps.targetResource) {
            const targetStack = core_1.Stack.of(targetProps.targetResource);
            const targetAccount = targetStack.account;
            const targetRegion = targetStack.region;
            const sourceStack = core_1.Stack.of(this);
            const sourceAccount = sourceStack.account;
            const sourceRegion = sourceStack.region;
            if (targetRegion !== sourceRegion) {
                throw new Error('Rule and target must be in the same region');
            }
            if (targetAccount !== sourceAccount) {
                // cross-account event - strap in, this works differently than regular events!
                // based on:
                // https://docs.aws.amazon.com/eventbridge/latest/userguide/eventbridge-cross-account-event-delivery.html
                // for cross-account events, we require concrete accounts
                if (core_1.Token.isUnresolved(targetAccount)) {
                    throw new Error('You need to provide a concrete account for the target stack when using cross-account events');
                }
                if (core_1.Token.isUnresolved(sourceAccount)) {
                    throw new Error('You need to provide a concrete account for the source stack when using cross-account events');
                }
                // and the target region has to be concrete as well
                if (core_1.Token.isUnresolved(targetRegion)) {
                    throw new Error('You need to provide a concrete region for the target stack when using cross-account events');
                }
                // the _actual_ target is just the event bus of the target's account
                // make sure we only add it once per account
                const exists = this.accountEventBusTargets[targetAccount];
                if (!exists) {
                    this.accountEventBusTargets[targetAccount] = true;
                    this.targets.push({
                        id,
                        arn: targetStack.formatArn({
                            service: 'events',
                            resource: 'event-bus',
                            resourceName: 'default',
                            region: targetRegion,
                            account: targetAccount,
                        }),
                    });
                }
                // Grant the source account permissions to publish events to the event bus of the target account.
                // Do it in a separate stack instead of the target stack (which seems like the obvious place to put it),
                // because it needs to be deployed before the rule containing the above event-bus target in the source stack
                // (EventBridge verifies whether you have permissions to the targets on rule creation),
                // but it's common for the target stack to depend on the source stack
                // (that's the case with CodePipeline, for example)
                const sourceApp = this.node.root;
                if (!sourceApp || !core_1.App.isApp(sourceApp)) {
                    throw new Error('Event stack which uses cross-account targets must be part of a CDK app');
                }
                const targetApp = constructs_1.Node.of(targetProps.targetResource).root;
                if (!targetApp || !core_1.App.isApp(targetApp)) {
                    throw new Error('Target stack which uses cross-account event targets must be part of a CDK app');
                }
                if (sourceApp !== targetApp) {
                    throw new Error('Event stack and target stack must belong to the same CDK app');
                }
                const stackId = `EventBusPolicy-${sourceAccount}-${targetRegion}-${targetAccount}`;
                let eventBusPolicyStack = sourceApp.node.tryFindChild(stackId);
                if (!eventBusPolicyStack) {
                    eventBusPolicyStack = new core_1.Stack(sourceApp, stackId, {
                        env: {
                            account: targetAccount,
                            region: targetRegion,
                        },
                        stackName: `${targetStack.stackName}-EventBusPolicy-support-${targetRegion}-${sourceAccount}`,
                    });
                    new events_generated_1.CfnEventBusPolicy(eventBusPolicyStack, 'GivePermToOtherAccount', {
                        action: 'events:PutEvents',
                        statementId: `Allow-account-${sourceAccount}`,
                        principal: sourceAccount,
                    });
                }
                // deploy the event bus permissions before the source stack
                sourceStack.addDependency(eventBusPolicyStack);
                // The actual rule lives in the target stack.
                // Other than the account, it's identical to this one
                // eventPattern is mutable through addEventPattern(), so we need to lazy evaluate it
                // but only Tokens can be lazy in the framework, so make a subclass instead
                const self = this;
                class CopyRule extends Rule {
                    _renderEventPattern() {
                        return self._renderEventPattern();
                    }
                    // we need to override validate(), as it uses the
                    // value of the eventPattern field,
                    // which might be empty in the case of the copied rule
                    // (as the patterns in the original might be added through addEventPattern(),
                    // not passed through the constructor).
                    // Anyway, even if the original rule is invalid,
                    // we would get duplicate errors if we didn't override this,
                    // which is probably a bad idea in and of itself
                    validate() {
                        return [];
                    }
                }
                new CopyRule(targetStack, `${core_1.Names.uniqueId(this)}-${id}`, {
                    targets: [target],
                    eventPattern: this.eventPattern,
                    schedule: this.scheduleExpression ? schedule_1.Schedule.expression(this.scheduleExpression) : undefined,
                    description: this.description,
                });
                return;
            }
        }
        this.targets.push({
            id,
            arn: targetProps.arn,
            roleArn,
            ecsParameters: targetProps.ecsParameters,
            kinesisParameters: targetProps.kinesisParameters,
            runCommandParameters: targetProps.runCommandParameters,
            batchParameters: targetProps.batchParameters,
            sqsParameters: targetProps.sqsParameters,
            input: inputProps && inputProps.input,
            inputPath: inputProps && inputProps.inputPath,
            inputTransformer: (inputProps === null || inputProps === void 0 ? void 0 : inputProps.inputTemplate) !== undefined ? {
                inputTemplate: inputProps.inputTemplate,
                inputPathsMap: inputProps.inputPathsMap,
            } : undefined,
        });
    }
    /**
     * Adds an event pattern filter to this rule.
     *
     * If a pattern was already specified,
     * these values are merged into the existing pattern.
     *
     * For example, if the rule already contains the pattern:
     *
     *     {
     *       "resources": [ "r1" ],
     *       "detail": {
     *         "hello": [ 1 ]
     *       }
     *     }
     *
     * And `addEventPattern` is called with the pattern:
     *
     *     {
     *       "resources": [ "r2" ],
     *       "detail": {
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * The resulting event pattern will be:
     *
     *     {
     *       "resources": [ "r1", "r2" ],
     *       "detail": {
     *         "hello": [ 1 ],
     *         "foo": [ "bar" ]
     *       }
     *     }
     *
     * @stability stable
     */
    addEventPattern(eventPattern) {
        if (!eventPattern) {
            return;
        }
        util_1.mergeEventPattern(this.eventPattern, eventPattern);
    }
    /**
     * Not private only to be overrideen in CopyRule.
     *
     * @internal
     */
    _renderEventPattern() {
        const eventPattern = this.eventPattern;
        if (Object.keys(eventPattern).length === 0) {
            return undefined;
        }
        // rename 'detailType' to 'detail-type'
        const out = {};
        for (let key of Object.keys(eventPattern)) {
            const value = eventPattern[key];
            if (key === 'detailType') {
                key = 'detail-type';
            }
            out[key] = value;
        }
        return out;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    validate() {
        if (Object.keys(this.eventPattern).length === 0 && !this.scheduleExpression) {
            return ['Either \'eventPattern\' or \'schedule\' must be defined'];
        }
        return [];
    }
    renderTargets() {
        if (this.targets.length === 0) {
            return undefined;
        }
        return this.targets;
    }
}
exports.Rule = Rule;
//# sourceMappingURL=data:application/json;base64,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