"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.EsbuildBundler = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const utils_1 = require("./private/utils");
const provider_1 = require("./provider");
/**
 * Low-level construct that can be used where `BundlingOptions` are required.
 * This class directly interfaces with esbuild and provides almost no configuration safeguards.
 *
 * @stability experimental
 */
class EsbuildBundler {
    /**
     * @stability experimental
     */
    constructor(
    /**
     * A path or list or map of paths to the entry points of your code.
     *
     * Relative paths are by default resolved from the current working directory.
     * To change the working directory, see `buildOptions.absWorkingDir`.
     *
     * Absolute paths can be used if files are part of the working directory.
     *
     * Examples:
     *  - `'src/index.ts'`
     *  - `require.resolve('./lambda')`
     *  - `['src/index.ts', 'src/util.ts']`
     *  - `{one: 'src/two.ts', two: 'src/one.ts'}`
     *
     * @stability experimental
     */
    entryPoints, 
    /**
     * Props to change the behavior of the bundler.
     *
     * @stability experimental
     */
    props) {
        this.entryPoints = entryPoints;
        this.props = props;
        /**
         * @deprecated This value is ignored since the bundler is always using a locally installed version of esbuild. However the property is required to comply with the `BundlingOptions` interface.
         *
         * @stability deprecated
         */
        this.image = aws_cdk_lib_1.DockerImage.fromRegistry('scratch');
        if (props?.buildOptions?.outfile && props?.buildOptions?.outdir) {
            throw new Error('Cannot use both "outfile" and "outdir"');
        }
        this.local = {
            tryBundle: (outputDir, _options) => {
                if (this.props.copyDir) {
                    const copyDir = this.getCopyDirList(this.props.copyDir);
                    copyDir.forEach(([dest, src]) => {
                        const srcDir = path_1.resolve(this.props?.buildOptions?.absWorkingDir ?? process.cwd(), src);
                        const destDir = path_1.resolve(outputDir, dest);
                        const destToOutput = path_1.relative(outputDir, destDir);
                        if (destToOutput.startsWith('..') || path_1.isAbsolute(destToOutput)) {
                            throw new Error('Cannot copy files to outside of the asset staging directory. See docs for details.');
                        }
                        if (fs_1.existsSync(destDir)) {
                            fs_1.rmSync(destDir, { recursive: true, force: true });
                        }
                        fs_1.mkdirSync(destDir, { recursive: true });
                        aws_cdk_lib_1.FileSystem.copyDirectory(srcDir, destDir);
                    });
                }
                try {
                    const provider = props.buildProvider ?? provider_1.EsbuildProvider.defaultBuildProvider();
                    provider.buildSync({
                        entryPoints: typeof entryPoints === 'string' ? [entryPoints] : entryPoints,
                        color: process.env.NO_COLOR ? Boolean(process.env.NO_COLOR) : undefined,
                        ...(this.props?.buildOptions || {}),
                        ...this.getOutputOptions(outputDir, { normalize: path_1.normalize, join: path_1.join }),
                    });
                }
                catch (error) {
                    if (utils_1.isEsbuildError(error)) {
                        throw new Error(`Esbuild failed to bundle ${entryPoints}`);
                    }
                    throw error;
                }
                return true;
            },
        };
    }
    getCopyDirList(copyDir) {
        // Nothing to copy
        if (!copyDir) {
            return [];
        }
        // List of strings
        if (Array.isArray(copyDir)) {
            return copyDir.map((src) => ['.', src]);
        }
        // A map
        if (typeof copyDir === 'object' &&
            !Array.isArray(copyDir) &&
            copyDir !== null) {
            return Object
                .entries(copyDir)
                .flatMap(([dest, sources]) => {
                if (Array.isArray(sources)) {
                    return sources.map((src) => [dest, src]);
                }
                return [[dest, sources]];
            });
        }
        // A single string
        return [['.', copyDir]];
    }
    getOutputOptions(cdkOutputDir, path = path_1.posix) {
        if (this.props?.buildOptions?.outfile) {
            return {
                outdir: undefined,
                outfile: path.normalize(path.join(...[cdkOutputDir, this.props?.buildOptions?.outfile].filter(Boolean))),
            };
        }
        return {
            outdir: path.normalize(path.join(...[cdkOutputDir, this.props?.buildOptions?.outdir].filter(Boolean))),
            outfile: undefined,
        };
    }
}
exports.EsbuildBundler = EsbuildBundler;
_a = JSII_RTTI_SYMBOL_1;
EsbuildBundler[_a] = { fqn: "@mrgrain/cdk-esbuild.EsbuildBundler", version: "4.1.0" };
//# sourceMappingURL=data:application/json;base64,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