"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DynamicPackage = void 0;
const child_process_1 = require("child_process");
const fs_1 = require("fs");
const os_1 = require("os");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
class DynamicPackage {
    constructor(
    /**
     * Name of the npm package
     * Version to install, or version constraint
     *
     * @default - no version constraint, install the latest version
     */
    packageSpec, props = {}) {
        const { name, version } = this.parsePackageSpec(packageSpec);
        this.name = name;
        this.version = version;
        this.installPath =
            props.installPath ||
                fs_1.mkdtempSync(path_1.join(os_1.tmpdir(), `cdk-dynamic-${this.spec}-`));
        this.searchPaths = props.searchPaths || [];
    }
    get spec() {
        if (!this.version) {
            return this.name;
        }
        return `${this.name}@${this.version}`;
    }
    tryResolve(paths) {
        try {
            return require.resolve(this.name, paths ? { paths } : undefined);
        }
        catch (_) {
            return;
        }
    }
    auto() {
        return this.tryResolve() || this.findInPaths() || this.install();
    }
    nodeJs() {
        return this.name;
    }
    findIn(paths) {
        paths.forEach((p) => process.stderr.write('trying... ' + p + '\n'));
        process.stderr.write('\n');
        return this.tryResolve([...paths].filter(Boolean));
    }
    findInPaths() {
        return (this.findInSearchPaths() ||
            this.findInLocalPaths() ||
            this.findInGlobalPaths());
    }
    findInSearchPaths() {
        return this.findIn(this.searchPaths);
    }
    findInLocalPaths() {
        return this.findIn([
            process.cwd(),
            process.env.PWD,
            path_1.resolve(process.env.PWD || process.cwd(), 'node_modules'),
            path_1.resolve(process.cwd(), 'node_modules'),
        ].filter(Boolean));
    }
    findInGlobalPaths() {
        return this.findIn([
            process.execPath,
            path_1.resolve(process.execPath, '../..'),
            path_1.resolve(process.execPath, '../../lib'),
            path_1.resolve(process.execPath, '../../node_modules'),
            path_1.resolve(process.execPath, '../../lib/node_modules'),
        ]);
    }
    install() {
        return aws_cdk_lib_1.Lazy.string({
            produce: () => {
                if (!DynamicPackage.installedPackagePath.has(this.spec)) {
                    const args = [
                        'install',
                        this.spec,
                        '--no-save',
                        '--prefix',
                        this.installPath,
                    ];
                    DynamicPackage.log(`Dynamically installing ${this.spec} into "${this.installPath}"...`, 'info');
                    child_process_1.execFileSync('npm', args);
                    DynamicPackage.installedPackagePath.set(this.spec, require.resolve(this.name, {
                        paths: [this.installPath],
                    }));
                }
                return DynamicPackage.installedPackagePath.get(this.spec);
            },
        });
    }
    static log(message, _level = 'info') {
        process.stderr.write(`⬥ ${message}\n`);
    }
    parsePackageSpec(spec) {
        const hasScope = spec.startsWith('@');
        if (hasScope) {
            spec = spec.substring(1);
        }
        const [module, ...version] = spec.split('@');
        const name = hasScope ? `@${module}` : module;
        if (version.length == 0) {
            return { name };
        }
        return { name, version: version?.join('@') };
    }
}
exports.DynamicPackage = DynamicPackage;
DynamicPackage.installedPackagePath = new Map();
//# sourceMappingURL=data:application/json;base64,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