import { BuildOptions, TransformOptions } from './esbuild-types';
export { EsbuildSource } from './private/esbuild-source';
export interface ProviderBuildOptions extends BuildOptions {
    /** Documentation: https://esbuild.github.io/api/#entry-points */
    readonly entryPoints?: string[] | Record<string, string>;
}
export interface ProviderTransformOptions extends TransformOptions {
}
/**
 * Provides an implementation of the esbuild Build API
 */
export interface IBuildProvider {
    /**
     * A method implementing the code build.
     *
     * During synth time, the method will receive all computed `BuildOptions` from the bundler.
     *
     * It MUST implement any output options to integrate correctly and MAY use any other options.
     * On failure, it SHOULD print any warnings & errors to stderr and throw a `BuildFailure` to inform the bundler.
     *
     * @throws `esbuild.BuildFailure`
     */
    buildSync(options: ProviderBuildOptions): void;
}
/**
 * Provides an implementation of the esbuild Transform API
 */
export interface ITransformProvider {
    /**
     * A method implementing the inline code transformation.
     *
     * During synth time, the method will receive the inline code and all computed `TransformOptions` from the bundler.
     *
     * MUST return the transformed code as a string to integrate correctly.
     * It MAY use these options to do so.
     * On failure, it SHOULD print any warnings & errors to stderr and throw a `TransformFailure` to inform the bundler.
     *
     * @throws `esbuild.TransformFailure`
     */
    transformSync(input: string, options?: ProviderTransformOptions): string;
}
/**
 * Provides an implementation of the esbuild Build & Transform API
 */
export interface IEsbuildProvider extends IBuildProvider, ITransformProvider {
}
/**
 * Configure the default EsbuildProvider
 */
export interface EsbuildProviderProps {
    /**
     * Path to the binary used by esbuild.
     *
     * This is the same as setting the ESBUILD_BINARY_PATH environment variable.
     *
     * @stability stable
     */
    readonly esbuildBinaryPath?: string;
    /**
     * Absolute path to the esbuild module JS file.
     *
     * E.g. "/home/user/.npm/node_modules/esbuild/lib/main.js"
     *
     * If not set, the module path will be determined in the following order:
     *
     * - Use a path from the `CDK_ESBUILD_MODULE_PATH` environment variable
     * - In TypeScript, fallback to the default Node.js package resolution mechanism
     * - All other languages (Python, Go, .NET, Java) use an automatic "best effort" resolution mechanism. \
     *   The exact algorithm of this mechanism is considered an implementation detail and should not be relied on.
     *   If `esbuild` cannot be found, it might be installed dynamically to a temporary location.
     *   To opt-out of this behavior, set either `esbuildModulePath` or `CDK_ESBUILD_MODULE_PATH` env variable.
     *
     * Use the static methods on `EsbuildSource` to customize the default behavior.
     *
     * @stability stable
     * @default - `CDK_ESBUILD_MODULE_PATH` or package resolution (see description)
     */
    readonly esbuildModulePath?: string;
}
/**
 * Default esbuild implementation calling esbuild's JavaScript API.
 */
export declare class EsbuildProvider implements IBuildProvider, ITransformProvider {
    private static _fallbackProvider;
    private static _buildProvider;
    private static _transformationProvider;
    /**
     * Set the default implementation for both Build and Transformation API
     */
    static overrideDefaultProvider(provider: IEsbuildProvider): void;
    /**
     * Set the default implementation for the Build API
     */
    static overrideDefaultBuildProvider(provider: IBuildProvider): void;
    /**
     * Get the default implementation for the Build API
     */
    static defaultBuildProvider(): IBuildProvider;
    /**
     * Set the default implementation for the Transformation API
     */
    static overrideDefaultTransformationProvider(provider: ITransformProvider): void;
    /**
     * Get the default implementation for the Transformation API
     */
    static defaultTransformationProvider(): ITransformProvider;
    private readonly esbuildBinaryPath?;
    private readonly esbuildModulePath?;
    constructor(props?: EsbuildProviderProps);
    buildSync(options: ProviderBuildOptions): void;
    transformSync(input: string, options?: ProviderTransformOptions): string;
    /**
     * Invoke a function with a specific `process.env.ESBUILD_BINARY_PATH`
     * and restore the env var afterwards.
     */
    private withEsbuildBinaryPath;
    /**
     * Load the esbuild module according to defined rules.
     */
    private require;
    /**
     * Wrapper for require
     */
    private _require;
    /**
     * Resolve a token without context
     */
    private resolve;
}
