import configparser
import shutil
from abc import ABCMeta, abstractmethod
from pathlib import Path
from typing import Optional

from ..fs import home, link, local, makedirs, transplant_path
from ..utils import Some, run, unsome, with_os
from .manual_version import ManualVersion


def icon_name(app_path: Path) -> Optional[str]:
    app = configparser.ConfigParser()
    app.read(app_path)
    return app["Desktop Entry"].get("Icon")


class ManualPackage(ManualVersion, metaclass=ABCMeta):
    binaries: list[str]
    apps: list[str]
    fonts: list[str]

    def __init__(
        self,
        *,
        as_global: bool = False,
        binary: Some[str] = None,
        binary_wrapper: bool = False,
        app: Some[str] = None,
        font: Some[str] = None,
        **kwargs,
    ) -> None:
        self.as_global = as_global
        self.binaries = unsome(binary)
        self.binary_wrapper = binary_wrapper
        self.apps = unsome(app)
        self.fonts = unsome(font)
        super().__init__(**kwargs)

    @property
    def local(self) -> Path:
        if self.as_global:
            return Path("/usr/local")
        else:
            return local()

    @abstractmethod
    def binary_path(self, binary: str) -> Path:
        pass

    def install_binary(self, name: str) -> None:
        link(
            self.binary_path(name),
            self.local / "bin" / name,
            sudo=self.as_global,
            method="binary_wrapper" if self.binary_wrapper else None,
        )

    @abstractmethod
    def app_path(self, name: str) -> Path:
        pass

    def icon_directory(self) -> Optional[Path]:
        return None

    def install_app(self, name: str) -> None:
        with_os(linux=self.install_app_linux, macos=self.install_app_macos)(name)

    def install_app_linux(self, name: str) -> None:
        path = self.app_path(name)
        target = self.local / "share" / "applications" / f"{name}.desktop"
        link(path, target, sudo=self.as_global)
        icons_source = self.icon_directory()  # pylint:disable=assignment-from-none
        if icons_source:
            icons_target = self.local / "share" / "icons"
            icon = icon_name(path)
            if icon:
                for icon_path in icons_source.rglob(f"{icon}.*"):
                    target = transplant_path(icons_source, icons_target, icon_path)
                    link(icon_path, target, sudo=self.as_global)

    def install_app_macos(self, name: str) -> None:
        # FIXME: copy to /Applications and/or ~/Applications; ensure names,
        # etc. are correct
        pass

    @abstractmethod
    def font_path(self, name: str) -> Path:
        pass

    def install_font(self, name: str) -> None:
        font_dir = with_os(
            linux=self.local / "share" / "fonts", macos=home() / "Library" / "Fonts"
        )
        makedirs(font_dir)
        source = self.font_path(name)
        target = font_dir / name
        link(source, target, sudo=self.as_global)
        if shutil.which("fc-cache"):
            run("fc-cache", "-f", str(font_dir))

    def install(self) -> None:
        for binary in self.binaries:
            self.install_binary(binary)
        for app in self.apps:
            self.install_app(app)
        for font in self.fonts:
            self.install_font(font)
        super().install()
