"""tests for file_lib"""
import glob
import os
import logging
import unittest

from pathlib import Path

from amilib.file_lib import FileLib
from amilib.util import Util

from test.resources import Resources
from test.test_all import AmiAnyTest

FILE_LIB = "file_lib"
PDF_LIB = "pdf_lib"
TEXT_LIB = "text_lib"
_SETUP = "_setup"
_TEARDOWN = "_teardown"
TEST = "test"

"""
a few tests on globbing, etc"""

logger = Util.get_logger(__name__)

class File0Test(AmiAnyTest):
    # nothing much here
    logger = logging.getLogger("test_file")
    TEST = "test"

    OPTIONS = [
        FILE_LIB,
        PDF_LIB,
        TEXT_LIB,
        _SETUP,
        _TEARDOWN,
    ]

    @classmethod
    def example_setup(cls, pyamix):
        """ setup test or examples
        these are general commands (i.e. not subcommands)
        :pyamix:
        """
        pyamix.run_commands([
            "--delete ${exam_temp}",
            "--copy ${examples_test.p} ${exam_temp} overwrite",
        ])

    @classmethod
    def example_teardown(cls, pyamix):
        """ clean example files

        """
        pyamix.run_command([
            "--delete", "${exam_temp}",
        ])

    @classmethod
    def run_arg_tests(cls, args):
        """This needs revision , maybe using Examples()"""
        cls.logger.warning(f"*****running tests : {args[TEST]}")
        if not args[TEST]:
            cls.logger.warning(f"No tests given: choose some/all of {TEST}")
            return
        if FILE_LIB in args[TEST]:
            cls.logger.warning("run test_file")
            cls.test_file.main()
        if PDF_LIB in args[TEST]:
            cls.logger.warning("run test_pdf")
            cls.test_pdf.test_read_pdf()
        if TEXT_LIB in args[TEST]:
            cls.logger.warning("run test_text NYI")


    @unittest.skip("if not windows")
    def test_glob_with_windows_FILE(self):
        """
        some instances of glob.glob with fully forward slashes generate backslash or double backslash on windows
        """
        input_dir = Path(Resources.TEST_RESOURCES_DIR, "ar6", "cleaned_content")
        assert input_dir.exists()
        glob_str = f"{str(input_dir)}/**/html_with_ids.html"
        files = FileLib.posix_glob(glob_str, recursive=True)
        assert len(files) > 70
        for file in files[:5]:
            logger.debug(f"file: {file}")
            assert not "\\" in file # backslash is not generated by UNIX/MacOS
            assert "/" in file

    def test_list_children(self):
        """
        test that children can be listed as dir/not_dir. Ignore hidden files as these
        are not very standard
        """
        top_dir = self.make_self_test_dir()
        if Path(top_dir).exists():
            FileLib.delete_directory_contents(top_dir, delete_directory=True)
        assert not top_dir.exists()
        top_dir.mkdir()
        assert top_dir.exists()
        dir1 = Path(top_dir, "dir1")
        dir1.mkdir()
        assert dir1.exists()
        file1 = Path(top_dir, "file1")
        file1.touch()
        assert file1.exists()
        files = [str(f) for f in FileLib.get_children(top_dir)]
        assert sorted(files) == ['/Users/pm286/workspace/amilib/temp/test_files/dir1',
                                 '/Users/pm286/workspace/amilib/temp/test_files/file1']
        files = [str(f) for f in FileLib.get_children(top_dir, dirx=True)]
        assert sorted(files) == ['/Users/pm286/workspace/amilib/temp/test_files/dir1']
        files = [str(f) for f in FileLib.get_children(top_dir, dirx=False)]
        assert sorted(files) == ['/Users/pm286/workspace/amilib/temp/test_files/file1']

    @classmethod
    @unittest.skip("not yet right")
    def test_expand_braces(cls):
        """
        no longer tests braces
        :return:
        """

        home = os.path.expanduser("~")
        file = __file__
        python_dir = os.path.abspath(file + "/../")
        resources_dir = os.path.join(python_dir, "resources")
        logger.debug("python", os.path.abspath(python_dir))
        physchem_dir = os.path.abspath(python_dir + "/../")
        open_diagram = os.path.abspath(physchem_dir + "/../")
        logger.debug(os.path.abspath(open_diagram))
        opend = os.path.join(open_diagram, "*")
        logger.debug("opend", opend)
        logger.debug(f"od0 {glob.glob(opend)}")
        open_diagram01 = os.path.join(physchem_dir, "**", "liion10", "**", "*.*ml")
        logger.debug(f"od01 {glob.glob(open_diagram01)}")

        # open_diagram02 = os.path.join(home, "projects", "*iagram", "{*ot*,*.md}")
        # logger.debug("od02", bg.braced_glob(open_diagram02))
        open_diagram11 = os.path.join(home, "projects", "*iagram", "*", "*", "*.xml")
        logger.debug("od", open_diagram11)
        logger.debug("b", glob.glob(open_diagram11))
        open_diagram12 = os.path.join(home, "projects", "*iagram", "**", "*.xml")
        logger.debug("od", open_diagram12)

    def test_get_input_strings(self):
        """
        reads lists of words with optional split
        """
        words = FileLib.get_input_strings("foo")
        assert words == ["foo"]
        words = FileLib.get_input_strings(["foo"])
        assert words == ["foo"]
        words = FileLib.get_input_strings(["foo", "bar"])
        assert words == ["foo", "bar"]
        words = FileLib.get_input_strings("foo bar")
        assert words == ["foo bar"]
        words = FileLib.get_input_strings("foo bar", split=True)
        assert words == ["foo", "bar"]

    def test_get_input_strings_from_file(self):
        """
        reads lists of words in file/s
        """
        words = FileLib.get_input_strings(Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_2.txt"))
        assert words == ["anthropogenic", "tropospheric"]
        words = FileLib.get_input_strings(Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_10.txt"))
        assert words == [
            'anthropogenic', 'physical-biogeochemical', 'peat drainage', 'tropospheric', 'permafrost',
            'centennial', 'aerosols', 'sequestration', 'albedo', 'exacerbating'
        ]
        words = FileLib.get_input_strings(Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_10.txt"), split=True)
        # assert words == [
        #     'anthropogenic', 'physical-biogeochemical', 'peat', 'drainage', 'tropospheric', 'permafrost',
        #     'centennial', 'aerosols', 'sequestration', 'albedo', 'exacerbating'
        # ]
        assert words ==['anthropogenic',
 'physical-biogeochemical',
 'peat',
 'drainage',
 'tropospheric',
 'permafrost',
 'centennial',
 'aerosols',
 'sequestration',
 'albedo',
 'exacerbating']
        # list of files
        words = FileLib.get_input_strings([
            Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_5.txt"),
            Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_10.txt")])

        # assert words == ['anthropogenic', 'physical-biogeochemical', 'peat drainage', 'tropospheric',
        #     'permafrost', 'centennial', 'aerosols', 'sequestration', 'albedo', 'exacerbating']
        assert words == ['anthropogenic',
 'tropospheric',
 'permafrost',
 'sequestration',
 'albedo',
 'anthropogenic',
 'physical-biogeochemical',
 'peat drainage',
 'tropospheric',
 'permafrost',
 'centennial',
 'aerosols',
 'sequestration',
 'albedo',
 'exacerbating']

        # list of files
        words = FileLib.get_input_strings([
            Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_5.txt"),
            Path(Resources.TEST_RESOURCES_DIR, "wordlists", "small_10.txt")], split=True)

        assert words == ['anthropogenic',
 'tropospheric',
 'permafrost',
 'sequestration',
 'albedo',
 'anthropogenic',
 'physical-biogeochemical',
 'peat',
 'drainage',
 'tropospheric',
 'permafrost',
 'centennial',
 'aerosols',
 'sequestration',
 'albedo',
 'exacerbating']


    def test_relative_pathname(self):
        """
        convert absolute pathname to relative to another path
        some of this is in Python 3.12 but we'll support earlier versions
        """
        abspath = Path("/a/b/c/d/e.txt")
        refpath = Path("/a/b/c/")
        diffpath = FileLib.get_reletive_path(abspath, refpath)
        assert str(diffpath) == "d/e.txt"

        diffpath = FileLib.get_reletive_path(refpath, abspath)
        assert str(diffpath) == "../.."

        # dont' use Python 3.12 walk_up
        refpath = Path("/a/b/c/x/y.txt")
        try:
            diffpath = FileLib.get_reletive_path(abspath, refpath, walk_up=False)
        except ValueError as e:
            assert str(e) == "'/a/b/c/d/e.txt' is not in the subpath of '/a/b/c/x/y.txt'"

        # use Python 3.12 walk_up
        refpath = Path("/a/b/c/x/y/z.txt")

        diffpath = FileLib.get_reletive_path(abspath, refpath, walk_up=True)
        assert str(diffpath) == ("../../../d/e.txt")
        diffpath = FileLib.get_reletive_path(refpath, abspath, walk_up=True)
        assert str(diffpath) == ("../../x/y/z.txt")

    def test_force_write_operations(self):
        """Test basic file writing operations"""
        test_file = self.make_self_test_dir() / "test.txt"
        
        # Test basic write
        FileLib.force_write(test_file, "hello")
        self.assertTrue(test_file.exists())
        self.assertEqual(test_file.read_text(), "hello")
        
        # Test overwrite=False
        FileLib.force_write(test_file, "world", overwrite=False)
        self.assertEqual(test_file.read_text(), "hello")  # Should not change
        
        # Test nested path creation
        nested_file = self.test_dir / "a/b/c/test.txt"
        FileLib.force_write(nested_file, "nested")
        self.assertTrue(nested_file.exists())
        self.assertEqual(nested_file.read_text(), "nested")

    def test_copy_operations(self):
        """Test file and directory copying"""
        # Setup test files

        # source_dir = self.test_dir / "source" PMR - there is no self.test_dir
        source_dir = self.make_self_test_dir() / "source"
        source_dir.mkdir(exist_ok=True)
        assert source_dir.is_dir()
        test_txt = source_dir / "test.txt"
        test_txt = (test_txt)
        logger.debug(f"abs path {test_txt.absolute()}")
        test_txt.write_text("hello")
        assert test_txt.is_file() and not test_txt.is_dir()
        self.assertEqual(test_txt.read_text(), "hello")
        (source_dir / "subdir").mkdir(exist_ok=True)
        (source_dir / "subdir/test2.txt").write_text("world")
        
        # Test file copy
        dest_file = Path(self.test_dir, "dest.txt")
        FileLib.copy_file_or_directory(
            dest_file,
            test_txt,
            overwrite=True
        )
        self.assertTrue(dest_file.exists())
        self.assertEqual(dest_file.read_text(), "hello")
        
        # Test directory copy
        dest_dir = self.test_dir / "dest"
        FileLib.copy_file_or_directory(
            dest_dir,
            source_dir,
            overwrite=True
        )
        self.assertTrue((dest_dir / "test.txt").exists())
        self.assertTrue((dest_dir / "subdir/test2.txt").exists())

    def test_input_string_edge_cases(self):
        """Test edge cases for input string handling"""
        # Test empty inputs
        self.assertEqual(FileLib.get_input_strings(None), [])
        self.assertEqual(FileLib.get_input_strings([]), [])
        
        # Test mixed input types
        self.make_self_test_dir()
        test_file = self.test_dir / "words.txt"
        test_file.write_text("one\ntwo")
        
        result = FileLib.get_input_strings([
            "hello",
            test_file,
            "world"
        ])
        self.assertEqual(
            result,
            ["hello", "one", "two", "world"]
        )

    def make_self_test_dir(self):
        try:
            self.test_dir = self.__getattribute__("test_dir")
        except AttributeError as ae:
            self.test_dir = None
        if self.test_dir is None:
            self.test_dir = Path(Resources.TEMP_DIR, "test_files")
        self.test_dir.mkdir(exist_ok=True)
        return self.test_dir


def main():
    File0Test.test_file_simple()
    File0Test.test_templates()
