"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataCatalogDatabase = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_glue_1 = require("aws-cdk-lib/aws-glue");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const utils_1 = require("../../utils");
/**
 * An AWS Glue Data Catalog Database configured with the location and a crawler.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/data-catalog-database
 *
 * @example
 * import { Bucket } from 'aws-cdk-lib/aws-s3';
 *
 * new dsf.governance.DataCatalogDatabase(this, 'ExampleDatabase', {
 *    locationBucket: new Bucket(scope, 'LocationBucket'),
 *    locationPrefix: '/databasePath',
 *    name: 'example-db'
 * });
 */
class DataCatalogDatabase extends utils_1.TrackedConstruct {
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: DataCatalogDatabase.name,
        };
        super(scope, id, trackedConstructProps);
        this.dataCatalogDatabaseProps = props;
        const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
        this.databaseName = props.name + '_' + aws_cdk_lib_1.Names.uniqueResourceName(scope, {}).toLowerCase();
        let locationPrefix = props.locationPrefix;
        if (!locationPrefix.endsWith('/')) {
            locationPrefix += '/';
        }
        const s3LocationUri = props.locationBucket.s3UrlForObject(locationPrefix);
        this.database = new aws_glue_1.CfnDatabase(this, 'GlueDatabase', {
            catalogId: aws_cdk_lib_1.Stack.of(this).account,
            databaseInput: {
                name: this.databaseName,
                locationUri: s3LocationUri,
            },
        });
        let autoCrawl = props.autoCrawl;
        if (autoCrawl === undefined || autoCrawl === null) {
            autoCrawl = true;
        }
        const autoCrawlSchedule = props.autoCrawlSchedule || {
            scheduleExpression: 'cron(1 0 * * ? *)',
        };
        const currentStack = aws_cdk_lib_1.Stack.of(this);
        if (autoCrawl) {
            const tableLevel = props.crawlerTableLevelDepth || this.calculateDefaultTableLevelDepth(locationPrefix);
            const crawlerRole = props.crawlerRole || new aws_iam_1.Role(this, 'CrawlerRole', {
                assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
                inlinePolicies: {
                    crawlerPermissions: new aws_iam_1.PolicyDocument({
                        statements: [
                            new aws_iam_1.PolicyStatement({
                                effect: aws_iam_1.Effect.ALLOW,
                                actions: [
                                    'glue:BatchCreatePartition',
                                    'glue:BatchDeletePartition',
                                    'glue:BatchDeleteTable',
                                    'glue:BatchDeleteTableVersion',
                                    'glue:BatchGetPartition',
                                    'glue:BatchUpdatePartition',
                                    'glue:CreatePartition',
                                    'glue:CreateTable',
                                    'glue:DeletePartition',
                                    'glue:DeleteTable',
                                    'glue:GetDatabase',
                                    'glue:GetDatabases',
                                    'glue:GetPartition',
                                    'glue:GetPartitions',
                                    'glue:GetTable',
                                    'glue:GetTables',
                                    'glue:UpdateDatabase',
                                    'glue:UpdatePartition',
                                    'glue:UpdateTable',
                                ],
                                resources: [
                                    `arn:aws:glue:${currentStack.region}:${currentStack.account}:catalog`,
                                    `arn:aws:glue:${currentStack.region}:${currentStack.account}:database/${this.databaseName}`,
                                    `arn:aws:glue:${currentStack.region}:${currentStack.account}:table/${this.databaseName}/*`,
                                ],
                            }),
                            new aws_iam_1.PolicyStatement({
                                effect: aws_iam_1.Effect.ALLOW,
                                actions: [
                                    'glue:GetSecurityConfigurations',
                                    'glue:GetSecurityConfiguration',
                                ],
                                resources: ['*'],
                            }),
                        ],
                    }),
                },
            });
            props.locationBucket.grantRead(crawlerRole, locationPrefix + '*');
            this.crawlerLogEncryptionKey = props.crawlerLogEncryptionKey || new aws_kms_1.Key(this, 'CrawlerLogKey', {
                enableKeyRotation: true,
                removalPolicy: removalPolicy,
            });
            this.crawlerLogEncryptionKey.grantEncryptDecrypt(crawlerRole);
            this.crawlerSecurityConfiguration = new aws_glue_1.CfnSecurityConfiguration(this, 'CrawlerSecConfiguration', {
                name: `${props.name}-secconfig-${aws_cdk_lib_1.Names.uniqueResourceName(this, {}).toLowerCase()}`,
                encryptionConfiguration: {
                    cloudWatchEncryption: {
                        cloudWatchEncryptionMode: 'SSE-KMS',
                        kmsKeyArn: this.crawlerLogEncryptionKey.keyArn,
                    },
                    s3Encryptions: [
                        {
                            s3EncryptionMode: 'DISABLED',
                        },
                    ],
                },
            });
            const crawlerName = `${this.databaseName}-crawler-${aws_cdk_lib_1.Names.uniqueResourceName(this, {})}`;
            this.crawler = new aws_glue_1.CfnCrawler(this, 'DatabaseAutoCrawler', {
                role: crawlerRole.roleArn,
                targets: {
                    s3Targets: [{
                            path: s3LocationUri,
                        }],
                },
                schedule: autoCrawlSchedule,
                databaseName: this.databaseName,
                name: crawlerName,
                crawlerSecurityConfiguration: this.crawlerSecurityConfiguration.name,
                configuration: JSON.stringify({
                    Version: 1.0,
                    Grouping: {
                        TableLevelConfiguration: tableLevel,
                    },
                }),
            });
            const logGroup = `arn:aws:logs:${currentStack.region}:${currentStack.account}:log-group:/aws-glue/crawlers*`;
            crawlerRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'logs:CreateLogGroup',
                    'logs:CreateLogStream',
                    'logs:PutLogEvents',
                    'logs:AssociateKmsKey',
                ],
                resources: [
                    logGroup,
                    `${logGroup}:*`,
                ],
            }));
            this.crawlerRole = crawlerRole;
            this.crawlerLogEncryptionKey.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'kms:Decrypt',
                    'kms:Encrypt',
                    'kms:ReEncrypt*',
                    'kms:GenerateDataKey*',
                ],
                resources: ['*'],
                principals: [
                    new aws_iam_1.ServicePrincipal(`logs.${currentStack.region}.amazonaws.com`),
                ],
                conditions: {
                    ArnEquals: {
                        'kms:EncryptionContext:aws:logs:arn': logGroup,
                    },
                },
            }));
        }
    }
    /**
     * Grants read access via identity based policy to the principal. This would attach an IAM policy to the principal allowing read access to the database and all its tables.
     * @param principal Principal to attach the database read access to
     * @returns `AddToPrincipalPolicyResult`
     */
    grantReadOnlyAccess(principal) {
        const currentStack = aws_cdk_lib_1.Stack.of(this);
        let locationPrefix = this.dataCatalogDatabaseProps.locationPrefix;
        if (!locationPrefix.endsWith('/')) {
            locationPrefix += '/';
        }
        this.dataCatalogDatabaseProps.locationBucket.grantRead(principal, locationPrefix + '*');
        return principal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'glue:GetTable',
                'glue:GetTables',
                'glue:BatchGetPartition',
                'glue:GetDatabase',
                'glue:GetDatabases',
                'glue:GetPartition',
                'glue:GetPartitions',
            ],
            resources: [
                `arn:aws:glue:${currentStack.region}:${currentStack.account}:catalog`,
                `arn:aws:glue:${currentStack.region}:${currentStack.account}:database/${this.databaseName}`,
                `arn:aws:glue:${currentStack.region}:${currentStack.account}:table/${this.databaseName}/*`,
            ],
        }));
    }
    calculateDefaultTableLevelDepth(locationPrefix) {
        const baseCount = 2;
        const locationTokens = locationPrefix.split('/');
        let ctrValidToken = 0;
        locationTokens.forEach((token) => {
            ctrValidToken += (token) ? 1 : 0;
        });
        return ctrValidToken + baseCount;
    }
}
_a = JSII_RTTI_SYMBOL_1;
DataCatalogDatabase[_a] = { fqn: "aws-dsf.governance.DataCatalogDatabase", version: "1.0.0-rc.3" };
exports.DataCatalogDatabase = DataCatalogDatabase;
//# sourceMappingURL=data:application/json;base64,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