"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeCatalog = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const data_catalog_database_1 = require("./data-catalog-database");
const utils_1 = require("../../utils");
/**
* Creates AWS Glue Catalog Database for each storage layer. Composed of 3 {@link DataCatalogDatabase} for Bronze, Silver, and Gold data.
* @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/data-lake-catalog
 *
* @example
* import { Key } from 'aws-cdk-lib/aws-kms';
*
* const logEncryptionKey = new Key(this, 'LogEncryptionKey');
* const storage = new dsf.storage.DataLakeStorage(this, "ExampleStorage");
* const dataLakeCatalog = new dsf.governance.DataLakeCatalog(this, "ExampleDataLakeCatalog", {
*   dataLakeStorage: storage,
*   databaseName: "exampledb",
*   crawlerLogEncryptionKey: logEncryptionKey
* })
*/
class DataLakeCatalog extends utils_1.TrackedConstruct {
    /**
     * Constructs a new instance of DataLakeCatalog
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataLakeCatalogProps} props the DataLakeCatalog properties
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: DataLakeCatalog.name,
        };
        super(scope, id, trackedConstructProps);
        const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
        if (props.autoCrawl) {
            this.crawlerLogEncryptionKey = props.crawlerLogEncryptionKey || new aws_kms_1.Key(this, 'CrawlerLogKey', {
                enableKeyRotation: true,
                removalPolicy: removalPolicy,
            });
        }
        const extractedBronzeBucketName = this.extractBucketName(props.dataLakeStorage.bronzeBucket);
        const extractedSilverBucketName = this.extractBucketName(props.dataLakeStorage.silverBucket);
        const extractedGoldBucketName = this.extractBucketName(props.dataLakeStorage.goldBucket);
        const locationPrefix = props.databaseName || '/';
        this.bronzeCatalogDatabase = new data_catalog_database_1.DataCatalogDatabase(this, 'BronzeCatalogDatabase', {
            locationBucket: props.dataLakeStorage.bronzeBucket,
            locationPrefix,
            name: props.databaseName ? `${extractedBronzeBucketName}_${props.databaseName}` : extractedBronzeBucketName,
            autoCrawl: props.autoCrawl,
            autoCrawlSchedule: props.autoCrawlSchedule,
            crawlerLogEncryptionKey: this.crawlerLogEncryptionKey,
            crawlerTableLevelDepth: props.crawlerTableLevelDepth,
            removalPolicy,
        });
        this.silverCatalogDatabase = new data_catalog_database_1.DataCatalogDatabase(this, 'SilverCatalogDatabase', {
            locationBucket: props.dataLakeStorage.silverBucket,
            locationPrefix,
            name: props.databaseName ? `${extractedSilverBucketName}_${props.databaseName}` : extractedSilverBucketName,
            autoCrawl: props.autoCrawl,
            autoCrawlSchedule: props.autoCrawlSchedule,
            crawlerLogEncryptionKey: this.crawlerLogEncryptionKey,
            crawlerTableLevelDepth: props.crawlerTableLevelDepth,
            removalPolicy,
        });
        this.goldCatalogDatabase = new data_catalog_database_1.DataCatalogDatabase(this, 'GoldCatalogDatabase', {
            locationBucket: props.dataLakeStorage.goldBucket,
            locationPrefix,
            name: props.databaseName ? `${extractedGoldBucketName}_${props.databaseName}` : extractedGoldBucketName,
            autoCrawl: props.autoCrawl,
            autoCrawlSchedule: props.autoCrawlSchedule,
            crawlerLogEncryptionKey: this.crawlerLogEncryptionKey,
            crawlerTableLevelDepth: props.crawlerTableLevelDepth,
            removalPolicy,
        });
    }
    /**
     * Extract the bucket prefix from the {@link AnalyticsBucket} bucket name.
     * @param {AnalyticsBucket} bucket
     * @returns
     */
    extractBucketName(bucket) {
        const tokens = aws_cdk_lib_1.Fn.split('-', bucket.bucketName);
        return aws_cdk_lib_1.Fn.select(0, tokens);
    }
}
_a = JSII_RTTI_SYMBOL_1;
DataLakeCatalog[_a] = { fqn: "aws-dsf.governance.DataLakeCatalog", version: "1.0.0-rc.3" };
exports.DataLakeCatalog = DataLakeCatalog;
//# sourceMappingURL=data:application/json;base64,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