import { Repository } from 'aws-cdk-lib/aws-codecommit';
import { ILogGroup } from 'aws-cdk-lib/aws-logs';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { CodeBuildStep, CodePipeline } from 'aws-cdk-lib/pipelines';
import { Construct } from 'constructs';
import { SparkEmrCICDPipelineProps } from './spark-emr-cicd-pipeline-props';
import { TrackedConstruct } from '../../../utils';
/**
 * The account information for deploying the Spark Application stack.
 */
export interface AccountInfo {
    /**
     * The account ID to deploy the Spark Application stack
     */
    readonly account: string;
    /**
     * The region to deploy the Spark Application stack
     */
    readonly region: string;
}
/**
 * A CICD Pipeline that tests and deploys a Spark application in cross-account environments using CDK Pipelines.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-cicd-pipeline
 *
 * @exampleMetadata fixture=imports-only
 * @example
 * import { Bucket } from 'aws-cdk-lib/aws-s3';
 *
 * interface MyApplicationStackProps extends cdk.StackProps {
 *   readonly stage: dsf.utils.CICDStage;
 * }
 *
 * class MyApplicationStack extends cdk.Stack {
 *   constructor(scope: cdk.Stack, props?: MyApplicationStackProps) {
 *     super(scope, 'MyApplicationStack');
 *     const bucket = new Bucket(this, 'TestBucket', {
 *       autoDeleteObjects: true,
 *       removalPolicy: cdk.RemovalPolicy.DESTROY,
 *     });
 *     new cdk.CfnOutput(this, 'BucketName', { value: bucket.bucketName });
 *   }
 * }
 *
 * class MyStackFactory implements dsf.utils.ApplicationStackFactory {
 *   createStack(scope: cdk.Stack, stage: dsf.utils.CICDStage): cdk.Stack {
 *     return new MyApplicationStack(scope, { stage });
 *   }
 * }
 *
 * class MyCICDStack extends cdk.Stack {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id);
 *     new dsf.processing.SparkEmrCICDPipeline(this, 'TestConstruct', {
 *        sparkApplicationName: 'test',
 *        applicationStackFactory: new MyStackFactory(),
 *        cdkApplicationPath: 'cdk/',
 *        sparkApplicationPath: 'spark/',
 *        sparkImage: dsf.processing.SparkImage.EMR_6_12,
 *        integTestScript: 'cdk/integ-test.sh',
 *        integTestEnv: {
 *          TEST_BUCKET: 'BucketName',
 *        },
 *     });
 *   }
 * }
 */
export declare class SparkEmrCICDPipeline extends TrackedConstruct {
    /**
     * Extract the path and the script name from a script path
     * @param path the script path
     * @return [path, scriptName]
     */
    private static extractPath;
    /**
     * Build the install commands for the CodeBuild step based on the runtime
     * @param cdkPath the path of the CDK application
     * @param sparkPath the path of the Spark application
     * @return installCommands
     */
    private static synthCommands;
    /**
     * The CodePipeline created as part of the Spark CICD Pipeline
     */
    readonly pipeline: CodePipeline;
    /**
     * The CodeCommit repository created as part of the Spark CICD Pipeline
     */
    readonly repository: Repository;
    /**
     * The S3 bucket for storing the artifacts
     */
    readonly artifactBucket: IBucket;
    /**
     * The S3 bucket for storing access logs on the artifact bucket
     */
    readonly artifactAccessLogsBucket: IBucket;
    /**
     * The CloudWatch Log Group for storing the CodePipeline logs
     */
    readonly pipelineLogGroup: ILogGroup;
    /**
     * The CodeBuild Step for the staging stage
     */
    readonly integrationTestStage?: CodeBuildStep;
    /**
     * Construct a new instance of the SparkCICDPipeline class.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {SparkEmrCICDPipelineProps} props the SparkCICDPipelineProps properties
     */
    constructor(scope: Construct, id: string, props: SparkEmrCICDPipelineProps);
    /**
     * Extract PROD and STAGING account IDs and regions from the CDK context
     */
    private getAccountFromContext;
}
