"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SparkEmrCICDPipeline = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codecommit_1 = require("aws-cdk-lib/aws-codecommit");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const pipelines_1 = require("aws-cdk-lib/pipelines");
const storage_1 = require("../../../storage");
const utils_1 = require("../../../utils");
const emr_releases_1 = require("../emr-releases");
/**
 * A CICD Pipeline that tests and deploys a Spark application in cross-account environments using CDK Pipelines.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-cicd-pipeline
 *
 * @exampleMetadata fixture=imports-only
 * @example
 * import { Bucket } from 'aws-cdk-lib/aws-s3';
 *
 * interface MyApplicationStackProps extends cdk.StackProps {
 *   readonly stage: dsf.utils.CICDStage;
 * }
 *
 * class MyApplicationStack extends cdk.Stack {
 *   constructor(scope: cdk.Stack, props?: MyApplicationStackProps) {
 *     super(scope, 'MyApplicationStack');
 *     const bucket = new Bucket(this, 'TestBucket', {
 *       autoDeleteObjects: true,
 *       removalPolicy: cdk.RemovalPolicy.DESTROY,
 *     });
 *     new cdk.CfnOutput(this, 'BucketName', { value: bucket.bucketName });
 *   }
 * }
 *
 * class MyStackFactory implements dsf.utils.ApplicationStackFactory {
 *   createStack(scope: cdk.Stack, stage: dsf.utils.CICDStage): cdk.Stack {
 *     return new MyApplicationStack(scope, { stage });
 *   }
 * }
 *
 * class MyCICDStack extends cdk.Stack {
 *   constructor(scope: Construct, id: string) {
 *     super(scope, id);
 *     new dsf.processing.SparkEmrCICDPipeline(this, 'TestConstruct', {
 *        sparkApplicationName: 'test',
 *        applicationStackFactory: new MyStackFactory(),
 *        cdkApplicationPath: 'cdk/',
 *        sparkApplicationPath: 'spark/',
 *        sparkImage: dsf.processing.SparkImage.EMR_6_12,
 *        integTestScript: 'cdk/integ-test.sh',
 *        integTestEnv: {
 *          TEST_BUCKET: 'BucketName',
 *        },
 *     });
 *   }
 * }
 */
class SparkEmrCICDPipeline extends utils_1.TrackedConstruct {
    /**
     * Extract the path and the script name from a script path
     * @param path the script path
     * @return [path, scriptName]
     */
    static extractPath(path) {
        const pathParts = path.split('/');
        const integPath = pathParts.length > 1 ? pathParts.slice(0, -1).join('/') : '.';
        const integScript = pathParts[pathParts.length - 1];
        return [integPath, integScript];
    }
    /**
     * Build the install commands for the CodeBuild step based on the runtime
     * @param cdkPath the path of the CDK application
     * @param sparkPath the path of the Spark application
     * @return installCommands
     */
    static synthCommands(cdkPath, sparkPath, sparkImage) {
        // Get the runtime of the CDK Construct
        const runtime = process.env.JSII_AGENT || 'node.js';
        let commands = [
            'curl -qLk -o jq https://stedolan.github.io/jq/download/linux64/jq && chmod +x ./jq',
            'curl -qL -o aws_credentials.json http://169.254.170.2/$AWS_CONTAINER_CREDENTIALS_RELATIVE_URI',
            "eval \"$(jq -r '@sh \"AWS_ACCESS_KEY_ID=\\\(.AccessKeyId) AWS_SECRET_ACCESS_KEY=\\\(.SecretAccessKey) AWS_SESSION_TOKEN=\\\(.Token)\"' aws_credentials.json)\"",
            'rm -f aws_credentials.json',
            `chmod -R o+w $(pwd)/${sparkPath}`,
            `docker run -i -v $(pwd)/${sparkPath}:/home/hadoop/ -e AWS_REGION=$AWS_REGION -e AWS_ACCESS_KEY_ID=$AWS_ACCESS_KEY_ID -e AWS_SECRET_ACCESS_KEY=$AWS_SECRET_ACCESS_KEY -e AWS_SESSION_TOKEN=$AWS_SESSION_TOKEN -e DISABLE_SSL=true --rm --name pytest ${sparkImage} sh -c \"export PATH=$PATH:/home/hadoop/.local/bin && export PYTHONPATH=$PYTHONPATH:/usr/lib/spark/python/lib/py4j-src.zip:/usr/lib/spark/python && python3 -m pip install pytest . && python3 -m pytest\"`,
            `cd ${cdkPath}`,
            'npm install -g aws-cdk',
        ];
        // Build the list of commands depending on the runtime
        switch (runtime.split('/')[0].toLowerCase()) {
            case 'node.js':
                commands = commands.concat([
                    'npm ci',
                    'npm run build',
                ]);
                break;
            case 'python':
                commands = commands.concat([
                    'pip install -r requirements.txt',
                ]);
                break;
            default:
                throw new Error('Runtime not supported');
        }
        // Full set of commands
        return commands.concat(['npx cdk synth --all']);
    }
    /**
     * Construct a new instance of the SparkCICDPipeline class.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {SparkEmrCICDPipelineProps} props the SparkCICDPipelineProps properties
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: SparkEmrCICDPipeline.name,
        };
        super(scope, id, trackedConstructProps);
        const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
        // Set the defaults
        const cdkPath = props.cdkApplicationPath ? props.cdkApplicationPath : '.';
        const sparkPath = props.sparkApplicationPath ? props.sparkApplicationPath : '.';
        const sparkImage = props.sparkImage ? props.sparkImage : emr_releases_1.DEFAULT_SPARK_IMAGE;
        // Create a CodeCommit repository to host the code
        this.repository = new aws_codecommit_1.Repository(this, 'CodeCommitRepository', {
            repositoryName: props.sparkApplicationName,
        });
        const buildStage = new pipelines_1.CodeBuildStep('CodeBuildSynthStep', {
            input: pipelines_1.CodePipelineSource.codeCommit(this.repository, 'main'),
            commands: SparkEmrCICDPipeline.synthCommands(cdkPath, sparkPath, sparkImage),
            primaryOutputDirectory: `${cdkPath}/cdk.out`,
        });
        this.artifactAccessLogsBucket = new storage_1.AccessLogsBucket(this, 'AccessLogsBucket', {
            removalPolicy: props?.removalPolicy,
        });
        const artifactBucketKey = new aws_kms_1.Key(this, 'ArtifactBucketKey', {
            removalPolicy,
            enableKeyRotation: true,
        });
        this.artifactBucket = new aws_s3_1.Bucket(this, 'ArtifactBucket', {
            enforceSSL: true,
            encryption: aws_s3_1.BucketEncryption.KMS,
            encryptionKey: artifactBucketKey,
            removalPolicy,
            serverAccessLogsBucket: this.artifactAccessLogsBucket,
            autoDeleteObjects: removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.pipelineLogGroup = new aws_logs_1.LogGroup(this, 'BuildLogGroup', {
            removalPolicy: removalPolicy,
        }),
            // Create the CodePipeline to run the CICD
            this.pipeline = new pipelines_1.CodePipeline(this, 'CodePipeline', {
                crossAccountKeys: true,
                enableKeyRotation: true,
                useChangeSets: false,
                synth: buildStage,
                dockerEnabledForSynth: true,
                artifactBucket: this.artifactBucket,
                codeBuildDefaults: {
                    logging: {
                        cloudWatch: {
                            logGroup: this.pipelineLogGroup,
                        },
                    },
                },
            });
        // Create the Staging stage of the CICD
        const staging = new utils_1.ApplicationStage(this, 'Staging', {
            env: this.getAccountFromContext('staging'),
            applicationStackFactory: props.applicationStackFactory,
            outputsEnv: props.integTestEnv,
            stage: utils_1.CICDStage.STAGING,
        });
        const stagingDeployment = this.pipeline.addStage(staging);
        if (props.integTestScript) {
            // Extract the path and script name from the integration tests script path
            const [integPath, integScript] = SparkEmrCICDPipeline.extractPath(props.integTestScript);
            this.integrationTestStage = new pipelines_1.CodeBuildStep('IntegrationTests', {
                input: buildStage.addOutputDirectory(integPath),
                commands: [`chmod +x ${integScript} && ./${integScript}`],
                envFromCfnOutputs: staging.stackOutputsEnv,
                rolePolicyStatements: props.integTestPermissions,
            });
            // Add a post step to run the integration tests
            stagingDeployment.addPost(this.integrationTestStage);
        }
        // Create the Production stage of the CICD
        this.pipeline.addStage(new utils_1.ApplicationStage(this, 'Production', {
            env: this.getAccountFromContext('prod'),
            applicationStackFactory: props.applicationStackFactory,
            stage: utils_1.CICDStage.PROD,
        }));
        // Create a CfnOutput to display the CodeCommit repository URL
        new aws_cdk_lib_1.CfnOutput(this, 'CodeCommitRepositoryCommand', {
            value: `git remote add ${this.repository.repositoryName} codecommit::${aws_cdk_lib_1.Aws.REGION}://${this.repository.repositoryName}`,
        });
    }
    /**
     * Extract PROD and STAGING account IDs and regions from the CDK context
     */
    getAccountFromContext(name) {
        const account = this.node.tryGetContext(name);
        if (!account)
            throw new Error(`Missing context variable ${name}`);
        return account;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SparkEmrCICDPipeline[_a] = { fqn: "aws-dsf.processing.SparkEmrCICDPipeline", version: "1.0.0-rc.3" };
exports.SparkEmrCICDPipeline = SparkEmrCICDPipeline;
//# sourceMappingURL=data:application/json;base64,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