import { IManagedPolicy, IRole } from 'aws-cdk-lib/aws-iam';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { Construct } from 'constructs';
import { PySparkApplicationPackageProps } from './pyspark-application-package-props';
import { TrackedConstruct } from '../../../utils';
/**
 * A construct that takes your PySpark application, packages its virtual environment and uploads it along its entrypoint to an Amazon S3 bucket
 * This construct requires Docker daemon installed locally to run.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/pyspark-application-package
 *
 * @example
 * let pysparkPacker = new dsf.processing.PySparkApplicationPackage (this, 'pysparkPacker', {
 *   applicationName: 'my-pyspark',
 *   entrypointPath: '/Users/my-user/my-spark-job/app/app-pyspark.py',
 *   dependenciesFolder: '/Users/my-user/my-spark-job/app',
 *   removalPolicy: cdk.RemovalPolicy.DESTROY,
 * });
 *
 * let sparkEnvConf: string = `--conf spark.archives=${pysparkPacker.venvArchiveS3Uri} --conf spark.emr-serverless.driverEnv.PYSPARK_DRIVER_PYTHON=./environment/bin/python --conf spark.emr-serverless.driverEnv.PYSPARK_PYTHON=./environment/bin/python --conf spark.emr-serverless.executorEnv.PYSPARK_PYTHON=./environment/bin/python`
 *
 * new dsf.processing.SparkEmrServerlessJob(this, 'SparkJobServerless', {
 *   name: 'MyPySpark',
 *   applicationId: 'xxxxxxxxx',
 *   executionRoleArn: 'ROLE-ARN',
 *   executionTimeoutMinutes: 30,
 *   s3LogUri: 's3://s3-bucket/monitoring-logs',
 *   cloudWatchLogGroupName: 'my-pyspark-serverless-log',
 *   sparkSubmitEntryPoint: `${pysparkPacker.entrypointS3Uri}`,
 *   sparkSubmitParameters: `--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.driver.memory=2G --conf spark.executor.cores=4 ${sparkEnvConf}`,
 * } as dsf.processing.SparkEmrServerlessJobProps);
 */
export declare class PySparkApplicationPackage extends TrackedConstruct {
    /**
     * The prefix used to store artifacts on the artifact bucket
     */
    static readonly ARTIFACTS_PREFIX = "emr-artifacts";
    /**
     * The S3 location where the entry point is saved in S3.
     * You pass this location to your Spark job.
     */
    readonly entrypointS3Uri: string;
    /**
     * The S3 location where the archive of the Python virtual environment with all dependencies is stored.
     * You pass this location to your Spark job.
     */
    readonly venvArchiveS3Uri?: string;
    /**
     * The Spark config containing the configuration of virtual environment archive with all dependencies.
     */
    readonly sparkVenvConf?: string;
    /**
     * The bucket storing the artifacts (entrypoint and virtual environment archive).
     */
    readonly artifactsBucket: IBucket;
    /**
     * The role used by the BucketDeployment to upload the artifacts to an s3 bucket.
     * In case you provide your own bucket for storing the artifacts (entrypoint and virtual environment archive),
     * you must provide s3 write access to this role to upload the artifacts.
     */
    readonly assetUploadRole: IRole;
    /**
     * The IAM managed policy used by the custom resource for the assets deployment
     */
    readonly assetUploadManagedPolicy: IManagedPolicy;
    /**
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param props {@link PySparkApplicationPackageProps}
     */
    constructor(scope: Construct, id: string, props: PySparkApplicationPackageProps);
}
