"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.PySparkApplicationPackage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const path = require("path");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const core_1 = require("aws-cdk-lib/core");
const utils_1 = require("../../../utils");
/**
 * A construct that takes your PySpark application, packages its virtual environment and uploads it along its entrypoint to an Amazon S3 bucket
 * This construct requires Docker daemon installed locally to run.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/pyspark-application-package
 *
 * @example
 * let pysparkPacker = new dsf.processing.PySparkApplicationPackage (this, 'pysparkPacker', {
 *   applicationName: 'my-pyspark',
 *   entrypointPath: '/Users/my-user/my-spark-job/app/app-pyspark.py',
 *   dependenciesFolder: '/Users/my-user/my-spark-job/app',
 *   removalPolicy: cdk.RemovalPolicy.DESTROY,
 * });
 *
 * let sparkEnvConf: string = `--conf spark.archives=${pysparkPacker.venvArchiveS3Uri} --conf spark.emr-serverless.driverEnv.PYSPARK_DRIVER_PYTHON=./environment/bin/python --conf spark.emr-serverless.driverEnv.PYSPARK_PYTHON=./environment/bin/python --conf spark.emr-serverless.executorEnv.PYSPARK_PYTHON=./environment/bin/python`
 *
 * new dsf.processing.SparkEmrServerlessJob(this, 'SparkJobServerless', {
 *   name: 'MyPySpark',
 *   applicationId: 'xxxxxxxxx',
 *   executionRoleArn: 'ROLE-ARN',
 *   executionTimeoutMinutes: 30,
 *   s3LogUri: 's3://s3-bucket/monitoring-logs',
 *   cloudWatchLogGroupName: 'my-pyspark-serverless-log',
 *   sparkSubmitEntryPoint: `${pysparkPacker.entrypointS3Uri}`,
 *   sparkSubmitParameters: `--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.driver.memory=2G --conf spark.executor.cores=4 ${sparkEnvConf}`,
 * } as dsf.processing.SparkEmrServerlessJobProps);
 */
class PySparkApplicationPackage extends utils_1.TrackedConstruct {
    /**
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param props {@link PySparkApplicationPackageProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: PySparkApplicationPackage.name,
        };
        super(scope, id, trackedConstructProps);
        const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
        const entrypointFileName = path.basename(props.entrypointPath);
        const entrypointDirectory = path.dirname(props.entrypointPath);
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        // Policy to allow lambda access to cloudwatch logs
        this.assetUploadManagedPolicy = new aws_iam_1.ManagedPolicy(this, 's3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct for PySparkApplicationPackage',
        });
        // Create or use the passed `assetUploadRole` as an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadRole = props.assetUploadRole || new aws_iam_1.Role(this, 's3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct for PySparkApplicationPackage',
            managedPolicies: [this.assetUploadManagedPolicy],
        });
        let artifactsBucket;
        if (!props.artifactsBucket) {
            artifactsBucket = new aws_s3_1.Bucket(this, 'ArtifactBucket', {
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                enforceSSL: true,
                removalPolicy: removalPolicy,
                autoDeleteObjects: removalPolicy === core_1.RemovalPolicy.DESTROY,
                serverAccessLogsPrefix: 'access-logs',
            });
        }
        else {
            artifactsBucket = props.artifactsBucket;
        }
        artifactsBucket.grantWrite(this.assetUploadRole);
        // package dependencies if there are any
        if (props.dependenciesFolder) {
            // The sparkVenvArchivePath is required if there are dependencies
            if (!props.venvArchivePath) {
                throw new Error('Virtual environment archive path is required if there are dependencies');
            }
            else {
                const venvArchiveFileName = path.basename(props.venvArchivePath);
                // Build dependencies using the Dockerfile in app/ folder and deploy a zip into CDK asset bucket
                const emrDepsAsset = new aws_s3_assets_1.Asset(this, 'EmrDepsAsset', {
                    path: props.dependenciesFolder,
                    bundling: {
                        image: core_1.DockerImage.fromBuild(props.dependenciesFolder),
                        outputType: core_1.BundlingOutput.ARCHIVED,
                        command: [
                            'sh',
                            '-c',
                            `cp ${props.venvArchivePath} /asset-output/`,
                        ],
                    },
                });
                emrDepsAsset.bucket.grantRead(this.assetUploadRole);
                // Move the asset with dependencies into the artifact bucket (because it's a different lifecycle than the CDK app)
                const emrDepsArtifacts = new aws_s3_deployment_1.BucketDeployment(this, 'EmrDepsArtifacts', {
                    sources: [
                        aws_s3_deployment_1.Source.bucket(emrDepsAsset.bucket, emrDepsAsset.s3ObjectKey),
                    ],
                    destinationBucket: artifactsBucket,
                    destinationKeyPrefix: `${PySparkApplicationPackage.ARTIFACTS_PREFIX}/${props.applicationName}`,
                    memoryLimit: 512,
                    ephemeralStorageSize: core_1.Size.mebibytes(1000),
                    prune: false,
                    extract: false,
                    role: this.assetUploadRole,
                    retainOnDelete: removalPolicy === core_1.RemovalPolicy.RETAIN,
                });
                this.venvArchiveS3Uri = emrDepsArtifacts.deployedBucket.s3UrlForObject(`${PySparkApplicationPackage.ARTIFACTS_PREFIX}/${props.applicationName}/${venvArchiveFileName}`);
            }
        }
        // Deploy a zip of the Pyspark entrypoint into the CDK asset bucket
        // We are using an S3 asset because the name of the file changes each time we deploy the app
        // If we used an S3 deployment directly, the entrypoint would still have the same name, and the resource wouldn't be updated by CDK
        const emrAppAsset = new aws_s3_assets_1.Asset(this, 'EmrAppAsset', {
            path: entrypointDirectory,
            bundling: {
                image: core_1.DockerImage.fromRegistry('public.ecr.aws/amazonlinux/amazonlinux:2023-minimal'),
                outputType: core_1.BundlingOutput.NOT_ARCHIVED,
                command: [
                    'sh',
                    '-c',
                    `cp /asset-input/${entrypointFileName} /asset-output/`,
                ],
            },
        });
        // Move the asset with the Pyspark entrypoint into the artifact bucket (because it's a different lifecycle than the CDK app)
        const emrAppArtifacts = new aws_s3_deployment_1.BucketDeployment(this, 'EmrAppArtifacts', {
            sources: [
                aws_s3_deployment_1.Source.bucket(emrAppAsset.bucket, emrAppAsset.s3ObjectKey),
            ],
            destinationBucket: artifactsBucket,
            destinationKeyPrefix: `${PySparkApplicationPackage.ARTIFACTS_PREFIX}/${props.applicationName}`,
            memoryLimit: 512,
            ephemeralStorageSize: core_1.Size.mebibytes(1000),
            prune: false,
            role: this.assetUploadRole,
            retainOnDelete: removalPolicy === core_1.RemovalPolicy.RETAIN,
        });
        this.entrypointS3Uri = emrAppArtifacts.deployedBucket.s3UrlForObject(`${PySparkApplicationPackage.ARTIFACTS_PREFIX}/${props.applicationName}/${entrypointFileName}`);
        this.artifactsBucket = artifactsBucket;
        this.sparkVenvConf = `--conf spark.archives=${this.venvArchiveS3Uri} --conf spark.emr-serverless.driverEnv.PYSPARK_DRIVER_PYTHON=./environment/bin/python --conf spark.emr-serverless.driverEnv.PYSPARK_PYTHON=./environment/bin/python --conf spark.emr-serverless.executorEnv.PYSPARK_PYTHON=./environment/bin/python`;
    }
}
_a = JSII_RTTI_SYMBOL_1;
PySparkApplicationPackage[_a] = { fqn: "aws-dsf.processing.PySparkApplicationPackage", version: "1.0.0-rc.3" };
/**
 * The prefix used to store artifacts on the artifact bucket
 */
PySparkApplicationPackage.ARTIFACTS_PREFIX = 'emr-artifacts';
exports.PySparkApplicationPackage = PySparkApplicationPackage;
//# sourceMappingURL=data:application/json;base64,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