import { IRole } from 'aws-cdk-lib/aws-iam';
import { FailProps } from 'aws-cdk-lib/aws-stepfunctions';
import { CallAwsServiceProps } from 'aws-cdk-lib/aws-stepfunctions-tasks';
import { Construct } from 'constructs';
import { SparkJob } from './spark-job';
import { SparkEmrEksJobApiProps, SparkEmrEksJobProps } from './spark-job-emr-eks-props';
/**
 * A construct to run Spark Jobs using EMR on EKS.
 * Creates a Step Functions State Machine that orchestrates the Spark Job.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-job
 *
 * @example
 * import { JsonPath } from 'aws-cdk-lib/aws-stepfunctions';
 *
 * const job = new dsf.processing.SparkEmrEksJob(this, 'SparkJob', {
 *   jobConfig:{
 *     "Name": JsonPath.format('ge_profile-{}', JsonPath.uuid()),
 *     "VirtualClusterId": "virtualClusterId",
 *     "ExecutionRoleArn": "ROLE-ARN",
 *     "JobDriver": {
 *       "SparkSubmit": {
 *           "EntryPoint": "s3://S3-BUCKET/pi.py",
 *           "EntryPointArguments": [],
 *           "SparkSubmitParameters": "--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.driver.memory=2G --conf spark.executor.cores=4"
 *       },
 *     }
 *   }
 * } as dsf.processing.SparkEmrEksJobApiProps);
 *
 * new cdk.CfnOutput(this, 'SparkJobStateMachine', {
 *   value: job.stateMachine!.stateMachineArn,
 * });
 */
export declare class SparkEmrEksJob extends SparkJob {
    private constructJobConfig;
    constructor(scope: Construct, id: string, props: SparkEmrEksJobProps | SparkEmrEksJobApiProps);
    /**
     * Returns the props for the Step Functions CallAwsService Construct that starts the Spark job, it calls the [StartJobRun API](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_StartJobRun.html)
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    protected returnJobStartTaskProps(): CallAwsServiceProps;
    /**
     * Returns the props for the Step Functions CallAwsService Construct that checks the execution status of the Spark job
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    protected returnJobMonitorTaskProps(): CallAwsServiceProps;
    /**
     * Returns the props for the Step Functions task that handles the failure  if the EMR Serverless job fails.
     * @returns FailProps The error details of the failed Spark Job
     */
    protected returnJobFailTaskProps(): FailProps;
    /**
     * Returns the status of the EMR on EKS job that succeeded  based on the GetJobRun API response
     * @returns string
     */
    protected returnJobStatusSucceed(): string;
    /**
     * Returns the status of the EMR on EKS job that failed based on the GetJobRun API response
     * @returns string
     */
    protected returnJobStatusFailed(): string;
    /**
     * Returns the status of the EMR Serverless job that is cancelled based on the GetJobRun API response
     * @returns string
     */
    protected returnJobStatusCancelled(): string;
    /**
     * Grants the necessary permissions to the Step Functions StateMachine to be able to start EMR on EKS job
     * @param role Step Functions StateMachine IAM role
     */
    protected grantExecutionRole(role: IRole): void;
    /**
     * Set defaults for the EmrOnEksSparkJobApiProps.
     * @param props EmrOnEksSparkJobApiProps
     */
    private setJobApiPropsDefaults;
    /**
     * Set defaults for the EmrOnEksSparkJobProps.
     * @param props EmrOnEksSparkJobProps
     */
    private setJobPropsDefaults;
}
