"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SparkEmrEksJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const spark_job_1 = require("./spark-job");
const utils_1 = require("../../../utils");
const emr_releases_1 = require("../emr-releases");
/**
 * A construct to run Spark Jobs using EMR on EKS.
 * Creates a Step Functions State Machine that orchestrates the Spark Job.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-job
 *
 * @example
 * import { JsonPath } from 'aws-cdk-lib/aws-stepfunctions';
 *
 * const job = new dsf.processing.SparkEmrEksJob(this, 'SparkJob', {
 *   jobConfig:{
 *     "Name": JsonPath.format('ge_profile-{}', JsonPath.uuid()),
 *     "VirtualClusterId": "virtualClusterId",
 *     "ExecutionRoleArn": "ROLE-ARN",
 *     "JobDriver": {
 *       "SparkSubmit": {
 *           "EntryPoint": "s3://S3-BUCKET/pi.py",
 *           "EntryPointArguments": [],
 *           "SparkSubmitParameters": "--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.driver.memory=2G --conf spark.executor.cores=4"
 *       },
 *     }
 *   }
 * } as dsf.processing.SparkEmrEksJobApiProps);
 *
 * new cdk.CfnOutput(this, 'SparkJobStateMachine', {
 *   value: job.stateMachine!.stateMachineArn,
 * });
 */
class SparkEmrEksJob extends spark_job_1.SparkJob {
    constructor(scope, id, props) {
        super(scope, id, SparkEmrEksJob.name, props);
        let sparkJobExecutionRole;
        if ('jobConfig' in props) {
            this.constructJobConfig = this.setJobApiPropsDefaults(props);
        }
        else {
            this.constructJobConfig = this.setJobPropsDefaults(props);
        }
        sparkJobExecutionRole = aws_iam_1.Role.fromRoleArn(this, `spakrJobRole-${id}`, this.constructJobConfig.jobConfig.ExecutionRoleArn);
        //Tag the AWS Step Functions State Machine
        if (!this.constructJobConfig.jobConfig.Tags) {
            this.constructJobConfig.jobConfig.Tags = {};
        }
        this.constructJobConfig.jobConfig.Tags[utils_1.TrackedConstruct.DSF_OWNED_TAG] = 'true';
        const executionTimeout = props.executionTimeoutMinutes ?? 30;
        this.stateMachine = this.createStateMachine(aws_cdk_lib_1.Duration.minutes(executionTimeout), this.constructJobConfig.schedule);
        this.s3LogBucket?.grantReadWrite(sparkJobExecutionRole);
        this.emrJobLogGroup?.grantRead(sparkJobExecutionRole);
        this.emrJobLogGroup?.grantWrite(sparkJobExecutionRole);
        if (this.emrJobLogGroup) {
            sparkJobExecutionRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: ['logs:DescribeLogGroups', 'logs:DescribeLogStreams'],
                resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group::log-stream:*`],
            }));
        }
    }
    /**
     * Returns the props for the Step Functions CallAwsService Construct that starts the Spark job, it calls the [StartJobRun API](https://docs.aws.amazon.com/emr-on-eks/latest/APIReference/API_StartJobRun.html)
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    returnJobStartTaskProps() {
        return {
            service: 'emrcontainers',
            action: 'startJobRun',
            iamAction: 'emr-containers:StartJobRun',
            parameters: this.constructJobConfig.jobConfig,
            iamResources: [`arn:aws:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/${this.constructJobConfig.jobConfig.VirtualClusterId}`],
            resultSelector: {
                'JobRunId.$': '$.Id',
            },
        };
    }
    /**
     * Returns the props for the Step Functions CallAwsService Construct that checks the execution status of the Spark job
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    returnJobMonitorTaskProps() {
        return {
            service: 'emrcontainers',
            action: 'describeJobRun',
            iamAction: 'emr-container:DescribeJobRun',
            parameters: {
                VirtualClusterId: this.constructJobConfig.jobConfig.VirtualClusterId,
                Id: aws_stepfunctions_1.JsonPath.stringAt('$.JobRunId'),
            },
            iamResources: [`arn:aws:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/${this.constructJobConfig.jobConfig.VirtualClusterId}/jobruns/*`],
            resultSelector: {
                'State.$': '$.State',
                'StateDetails.$': '$.StateDetails',
            },
            resultPath: '$.JobRunState',
        };
    }
    /**
     * Returns the props for the Step Functions task that handles the failure  if the EMR Serverless job fails.
     * @returns FailProps The error details of the failed Spark Job
     */
    returnJobFailTaskProps() {
        return {
            cause: 'EMRonEKSJobFailed',
            error: aws_stepfunctions_1.JsonPath.stringAt('$.JobRunState.StateDetails'),
        };
    }
    /**
     * Returns the status of the EMR on EKS job that succeeded  based on the GetJobRun API response
     * @returns string
     */
    returnJobStatusSucceed() {
        return 'COMPLETED';
    }
    /**
     * Returns the status of the EMR on EKS job that failed based on the GetJobRun API response
     * @returns string
     */
    returnJobStatusFailed() {
        return 'FAILED';
    }
    /**
     * Returns the status of the EMR Serverless job that is cancelled based on the GetJobRun API response
     * @returns string
     */
    returnJobStatusCancelled() {
        return 'CANCELLED';
    }
    /**
     * Grants the necessary permissions to the Step Functions StateMachine to be able to start EMR on EKS job
     * @param role Step Functions StateMachine IAM role
     */
    grantExecutionRole(role) {
        role.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'emr-containers:StartJobRun',
                'emr-containers:DescribeJobRun',
            ],
            resources: [`arn:aws:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/${this.constructJobConfig.jobConfig.VirtualClusterId}`, `arn:aws:emr-containers:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/virtualclusters/${this.constructJobConfig.jobConfig.VirtualClusterId}/jobruns/*`],
            conditions: {
                StringEquals: {
                    'emr-containers:ExecutionRoleArn': this.constructJobConfig.jobConfig.ExecutionRoleArn,
                },
            },
        }));
    }
    /**
     * Set defaults for the EmrOnEksSparkJobApiProps.
     * @param props EmrOnEksSparkJobApiProps
     */
    setJobApiPropsDefaults(props) {
        const propsPascalCase = utils_1.StepFunctionUtils.camelToPascal(props.jobConfig);
        //Set defaults
        propsPascalCase.ClientToken ?? (propsPascalCase.ClientToken = aws_stepfunctions_1.JsonPath.uuid());
        propsPascalCase.ReleaseLabel ?? (propsPascalCase.ReleaseLabel = emr_releases_1.EMR_DEFAULT_VERSION);
        return {
            jobConfig: propsPascalCase,
            removalPolicy: props.removalPolicy,
            schedule: props.schedule,
        };
    }
    /**
     * Set defaults for the EmrOnEksSparkJobProps.
     * @param props EmrOnEksSparkJobProps
     */
    setJobPropsDefaults(props) {
        const config = {
            jobConfig: {
                ConfigurationOverrides: {
                    MonitoringConfiguration: {
                        S3MonitoringConfiguration: {},
                    },
                },
                RetryPolicyConfiguration: {},
                JobDriver: {
                    SparkSubmitJobDriver: {},
                },
            },
        };
        config.jobConfig.Name = props.name;
        config.jobConfig.ClientToken = aws_stepfunctions_1.JsonPath.uuid();
        config.jobConfig.VirtualClusterId = props.virtualClusterId;
        config.jobConfig.ExecutionRoleArn = props.executionRoleArn;
        config.jobConfig.JobDriver.SparkSubmitJobDriver.EntryPoint = props.sparkSubmitEntryPoint;
        if (props.sparkSubmitEntryPointArguments) {
            config.jobConfig.JobDriver.SparkSubmitJobDriver.EntryPointArguments = props.sparkSubmitEntryPointArguments;
        }
        if (props.sparkSubmitParameters) {
            config.jobConfig.JobDriver.SparkSubmitJobDriver.SparkSubmitParameters = props.sparkSubmitParameters;
        }
        if (props.applicationConfiguration) {
            config.jobConfig.ConfigurationOverrides.ApplicationConfiguration = utils_1.StepFunctionUtils.camelToPascal(props.applicationConfiguration);
        }
        config.jobConfig.RetryPolicyConfiguration.MaxAttempts = props.maxRetries ?? 0;
        if (props.s3LogUri && !props.s3LogUri.match(/^s3:\/\/([^\/]+)/) && !props.s3LogUri.match(/^Token\[([0-9]+)\]$/)) {
            throw new Error(`Invalid S3 URI: ${props.s3LogUri}`);
        }
        config.jobConfig.ConfigurationOverrides.MonitoringConfiguration.S3MonitoringConfiguration.LogUri =
            this.createS3LogBucket(props.s3LogUri);
        if (props.cloudWatchLogGroupName) {
            this.createCloudWatchLogsLogGroup(props.cloudWatchLogGroupName);
            config.jobConfig.ConfigurationOverrides.MonitoringConfiguration.CloudWatchMonitoringConfiguration = {
                LogGroupName: props.cloudWatchLogGroupName,
                LogStreamNamePrefix: props.cloudWatchLogGroupStreamPrefix ?? props.name,
            };
        }
        config.jobConfig.Tags = props.tags;
        return config;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SparkEmrEksJob[_a] = { fqn: "aws-dsf.processing.SparkEmrEksJob", version: "1.0.0-rc.3" };
exports.SparkEmrEksJob = SparkEmrEksJob;
//# sourceMappingURL=data:application/json;base64,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