import { IRole } from 'aws-cdk-lib/aws-iam';
import { FailProps } from 'aws-cdk-lib/aws-stepfunctions';
import { CallAwsServiceProps } from 'aws-cdk-lib/aws-stepfunctions-tasks';
import { Construct } from 'constructs';
import { SparkJob } from './spark-job';
import { SparkEmrServerlessJobApiProps, SparkEmrServerlessJobProps } from './spark-job-emr-serverless-props';
/**
 * A construct to run Spark Jobs using EMR Serverless.
 * Creates a State Machine that orchestrates the Spark Job.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-job
 *
 * @example
 * import { PolicyDocument, PolicyStatement } from 'aws-cdk-lib/aws-iam';
 * import { JsonPath } from 'aws-cdk-lib/aws-stepfunctions';
 *
 * const myFileSystemPolicy = new PolicyDocument({
 *   statements: [new PolicyStatement({
 *     actions: [
 *       's3:GetObject',
 *     ],
 *     resources: ['*'],
 *   })],
 * });
 *
 *
 * const myExecutionRole = dsf.processing.SparkEmrServerlessRuntime.createExecutionRole(this, 'execRole1', myFileSystemPolicy);
 * const applicationId = "APPLICATION_ID";
 * const job = new dsf.processing.SparkEmrServerlessJob(this, 'SparkJob', {
 *   jobConfig:{
 *     "Name": JsonPath.format('ge_profile-{}', JsonPath.uuid()),
 *     "ApplicationId": applicationId,
 *     "ExecutionRoleArn": myExecutionRole.roleArn,
 *     "JobDriver": {
 *       "SparkSubmit": {
 *           "EntryPoint": "s3://S3-BUCKET/pi.py",
 *           "EntryPointArguments": [],
 *           "SparkSubmitParameters": "--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.driver.memory=2G --conf spark.executor.cores=4"
 *       },
 *     }
 *   }
 * } as dsf.processing.SparkEmrServerlessJobApiProps);
 *
 * new cdk.CfnOutput(this, 'SparkJobStateMachine', {
 *   value: job.stateMachine!.stateMachineArn,
 * });
 */
export declare class SparkEmrServerlessJob extends SparkJob {
    /**
     * Spark Job execution role. Use this property to add additional IAM permissions if necessary.
     */
    sparkJobExecutionRole?: IRole;
    private constructJobConfig;
    constructor(scope: Construct, id: string, props: SparkEmrServerlessJobProps | SparkEmrServerlessJobApiProps);
    /**
     * Returns the props for the Step Functions CallAwsService Construct that starts the Spark job, it calls the [StartJobRun API](https://docs.aws.amazon.com/emr-serverless/latest/APIReference/API_StartJobRun.html)
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    protected returnJobStartTaskProps(): CallAwsServiceProps;
    /**
     * Returns the props for the Step Functions CallAwsService Construct that checks the execution status of the Spark job, it calls the [GetJobRun API](https://docs.aws.amazon.com/emr-serverless/latest/APIReference/API_GetJobRun.html)
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    protected returnJobMonitorTaskProps(): CallAwsServiceProps;
    /**
     * Returns the props for the step function task that handles the failure if the EMR Serverless job fails.
     * @returns FailProps The error details of the failed Spark Job
     */
    protected returnJobFailTaskProps(): FailProps;
    /**
     * Returns the status of the EMR Serverless job that succeeded based on the GetJobRun API response
     * @returns string
     */
    protected returnJobStatusSucceed(): string;
    /**
     * Returns the status of the EMR Serverless job that failed based on the GetJobRun API response
     * @returns string
     */
    protected returnJobStatusFailed(): string;
    /**
     * Returns the status of the EMR Serverless job that is cancelled based on the GetJobRun API response
     * @returns string
     */
    protected returnJobStatusCancelled(): string;
    /**
     * Grants the necessary permissions to the Step Functions StateMachine to be able to start EMR Serverless job
     * @param role Step Functions StateMachine IAM role
     * @see SparkRuntimeServerless.grantJobExecution
     */
    protected grantExecutionRole(role: IRole): void;
    /**
     * Set defaults for the EmrServerlessSparkJobApiProps.
     * @param props EmrServerlessSparkJobApiProps
     */
    private setJobApiPropsDefaults;
    /**
     * Set defaults for the EmrOnEksSparkJobProps.
     * @param props EmrOnEksSparkJobProps
     */
    private setJobPropsDefaults;
}
