"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SparkEmrServerlessJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const spark_job_1 = require("./spark-job");
const utils_1 = require("../../../utils");
const spark_runtime_1 = require("../spark-runtime");
/**
 * A construct to run Spark Jobs using EMR Serverless.
 * Creates a State Machine that orchestrates the Spark Job.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-job
 *
 * @example
 * import { PolicyDocument, PolicyStatement } from 'aws-cdk-lib/aws-iam';
 * import { JsonPath } from 'aws-cdk-lib/aws-stepfunctions';
 *
 * const myFileSystemPolicy = new PolicyDocument({
 *   statements: [new PolicyStatement({
 *     actions: [
 *       's3:GetObject',
 *     ],
 *     resources: ['*'],
 *   })],
 * });
 *
 *
 * const myExecutionRole = dsf.processing.SparkEmrServerlessRuntime.createExecutionRole(this, 'execRole1', myFileSystemPolicy);
 * const applicationId = "APPLICATION_ID";
 * const job = new dsf.processing.SparkEmrServerlessJob(this, 'SparkJob', {
 *   jobConfig:{
 *     "Name": JsonPath.format('ge_profile-{}', JsonPath.uuid()),
 *     "ApplicationId": applicationId,
 *     "ExecutionRoleArn": myExecutionRole.roleArn,
 *     "JobDriver": {
 *       "SparkSubmit": {
 *           "EntryPoint": "s3://S3-BUCKET/pi.py",
 *           "EntryPointArguments": [],
 *           "SparkSubmitParameters": "--conf spark.executor.instances=2 --conf spark.executor.memory=2G --conf spark.driver.memory=2G --conf spark.executor.cores=4"
 *       },
 *     }
 *   }
 * } as dsf.processing.SparkEmrServerlessJobApiProps);
 *
 * new cdk.CfnOutput(this, 'SparkJobStateMachine', {
 *   value: job.stateMachine!.stateMachineArn,
 * });
 */
class SparkEmrServerlessJob extends spark_job_1.SparkJob {
    constructor(scope, id, props) {
        super(scope, id, SparkEmrServerlessJob.name, props);
        let sparkJobExecutionRole;
        if ('jobConfig' in props) {
            this.constructJobConfig = this.setJobApiPropsDefaults(props);
        }
        else {
            this.constructJobConfig = this.setJobPropsDefaults(props);
        }
        //Tag the AWs Step Functions State Machine
        if (!this.constructJobConfig.jobConfig.Tags) {
            this.constructJobConfig.jobConfig.Tags = {};
        }
        this.constructJobConfig.jobConfig.Tags[utils_1.TrackedConstruct.DSF_OWNED_TAG] = 'true';
        sparkJobExecutionRole = aws_iam_1.Role.fromRoleArn(this, `spakrJobRole-${id}`, this.constructJobConfig.jobConfig.ExecutionRoleArn);
        this.stateMachine = this.createStateMachine(aws_cdk_lib_1.Duration.minutes(5 + this.constructJobConfig.jobConfig.ExecutionTimeoutMinutes), this.constructJobConfig.schedule);
        this.s3LogBucket?.grantReadWrite(sparkJobExecutionRole);
        this.emrJobLogGroup?.grantRead(sparkJobExecutionRole);
        this.emrJobLogGroup?.grantWrite(sparkJobExecutionRole);
        if (this.emrJobLogGroup) {
            sparkJobExecutionRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
                actions: ['logs:DescribeLogGroups', 'logs:DescribeLogStreams'],
                resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:log-group::log-stream:*`],
            }));
        }
    }
    /**
     * Returns the props for the Step Functions CallAwsService Construct that starts the Spark job, it calls the [StartJobRun API](https://docs.aws.amazon.com/emr-serverless/latest/APIReference/API_StartJobRun.html)
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    returnJobStartTaskProps() {
        return {
            service: 'emrserverless',
            action: 'startJobRun',
            iamAction: 'emrserverless:StartJobRun',
            parameters: this.constructJobConfig.jobConfig,
            iamResources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:emr-serverless:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/applications/${this.constructJobConfig.jobConfig.ApplicationId}/jobruns/*`],
            resultSelector: {
                'JobRunId.$': '$.JobRunId',
            },
        };
    }
    /**
     * Returns the props for the Step Functions CallAwsService Construct that checks the execution status of the Spark job, it calls the [GetJobRun API](https://docs.aws.amazon.com/emr-serverless/latest/APIReference/API_GetJobRun.html)
     * @see CallAwsService @link[https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_stepfunctions_tasks.CallAwsService.html]
     * @returns CallAwsServiceProps
     */
    returnJobMonitorTaskProps() {
        return {
            service: 'emrserverless',
            action: 'getJobRun',
            iamAction: 'emrserverless:GetJobRun',
            parameters: {
                ApplicationId: this.constructJobConfig.jobConfig.ApplicationId,
                JobRunId: aws_stepfunctions_1.JsonPath.stringAt('$.JobRunId'),
            },
            iamResources: [`arn:${aws_cdk_lib_1.Aws.PARTITION}:emr-serverless:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/applications/${this.constructJobConfig.jobConfig.ApplicationId}/jobruns/*`],
            resultSelector: {
                'State.$': '$.JobRun.State',
                'StateDetails.$': '$.JobRun.StateDetails',
            },
            resultPath: '$.JobRunState',
        };
    }
    /**
     * Returns the props for the step function task that handles the failure if the EMR Serverless job fails.
     * @returns FailProps The error details of the failed Spark Job
     */
    returnJobFailTaskProps() {
        return {
            cause: 'EMRServerlessJobFailed',
            error: aws_stepfunctions_1.JsonPath.stringAt('$.JobRunState.StateDetails'),
        };
    }
    /**
     * Returns the status of the EMR Serverless job that succeeded based on the GetJobRun API response
     * @returns string
     */
    returnJobStatusSucceed() {
        return 'SUCCESS';
    }
    /**
     * Returns the status of the EMR Serverless job that failed based on the GetJobRun API response
     * @returns string
     */
    returnJobStatusFailed() {
        return 'FAILED';
    }
    /**
     * Returns the status of the EMR Serverless job that is cancelled based on the GetJobRun API response
     * @returns string
     */
    returnJobStatusCancelled() {
        return 'CANCELLED';
    }
    /**
     * Grants the necessary permissions to the Step Functions StateMachine to be able to start EMR Serverless job
     * @param role Step Functions StateMachine IAM role
     * @see SparkRuntimeServerless.grantJobExecution
     */
    grantExecutionRole(role) {
        const arn = `arn:aws:emr-serverless:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:/applications/${this.constructJobConfig.jobConfig.ApplicationId}`;
        spark_runtime_1.SparkEmrServerlessRuntime.grantStartJobExecution(role, [this.constructJobConfig.jobConfig.ExecutionRoleArn], [arn, `${arn}/jobruns/*`]);
    }
    /**
     * Set defaults for the EmrServerlessSparkJobApiProps.
     * @param props EmrServerlessSparkJobApiProps
     */
    setJobApiPropsDefaults(props) {
        const propsPascalCase = utils_1.StepFunctionUtils.camelToPascal(props.jobConfig);
        //Set defaults
        propsPascalCase.ClientToken ?? (propsPascalCase.ClientToken = aws_stepfunctions_1.JsonPath.uuid());
        propsPascalCase.ExecutionTimeoutMinutes ?? (propsPascalCase.ExecutionTimeoutMinutes = 30);
        let config = {
            jobConfig: propsPascalCase,
            removalPolicy: props.removalPolicy,
            schedule: props.schedule,
        };
        return config;
    }
    /**
     * Set defaults for the EmrOnEksSparkJobProps.
     * @param props EmrOnEksSparkJobProps
     */
    setJobPropsDefaults(props) {
        const config = {
            jobConfig: {
                ConfigurationOverrides: {
                    MonitoringConfiguration: {
                        S3MonitoringConfiguration: {},
                    },
                },
                JobDriver: {
                    SparkSubmit: {},
                },
            },
            removalPolicy: props.removalPolicy,
            schedule: props.schedule,
        };
        config.jobConfig.Name = props.name;
        config.jobConfig.ClientToken = aws_stepfunctions_1.JsonPath.uuid();
        config.jobConfig.ExecutionTimeoutMinutes = props.executionTimeoutMinutes ?? 30;
        config.jobConfig.ExecutionRoleArn = props.executionRoleArn;
        config.jobConfig.ApplicationId = props.applicationId;
        config.jobConfig.JobDriver.SparkSubmit.EntryPoint = props.sparkSubmitEntryPoint;
        if (props.sparkSubmitEntryPointArguments) {
            config.jobConfig.JobDriver.SparkSubmit.EntryPointArguments = props.sparkSubmitEntryPointArguments;
        }
        if (props.sparkSubmitParameters) {
            config.jobConfig.JobDriver.SparkSubmit.SparkSubmitParameters = props.sparkSubmitParameters;
        }
        if (props.applicationConfiguration) {
            config.jobConfig.ConfigurationOverrides.ApplicationConfiguration = utils_1.StepFunctionUtils.camelToPascal(props.applicationConfiguration);
        }
        if (props.s3LogUri && !props.s3LogUri.match(/^s3:\/\/([^\/]+)/)) {
            throw new Error(`Invalid S3 URI: ${props.s3LogUri}`);
        }
        config.jobConfig.ConfigurationOverrides.MonitoringConfiguration.S3MonitoringConfiguration.LogUri =
            this.createS3LogBucket(props.s3LogUri, props.s3LogUriKeyArn);
        if (props.s3LogUriKeyArn) {
            config.jobConfig.ConfigurationOverrides.MonitoringConfiguration.S3MonitoringConfiguration.EncryptionKeyArn = props.s3LogUriKeyArn;
        }
        if (props.cloudWatchLogGroupName) {
            this.createCloudWatchLogsLogGroup(props.cloudWatchLogGroupName, props.cloudWatchEncryptionKeyArn);
            config.jobConfig.ConfigurationOverrides.MonitoringConfiguration.CloudWatchLoggingConfiguration = {
                Enabled: true,
                EncryptionKeyArn: props.cloudWatchEncryptionKeyArn,
                LogGroupName: props.cloudWatchLogGroupName ?? props.name,
                LogStreamNamePrefix: props.cloudWatchLogGroupStreamPrefix,
            };
        }
        config.jobConfig.Tags = props.tags;
        return config;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SparkEmrServerlessJob[_a] = { fqn: "aws-dsf.processing.SparkEmrServerlessJob", version: "1.0.0-rc.3" };
exports.SparkEmrServerlessJob = SparkEmrServerlessJob;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3Bhcmstam9iLWVtci1zZXJ2ZXJsZXNzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vLi4vc3JjL3Byb2Nlc3NpbmcvbGliL3NwYXJrLWpvYi9zcGFyay1qb2ItZW1yLXNlcnZlcmxlc3MudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQSxxRUFBcUU7QUFDckUsaUNBQWlDO0FBRWpDLDZDQUE0QztBQUM1QyxpREFBbUU7QUFDbkUscUVBQW9FO0FBR3BFLDJDQUF1QztBQUd2QywwQ0FBcUU7QUFDckUsb0RBQTZEO0FBRTdEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7R0F1Q0c7QUFDSCxNQUFhLHFCQUFzQixTQUFRLG9CQUFRO0lBUWpELFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBaUU7UUFDekcsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUscUJBQXFCLENBQUMsSUFBSSxFQUFFLEtBQXNCLENBQUMsQ0FBQztRQUVyRSxJQUFJLHFCQUE0QixDQUFDO1FBRWpDLElBQUksV0FBVyxJQUFJLEtBQUssRUFBRTtZQUN4QixJQUFJLENBQUMsa0JBQWtCLEdBQUcsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEtBQXNDLENBQUMsQ0FBQztTQUMvRjthQUFNO1lBQ0wsSUFBSSxDQUFDLGtCQUFrQixHQUFHLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFtQyxDQUFDLENBQUM7U0FDekY7UUFDRCwwQ0FBMEM7UUFDMUMsSUFBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFO1lBQzNDLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsSUFBSSxHQUFHLEVBQUUsQ0FBQztTQUM3QztRQUVELElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLHdCQUFnQixDQUFDLGFBQWEsQ0FBQyxHQUFHLE1BQU0sQ0FBQztRQUVoRixxQkFBcUIsR0FBRyxjQUFJLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxFQUFFLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBR3pILElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUN6QyxzQkFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyx1QkFBd0IsQ0FBQyxFQUNoRixJQUFJLENBQUMsa0JBQWtCLENBQUMsUUFBUSxDQUFDLENBQUM7UUFFcEMsSUFBSSxDQUFDLFdBQVcsRUFBRSxjQUFjLENBQUMscUJBQXFCLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsY0FBYyxFQUFFLFNBQVMsQ0FBQyxxQkFBcUIsQ0FBQyxDQUFDO1FBQ3RELElBQUksQ0FBQyxjQUFjLEVBQUUsVUFBVSxDQUFDLHFCQUFxQixDQUFDLENBQUM7UUFFdkQsSUFBSSxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3ZCLHFCQUFxQixDQUFDLG9CQUFvQixDQUFDLElBQUkseUJBQWUsQ0FBQztnQkFDN0QsT0FBTyxFQUFFLENBQUMsd0JBQXdCLEVBQUUseUJBQXlCLENBQUM7Z0JBQzlELFNBQVMsRUFBRSxDQUFDLGdCQUFnQixpQkFBRyxDQUFDLE1BQU0sSUFBSSxpQkFBRyxDQUFDLFVBQVUsMEJBQTBCLENBQUM7YUFDcEYsQ0FBQyxDQUFDLENBQUM7U0FDTDtJQUNILENBQUM7SUFHRDs7OztPQUlHO0lBQ08sdUJBQXVCO1FBQy9CLE9BQU87WUFDTCxPQUFPLEVBQUUsZUFBZTtZQUN4QixNQUFNLEVBQUUsYUFBYTtZQUNyQixTQUFTLEVBQUUsMkJBQTJCO1lBQ3RDLFVBQVUsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUztZQUM3QyxZQUFZLEVBQUUsQ0FBQyxPQUFPLGlCQUFHLENBQUMsU0FBUyxtQkFBbUIsaUJBQUcsQ0FBQyxNQUFNLElBQUksaUJBQUcsQ0FBQyxVQUFVLGtCQUFrQixJQUFJLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLGFBQWEsWUFBWSxDQUFDO1lBQ2hLLGNBQWMsRUFBRTtnQkFDZCxZQUFZLEVBQUUsWUFBWTthQUMzQjtTQUNxQixDQUFDO0lBQzNCLENBQUM7SUFFRDs7OztPQUlHO0lBQ08seUJBQXlCO1FBQ2pDLE9BQU87WUFDTCxPQUFPLEVBQUUsZUFBZTtZQUN4QixNQUFNLEVBQUUsV0FBVztZQUNuQixTQUFTLEVBQUUseUJBQXlCO1lBQ3BDLFVBQVUsRUFBRTtnQkFDVixhQUFhLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxhQUFhO2dCQUM5RCxRQUFRLEVBQUUsNEJBQVEsQ0FBQyxRQUFRLENBQUMsWUFBWSxDQUFDO2FBQzFDO1lBQ0QsWUFBWSxFQUFFLENBQUMsT0FBTyxpQkFBRyxDQUFDLFNBQVMsbUJBQW1CLGlCQUFHLENBQUMsTUFBTSxJQUFJLGlCQUFHLENBQUMsVUFBVSxrQkFBa0IsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxhQUFhLFlBQVksQ0FBQztZQUNoSyxjQUFjLEVBQUU7Z0JBQ2QsU0FBUyxFQUFFLGdCQUFnQjtnQkFDM0IsZ0JBQWdCLEVBQUUsdUJBQXVCO2FBQzFDO1lBQ0QsVUFBVSxFQUFFLGVBQWU7U0FDTCxDQUFDO0lBQzNCLENBQUM7SUFFRDs7O09BR0c7SUFDTyxzQkFBc0I7UUFDOUIsT0FBTztZQUNMLEtBQUssRUFBRSx3QkFBd0I7WUFDL0IsS0FBSyxFQUFFLDRCQUFRLENBQUMsUUFBUSxDQUFDLDRCQUE0QixDQUFDO1NBQzFDLENBQUM7SUFDakIsQ0FBQztJQUdEOzs7T0FHRztJQUNPLHNCQUFzQjtRQUM5QixPQUFPLFNBQVMsQ0FBQztJQUNuQixDQUFDO0lBRUQ7OztPQUdHO0lBQ08scUJBQXFCO1FBQzdCLE9BQU8sUUFBUSxDQUFDO0lBQ2xCLENBQUM7SUFFRDs7O09BR0c7SUFDTyx3QkFBd0I7UUFDaEMsT0FBTyxXQUFXLENBQUM7SUFDckIsQ0FBQztJQUVEOzs7O09BSUc7SUFFTyxrQkFBa0IsQ0FBQyxJQUFXO1FBRXRDLE1BQU0sR0FBRyxHQUFHLDBCQUEwQixpQkFBRyxDQUFDLE1BQU0sSUFBSSxpQkFBRyxDQUFDLFVBQVUsa0JBQWtCLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxTQUFTLENBQUMsYUFBYSxFQUFFLENBQUM7UUFFdEkseUNBQXlCLENBQUMsc0JBQXNCLENBQUMsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLENBQUMsR0FBRyxFQUFFLEdBQUcsR0FBRyxZQUFZLENBQUMsQ0FBQyxDQUFDO0lBQzFJLENBQUM7SUFFRDs7O09BR0c7SUFDSyxzQkFBc0IsQ0FBQyxLQUFvQztRQUVqRSxNQUFNLGVBQWUsR0FBRyx5QkFBaUIsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3pFLGNBQWM7UUFDZCxlQUFlLENBQUMsV0FBVyxLQUEzQixlQUFlLENBQUMsV0FBVyxHQUFLLDRCQUFRLENBQUMsSUFBSSxFQUFFLEVBQUM7UUFDaEQsZUFBZSxDQUFDLHVCQUF1QixLQUF2QyxlQUFlLENBQUMsdUJBQXVCLEdBQUssRUFBRSxFQUFDO1FBRS9DLElBQUksTUFBTSxHQUFHO1lBQ1gsU0FBUyxFQUFFLGVBQWU7WUFDMUIsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtTQUN6QixDQUFDO1FBRUYsT0FBTyxNQUFNLENBQUM7SUFDaEIsQ0FBQztJQUVEOzs7T0FHRztJQUNLLG1CQUFtQixDQUFDLEtBQWlDO1FBRTNELE1BQU0sTUFBTSxHQUFHO1lBQ2IsU0FBUyxFQUFFO2dCQUNULHNCQUFzQixFQUFFO29CQUN0Qix1QkFBdUIsRUFBRTt3QkFDdkIseUJBQXlCLEVBQUUsRUFBRTtxQkFDOUI7aUJBQ0Y7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULFdBQVcsRUFBRSxFQUFFO2lCQUNoQjthQUNGO1lBQ0QsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtTQUNRLENBQUM7UUFFbkMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQztRQUNuQyxNQUFNLENBQUMsU0FBUyxDQUFDLFdBQVcsR0FBRyw0QkFBUSxDQUFDLElBQUksRUFBRSxDQUFDO1FBQy9DLE1BQU0sQ0FBQyxTQUFTLENBQUMsdUJBQXVCLEdBQUcsS0FBSyxDQUFDLHVCQUF1QixJQUFJLEVBQUUsQ0FBQztRQUMvRSxNQUFNLENBQUMsU0FBUyxDQUFDLGdCQUFnQixHQUFDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQztRQUN6RCxNQUFNLENBQUMsU0FBUyxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQ3JELE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1FBRWhGLElBQUksS0FBSyxDQUFDLDhCQUE4QixFQUFFO1lBQ3hDLE1BQU0sQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsOEJBQThCLENBQUM7U0FDbkc7UUFDRCxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsRUFBRTtZQUMvQixNQUFNLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMscUJBQXFCLEdBQUcsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1NBQzVGO1FBRUQsSUFBSSxLQUFLLENBQUMsd0JBQXdCLEVBQUU7WUFDbEMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyx3QkFBd0IsR0FBRyx5QkFBaUIsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDLENBQUM7U0FDcEk7UUFFRCxJQUFJLEtBQUssQ0FBQyxRQUFRLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1lBQy9ELE1BQU0sSUFBSSxLQUFLLENBQUMsbUJBQW1CLEtBQUssQ0FBQyxRQUFRLEVBQUUsQ0FBQyxDQUFDO1NBQ3REO1FBRUQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyx1QkFBdUIsQ0FBQyx5QkFBMEIsQ0FBQyxNQUFNO1lBQ2pHLElBQUksQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUU3RCxJQUFLLEtBQUssQ0FBQyxjQUFjLEVBQUc7WUFDMUIsTUFBTSxDQUFDLFNBQVMsQ0FBQyxzQkFBc0IsQ0FBQyx1QkFBdUIsQ0FBQyx5QkFBMEIsQ0FBQyxnQkFBZ0IsR0FBRyxLQUFLLENBQUMsY0FBYyxDQUFDO1NBQ3BJO1FBR0QsSUFBSSxLQUFLLENBQUMsc0JBQXNCLEVBQUU7WUFDaEMsSUFBSSxDQUFDLDRCQUE0QixDQUFDLEtBQUssQ0FBQyxzQkFBc0IsRUFBRSxLQUFLLENBQUMsMEJBQTBCLENBQUMsQ0FBQztZQUNsRyxNQUFNLENBQUMsU0FBUyxDQUFDLHNCQUFzQixDQUFDLHVCQUF1QixDQUFDLDhCQUE4QixHQUFHO2dCQUMvRixPQUFPLEVBQUUsSUFBSTtnQkFDYixnQkFBZ0IsRUFBRSxLQUFLLENBQUMsMEJBQTBCO2dCQUNsRCxZQUFZLEVBQUUsS0FBSyxDQUFDLHNCQUFzQixJQUFJLEtBQUssQ0FBQyxJQUFJO2dCQUN4RCxtQkFBbUIsRUFBRSxLQUFLLENBQUMsOEJBQThCO2FBQzFELENBQUM7U0FDSDtRQUVELE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUM7UUFFbkMsT0FBTyxNQUFNLENBQUM7SUFFaEIsQ0FBQzs7OztBQTVOVSxzREFBcUIiLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbi8vIFNQRFgtTGljZW5zZS1JZGVudGlmaWVyOiBNSVQtMFxuXG5pbXBvcnQgeyBBd3MsIER1cmF0aW9uIH0gZnJvbSAnYXdzLWNkay1saWInO1xuaW1wb3J0IHsgUG9saWN5U3RhdGVtZW50LCBSb2xlLCBJUm9sZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1pYW0nO1xuaW1wb3J0IHsgRmFpbFByb3BzLCBKc29uUGF0aCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCB7IENhbGxBd3NTZXJ2aWNlUHJvcHMgfSBmcm9tICdhd3MtY2RrLWxpYi9hd3Mtc3RlcGZ1bmN0aW9ucy10YXNrcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcbmltcG9ydCB7IFNwYXJrSm9iIH0gZnJvbSAnLi9zcGFyay1qb2InO1xuaW1wb3J0IHsgU3BhcmtFbXJTZXJ2ZXJsZXNzSm9iQXBpUHJvcHMsIFNwYXJrRW1yU2VydmVybGVzc0pvYlByb3BzIH0gZnJvbSAnLi9zcGFyay1qb2ItZW1yLXNlcnZlcmxlc3MtcHJvcHMnO1xuaW1wb3J0IHsgU3BhcmtKb2JQcm9wcyB9IGZyb20gJy4vc3Bhcmstam9iLXByb3BzJztcbmltcG9ydCB7IFN0ZXBGdW5jdGlvblV0aWxzLCBUcmFja2VkQ29uc3RydWN0IH0gZnJvbSAnLi4vLi4vLi4vdXRpbHMnO1xuaW1wb3J0IHsgU3BhcmtFbXJTZXJ2ZXJsZXNzUnVudGltZSB9IGZyb20gJy4uL3NwYXJrLXJ1bnRpbWUnO1xuXG4vKipcbiAqIEEgY29uc3RydWN0IHRvIHJ1biBTcGFyayBKb2JzIHVzaW5nIEVNUiBTZXJ2ZXJsZXNzLlxuICogQ3JlYXRlcyBhIFN0YXRlIE1hY2hpbmUgdGhhdCBvcmNoZXN0cmF0ZXMgdGhlIFNwYXJrIEpvYi5cbiAqIEBzZWUgaHR0cHM6Ly9hd3NsYWJzLmdpdGh1Yi5pby9kYXRhLXNvbHV0aW9ucy1mcmFtZXdvcmstb24tYXdzL2RvY3MvY29uc3RydWN0cy9saWJyYXJ5L3NwYXJrLWpvYlxuICpcbiAqIEBleGFtcGxlXG4gKiBpbXBvcnQgeyBQb2xpY3lEb2N1bWVudCwgUG9saWN5U3RhdGVtZW50IH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG4gKiBpbXBvcnQgeyBKc29uUGF0aCB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zJztcbiAqXG4gKiBjb25zdCBteUZpbGVTeXN0ZW1Qb2xpY3kgPSBuZXcgUG9saWN5RG9jdW1lbnQoe1xuICogICBzdGF0ZW1lbnRzOiBbbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gKiAgICAgYWN0aW9uczogW1xuICogICAgICAgJ3MzOkdldE9iamVjdCcsXG4gKiAgICAgXSxcbiAqICAgICByZXNvdXJjZXM6IFsnKiddLFxuICogICB9KV0sXG4gKiB9KTtcbiAqXG4gKlxuICogY29uc3QgbXlFeGVjdXRpb25Sb2xlID0gZHNmLnByb2Nlc3NpbmcuU3BhcmtFbXJTZXJ2ZXJsZXNzUnVudGltZS5jcmVhdGVFeGVjdXRpb25Sb2xlKHRoaXMsICdleGVjUm9sZTEnLCBteUZpbGVTeXN0ZW1Qb2xpY3kpO1xuICogY29uc3QgYXBwbGljYXRpb25JZCA9IFwiQVBQTElDQVRJT05fSURcIjtcbiAqIGNvbnN0IGpvYiA9IG5ldyBkc2YucHJvY2Vzc2luZy5TcGFya0VtclNlcnZlcmxlc3NKb2IodGhpcywgJ1NwYXJrSm9iJywge1xuICogICBqb2JDb25maWc6e1xuICogICAgIFwiTmFtZVwiOiBKc29uUGF0aC5mb3JtYXQoJ2dlX3Byb2ZpbGUte30nLCBKc29uUGF0aC51dWlkKCkpLFxuICogICAgIFwiQXBwbGljYXRpb25JZFwiOiBhcHBsaWNhdGlvbklkLFxuICogICAgIFwiRXhlY3V0aW9uUm9sZUFyblwiOiBteUV4ZWN1dGlvblJvbGUucm9sZUFybixcbiAqICAgICBcIkpvYkRyaXZlclwiOiB7XG4gKiAgICAgICBcIlNwYXJrU3VibWl0XCI6IHtcbiAqICAgICAgICAgICBcIkVudHJ5UG9pbnRcIjogXCJzMzovL1MzLUJVQ0tFVC9waS5weVwiLFxuICogICAgICAgICAgIFwiRW50cnlQb2ludEFyZ3VtZW50c1wiOiBbXSxcbiAqICAgICAgICAgICBcIlNwYXJrU3VibWl0UGFyYW1ldGVyc1wiOiBcIi0tY29uZiBzcGFyay5leGVjdXRvci5pbnN0YW5jZXM9MiAtLWNvbmYgc3BhcmsuZXhlY3V0b3IubWVtb3J5PTJHIC0tY29uZiBzcGFyay5kcml2ZXIubWVtb3J5PTJHIC0tY29uZiBzcGFyay5leGVjdXRvci5jb3Jlcz00XCJcbiAqICAgICAgIH0sXG4gKiAgICAgfVxuICogICB9XG4gKiB9IGFzIGRzZi5wcm9jZXNzaW5nLlNwYXJrRW1yU2VydmVybGVzc0pvYkFwaVByb3BzKTtcbiAqXG4gKiBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnU3BhcmtKb2JTdGF0ZU1hY2hpbmUnLCB7XG4gKiAgIHZhbHVlOiBqb2Iuc3RhdGVNYWNoaW5lIS5zdGF0ZU1hY2hpbmVBcm4sXG4gKiB9KTtcbiAqL1xuZXhwb3J0IGNsYXNzIFNwYXJrRW1yU2VydmVybGVzc0pvYiBleHRlbmRzIFNwYXJrSm9iIHtcblxuICAvKipcbiAgICogU3BhcmsgSm9iIGV4ZWN1dGlvbiByb2xlLiBVc2UgdGhpcyBwcm9wZXJ0eSB0byBhZGQgYWRkaXRpb25hbCBJQU0gcGVybWlzc2lvbnMgaWYgbmVjZXNzYXJ5LlxuICAgKi9cbiAgcHVibGljIHNwYXJrSm9iRXhlY3V0aW9uUm9sZT86IElSb2xlO1xuICBwcml2YXRlIGNvbnN0cnVjdEpvYkNvbmZpZzogU3BhcmtFbXJTZXJ2ZXJsZXNzSm9iQXBpUHJvcHM7XG5cbiAgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFNwYXJrRW1yU2VydmVybGVzc0pvYlByb3BzIHwgU3BhcmtFbXJTZXJ2ZXJsZXNzSm9iQXBpUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQsIFNwYXJrRW1yU2VydmVybGVzc0pvYi5uYW1lLCBwcm9wcyBhcyBTcGFya0pvYlByb3BzKTtcblxuICAgIGxldCBzcGFya0pvYkV4ZWN1dGlvblJvbGU6IElSb2xlO1xuXG4gICAgaWYgKCdqb2JDb25maWcnIGluIHByb3BzKSB7XG4gICAgICB0aGlzLmNvbnN0cnVjdEpvYkNvbmZpZyA9IHRoaXMuc2V0Sm9iQXBpUHJvcHNEZWZhdWx0cyhwcm9wcyBhcyBTcGFya0VtclNlcnZlcmxlc3NKb2JBcGlQcm9wcyk7XG4gICAgfSBlbHNlIHtcbiAgICAgIHRoaXMuY29uc3RydWN0Sm9iQ29uZmlnID0gdGhpcy5zZXRKb2JQcm9wc0RlZmF1bHRzKHByb3BzIGFzIFNwYXJrRW1yU2VydmVybGVzc0pvYlByb3BzKTtcbiAgICB9XG4gICAgLy9UYWcgdGhlIEFXcyBTdGVwIEZ1bmN0aW9ucyBTdGF0ZSBNYWNoaW5lXG4gICAgaWYgKCF0aGlzLmNvbnN0cnVjdEpvYkNvbmZpZy5qb2JDb25maWcuVGFncykge1xuICAgICAgdGhpcy5jb25zdHJ1Y3RKb2JDb25maWcuam9iQ29uZmlnLlRhZ3MgPSB7fTtcbiAgICB9XG5cbiAgICB0aGlzLmNvbnN0cnVjdEpvYkNvbmZpZy5qb2JDb25maWcuVGFnc1tUcmFja2VkQ29uc3RydWN0LkRTRl9PV05FRF9UQUddID0gJ3RydWUnO1xuXG4gICAgc3BhcmtKb2JFeGVjdXRpb25Sb2xlID0gUm9sZS5mcm9tUm9sZUFybih0aGlzLCBgc3Bha3JKb2JSb2xlLSR7aWR9YCwgdGhpcy5jb25zdHJ1Y3RKb2JDb25maWcuam9iQ29uZmlnLkV4ZWN1dGlvblJvbGVBcm4pO1xuXG5cbiAgICB0aGlzLnN0YXRlTWFjaGluZSA9IHRoaXMuY3JlYXRlU3RhdGVNYWNoaW5lKFxuICAgICAgRHVyYXRpb24ubWludXRlcyg1ICsgdGhpcy5jb25zdHJ1Y3RKb2JDb25maWcuam9iQ29uZmlnLkV4ZWN1dGlvblRpbWVvdXRNaW51dGVzISksXG4gICAgICB0aGlzLmNvbnN0cnVjdEpvYkNvbmZpZy5zY2hlZHVsZSk7XG5cbiAgICB0aGlzLnMzTG9nQnVja2V0Py5ncmFudFJlYWRXcml0ZShzcGFya0pvYkV4ZWN1dGlvblJvbGUpO1xuICAgIHRoaXMuZW1ySm9iTG9nR3JvdXA/LmdyYW50UmVhZChzcGFya0pvYkV4ZWN1dGlvblJvbGUpO1xuICAgIHRoaXMuZW1ySm9iTG9nR3JvdXA/LmdyYW50V3JpdGUoc3BhcmtKb2JFeGVjdXRpb25Sb2xlKTtcblxuICAgIGlmICh0aGlzLmVtckpvYkxvZ0dyb3VwKSB7XG4gICAgICBzcGFya0pvYkV4ZWN1dGlvblJvbGUuYWRkVG9QcmluY2lwYWxQb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFsnbG9nczpEZXNjcmliZUxvZ0dyb3VwcycsICdsb2dzOkRlc2NyaWJlTG9nU3RyZWFtcyddLFxuICAgICAgICByZXNvdXJjZXM6IFtgYXJuOmF3czpsb2dzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06bG9nLWdyb3VwOjpsb2ctc3RyZWFtOipgXSxcbiAgICAgIH0pKTtcbiAgICB9XG4gIH1cblxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBwcm9wcyBmb3IgdGhlIFN0ZXAgRnVuY3Rpb25zIENhbGxBd3NTZXJ2aWNlIENvbnN0cnVjdCB0aGF0IHN0YXJ0cyB0aGUgU3Bhcmsgam9iLCBpdCBjYWxscyB0aGUgW1N0YXJ0Sm9iUnVuIEFQSV0oaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Vtci1zZXJ2ZXJsZXNzL2xhdGVzdC9BUElSZWZlcmVuY2UvQVBJX1N0YXJ0Sm9iUnVuLmh0bWwpXG4gICAqIEBzZWUgQ2FsbEF3c1NlcnZpY2UgQGxpbmtbaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nkay9hcGkvdjIvZG9jcy9hd3MtY2RrLWxpYi5hd3Nfc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZS5odG1sXVxuICAgKiBAcmV0dXJucyBDYWxsQXdzU2VydmljZVByb3BzXG4gICAqL1xuICBwcm90ZWN0ZWQgcmV0dXJuSm9iU3RhcnRUYXNrUHJvcHMoKTogQ2FsbEF3c1NlcnZpY2VQcm9wcyB7XG4gICAgcmV0dXJuIHtcbiAgICAgIHNlcnZpY2U6ICdlbXJzZXJ2ZXJsZXNzJyxcbiAgICAgIGFjdGlvbjogJ3N0YXJ0Sm9iUnVuJyxcbiAgICAgIGlhbUFjdGlvbjogJ2VtcnNlcnZlcmxlc3M6U3RhcnRKb2JSdW4nLFxuICAgICAgcGFyYW1ldGVyczogdGhpcy5jb25zdHJ1Y3RKb2JDb25maWcuam9iQ29uZmlnLFxuICAgICAgaWFtUmVzb3VyY2VzOiBbYGFybjoke0F3cy5QQVJUSVRJT059OmVtci1zZXJ2ZXJsZXNzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06L2FwcGxpY2F0aW9ucy8ke3RoaXMuY29uc3RydWN0Sm9iQ29uZmlnLmpvYkNvbmZpZy5BcHBsaWNhdGlvbklkfS9qb2JydW5zLypgXSxcbiAgICAgIHJlc3VsdFNlbGVjdG9yOiB7XG4gICAgICAgICdKb2JSdW5JZC4kJzogJyQuSm9iUnVuSWQnLFxuICAgICAgfSxcbiAgICB9IGFzIENhbGxBd3NTZXJ2aWNlUHJvcHM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgcHJvcHMgZm9yIHRoZSBTdGVwIEZ1bmN0aW9ucyBDYWxsQXdzU2VydmljZSBDb25zdHJ1Y3QgdGhhdCBjaGVja3MgdGhlIGV4ZWN1dGlvbiBzdGF0dXMgb2YgdGhlIFNwYXJrIGpvYiwgaXQgY2FsbHMgdGhlIFtHZXRKb2JSdW4gQVBJXShodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZW1yLXNlcnZlcmxlc3MvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0Sm9iUnVuLmh0bWwpXG4gICAqIEBzZWUgQ2FsbEF3c1NlcnZpY2UgQGxpbmtbaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2Nkay9hcGkvdjIvZG9jcy9hd3MtY2RrLWxpYi5hd3Nfc3RlcGZ1bmN0aW9uc190YXNrcy5DYWxsQXdzU2VydmljZS5odG1sXVxuICAgKiBAcmV0dXJucyBDYWxsQXdzU2VydmljZVByb3BzXG4gICAqL1xuICBwcm90ZWN0ZWQgcmV0dXJuSm9iTW9uaXRvclRhc2tQcm9wcygpOiBDYWxsQXdzU2VydmljZVByb3BzIHtcbiAgICByZXR1cm4ge1xuICAgICAgc2VydmljZTogJ2VtcnNlcnZlcmxlc3MnLFxuICAgICAgYWN0aW9uOiAnZ2V0Sm9iUnVuJyxcbiAgICAgIGlhbUFjdGlvbjogJ2VtcnNlcnZlcmxlc3M6R2V0Sm9iUnVuJyxcbiAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgQXBwbGljYXRpb25JZDogdGhpcy5jb25zdHJ1Y3RKb2JDb25maWcuam9iQ29uZmlnLkFwcGxpY2F0aW9uSWQsXG4gICAgICAgIEpvYlJ1bklkOiBKc29uUGF0aC5zdHJpbmdBdCgnJC5Kb2JSdW5JZCcpLFxuICAgICAgfSxcbiAgICAgIGlhbVJlc291cmNlczogW2Bhcm46JHtBd3MuUEFSVElUSU9OfTplbXItc2VydmVybGVzczoke0F3cy5SRUdJT059OiR7QXdzLkFDQ09VTlRfSUR9Oi9hcHBsaWNhdGlvbnMvJHt0aGlzLmNvbnN0cnVjdEpvYkNvbmZpZy5qb2JDb25maWcuQXBwbGljYXRpb25JZH0vam9icnVucy8qYF0sXG4gICAgICByZXN1bHRTZWxlY3Rvcjoge1xuICAgICAgICAnU3RhdGUuJCc6ICckLkpvYlJ1bi5TdGF0ZScsXG4gICAgICAgICdTdGF0ZURldGFpbHMuJCc6ICckLkpvYlJ1bi5TdGF0ZURldGFpbHMnLFxuICAgICAgfSxcbiAgICAgIHJlc3VsdFBhdGg6ICckLkpvYlJ1blN0YXRlJyxcbiAgICB9IGFzIENhbGxBd3NTZXJ2aWNlUHJvcHM7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJucyB0aGUgcHJvcHMgZm9yIHRoZSBzdGVwIGZ1bmN0aW9uIHRhc2sgdGhhdCBoYW5kbGVzIHRoZSBmYWlsdXJlIGlmIHRoZSBFTVIgU2VydmVybGVzcyBqb2IgZmFpbHMuXG4gICAqIEByZXR1cm5zIEZhaWxQcm9wcyBUaGUgZXJyb3IgZGV0YWlscyBvZiB0aGUgZmFpbGVkIFNwYXJrIEpvYlxuICAgKi9cbiAgcHJvdGVjdGVkIHJldHVybkpvYkZhaWxUYXNrUHJvcHMoKTogRmFpbFByb3BzIHtcbiAgICByZXR1cm4ge1xuICAgICAgY2F1c2U6ICdFTVJTZXJ2ZXJsZXNzSm9iRmFpbGVkJyxcbiAgICAgIGVycm9yOiBKc29uUGF0aC5zdHJpbmdBdCgnJC5Kb2JSdW5TdGF0ZS5TdGF0ZURldGFpbHMnKSxcbiAgICB9IGFzIEZhaWxQcm9wcztcbiAgfVxuXG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHN0YXR1cyBvZiB0aGUgRU1SIFNlcnZlcmxlc3Mgam9iIHRoYXQgc3VjY2VlZGVkIGJhc2VkIG9uIHRoZSBHZXRKb2JSdW4gQVBJIHJlc3BvbnNlXG4gICAqIEByZXR1cm5zIHN0cmluZ1xuICAgKi9cbiAgcHJvdGVjdGVkIHJldHVybkpvYlN0YXR1c1N1Y2NlZWQoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gJ1NVQ0NFU1MnO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgdGhlIHN0YXR1cyBvZiB0aGUgRU1SIFNlcnZlcmxlc3Mgam9iIHRoYXQgZmFpbGVkIGJhc2VkIG9uIHRoZSBHZXRKb2JSdW4gQVBJIHJlc3BvbnNlXG4gICAqIEByZXR1cm5zIHN0cmluZ1xuICAgKi9cbiAgcHJvdGVjdGVkIHJldHVybkpvYlN0YXR1c0ZhaWxlZCgpOiBzdHJpbmcge1xuICAgIHJldHVybiAnRkFJTEVEJztcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBzdGF0dXMgb2YgdGhlIEVNUiBTZXJ2ZXJsZXNzIGpvYiB0aGF0IGlzIGNhbmNlbGxlZCBiYXNlZCBvbiB0aGUgR2V0Sm9iUnVuIEFQSSByZXNwb25zZVxuICAgKiBAcmV0dXJucyBzdHJpbmdcbiAgICovXG4gIHByb3RlY3RlZCByZXR1cm5Kb2JTdGF0dXNDYW5jZWxsZWQoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gJ0NBTkNFTExFRCc7XG4gIH1cblxuICAvKipcbiAgICogR3JhbnRzIHRoZSBuZWNlc3NhcnkgcGVybWlzc2lvbnMgdG8gdGhlIFN0ZXAgRnVuY3Rpb25zIFN0YXRlTWFjaGluZSB0byBiZSBhYmxlIHRvIHN0YXJ0IEVNUiBTZXJ2ZXJsZXNzIGpvYlxuICAgKiBAcGFyYW0gcm9sZSBTdGVwIEZ1bmN0aW9ucyBTdGF0ZU1hY2hpbmUgSUFNIHJvbGVcbiAgICogQHNlZSBTcGFya1J1bnRpbWVTZXJ2ZXJsZXNzLmdyYW50Sm9iRXhlY3V0aW9uXG4gICAqL1xuXG4gIHByb3RlY3RlZCBncmFudEV4ZWN1dGlvblJvbGUocm9sZTogSVJvbGUpOiB2b2lkIHtcblxuICAgIGNvbnN0IGFybiA9IGBhcm46YXdzOmVtci1zZXJ2ZXJsZXNzOiR7QXdzLlJFR0lPTn06JHtBd3MuQUNDT1VOVF9JRH06L2FwcGxpY2F0aW9ucy8ke3RoaXMuY29uc3RydWN0Sm9iQ29uZmlnLmpvYkNvbmZpZy5BcHBsaWNhdGlvbklkfWA7XG5cbiAgICBTcGFya0VtclNlcnZlcmxlc3NSdW50aW1lLmdyYW50U3RhcnRKb2JFeGVjdXRpb24ocm9sZSwgW3RoaXMuY29uc3RydWN0Sm9iQ29uZmlnLmpvYkNvbmZpZy5FeGVjdXRpb25Sb2xlQXJuXSwgW2FybiwgYCR7YXJufS9qb2JydW5zLypgXSk7XG4gIH1cblxuICAvKipcbiAgICogU2V0IGRlZmF1bHRzIGZvciB0aGUgRW1yU2VydmVybGVzc1NwYXJrSm9iQXBpUHJvcHMuXG4gICAqIEBwYXJhbSBwcm9wcyBFbXJTZXJ2ZXJsZXNzU3BhcmtKb2JBcGlQcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSBzZXRKb2JBcGlQcm9wc0RlZmF1bHRzKHByb3BzOiBTcGFya0VtclNlcnZlcmxlc3NKb2JBcGlQcm9wcyk6IFNwYXJrRW1yU2VydmVybGVzc0pvYkFwaVByb3BzIHtcblxuICAgIGNvbnN0IHByb3BzUGFzY2FsQ2FzZSA9IFN0ZXBGdW5jdGlvblV0aWxzLmNhbWVsVG9QYXNjYWwocHJvcHMuam9iQ29uZmlnKTtcbiAgICAvL1NldCBkZWZhdWx0c1xuICAgIHByb3BzUGFzY2FsQ2FzZS5DbGllbnRUb2tlbiA/Pz0gSnNvblBhdGgudXVpZCgpO1xuICAgIHByb3BzUGFzY2FsQ2FzZS5FeGVjdXRpb25UaW1lb3V0TWludXRlcyA/Pz0gMzA7XG5cbiAgICBsZXQgY29uZmlnID0ge1xuICAgICAgam9iQ29uZmlnOiBwcm9wc1Bhc2NhbENhc2UsXG4gICAgICByZW1vdmFsUG9saWN5OiBwcm9wcy5yZW1vdmFsUG9saWN5LFxuICAgICAgc2NoZWR1bGU6IHByb3BzLnNjaGVkdWxlLFxuICAgIH07XG5cbiAgICByZXR1cm4gY29uZmlnO1xuICB9XG5cbiAgLyoqXG4gICAqIFNldCBkZWZhdWx0cyBmb3IgdGhlIEVtck9uRWtzU3BhcmtKb2JQcm9wcy5cbiAgICogQHBhcmFtIHByb3BzIEVtck9uRWtzU3BhcmtKb2JQcm9wc1xuICAgKi9cbiAgcHJpdmF0ZSBzZXRKb2JQcm9wc0RlZmF1bHRzKHByb3BzOiBTcGFya0VtclNlcnZlcmxlc3NKb2JQcm9wcyk6IFNwYXJrRW1yU2VydmVybGVzc0pvYkFwaVByb3BzIHtcblxuICAgIGNvbnN0IGNvbmZpZyA9IHtcbiAgICAgIGpvYkNvbmZpZzoge1xuICAgICAgICBDb25maWd1cmF0aW9uT3ZlcnJpZGVzOiB7XG4gICAgICAgICAgTW9uaXRvcmluZ0NvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgICAgIFMzTW9uaXRvcmluZ0NvbmZpZ3VyYXRpb246IHt9LFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIEpvYkRyaXZlcjoge1xuICAgICAgICAgIFNwYXJrU3VibWl0OiB7fSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICByZW1vdmFsUG9saWN5OiBwcm9wcy5yZW1vdmFsUG9saWN5LFxuICAgICAgc2NoZWR1bGU6IHByb3BzLnNjaGVkdWxlLFxuICAgIH0gYXMgU3BhcmtFbXJTZXJ2ZXJsZXNzSm9iQXBpUHJvcHM7XG5cbiAgICBjb25maWcuam9iQ29uZmlnLk5hbWUgPSBwcm9wcy5uYW1lO1xuICAgIGNvbmZpZy5qb2JDb25maWcuQ2xpZW50VG9rZW4gPSBKc29uUGF0aC51dWlkKCk7XG4gICAgY29uZmlnLmpvYkNvbmZpZy5FeGVjdXRpb25UaW1lb3V0TWludXRlcyA9IHByb3BzLmV4ZWN1dGlvblRpbWVvdXRNaW51dGVzID8/IDMwO1xuICAgIGNvbmZpZy5qb2JDb25maWcuRXhlY3V0aW9uUm9sZUFybj1wcm9wcy5leGVjdXRpb25Sb2xlQXJuO1xuICAgIGNvbmZpZy5qb2JDb25maWcuQXBwbGljYXRpb25JZCA9IHByb3BzLmFwcGxpY2F0aW9uSWQ7XG4gICAgY29uZmlnLmpvYkNvbmZpZy5Kb2JEcml2ZXIuU3BhcmtTdWJtaXQuRW50cnlQb2ludCA9IHByb3BzLnNwYXJrU3VibWl0RW50cnlQb2ludDtcblxuICAgIGlmIChwcm9wcy5zcGFya1N1Ym1pdEVudHJ5UG9pbnRBcmd1bWVudHMpIHtcbiAgICAgIGNvbmZpZy5qb2JDb25maWcuSm9iRHJpdmVyLlNwYXJrU3VibWl0LkVudHJ5UG9pbnRBcmd1bWVudHMgPSBwcm9wcy5zcGFya1N1Ym1pdEVudHJ5UG9pbnRBcmd1bWVudHM7XG4gICAgfVxuICAgIGlmIChwcm9wcy5zcGFya1N1Ym1pdFBhcmFtZXRlcnMpIHtcbiAgICAgIGNvbmZpZy5qb2JDb25maWcuSm9iRHJpdmVyLlNwYXJrU3VibWl0LlNwYXJrU3VibWl0UGFyYW1ldGVycyA9IHByb3BzLnNwYXJrU3VibWl0UGFyYW1ldGVycztcbiAgICB9XG5cbiAgICBpZiAocHJvcHMuYXBwbGljYXRpb25Db25maWd1cmF0aW9uKSB7XG4gICAgICBjb25maWcuam9iQ29uZmlnLkNvbmZpZ3VyYXRpb25PdmVycmlkZXMuQXBwbGljYXRpb25Db25maWd1cmF0aW9uID0gU3RlcEZ1bmN0aW9uVXRpbHMuY2FtZWxUb1Bhc2NhbChwcm9wcy5hcHBsaWNhdGlvbkNvbmZpZ3VyYXRpb24pO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5zM0xvZ1VyaSAmJiAhcHJvcHMuczNMb2dVcmkubWF0Y2goL15zMzpcXC9cXC8oW15cXC9dKykvKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBJbnZhbGlkIFMzIFVSSTogJHtwcm9wcy5zM0xvZ1VyaX1gKTtcbiAgICB9XG5cbiAgICBjb25maWcuam9iQ29uZmlnLkNvbmZpZ3VyYXRpb25PdmVycmlkZXMuTW9uaXRvcmluZ0NvbmZpZ3VyYXRpb24uUzNNb25pdG9yaW5nQ29uZmlndXJhdGlvbiEuTG9nVXJpID1cbiAgICB0aGlzLmNyZWF0ZVMzTG9nQnVja2V0KHByb3BzLnMzTG9nVXJpLCBwcm9wcy5zM0xvZ1VyaUtleUFybik7XG5cbiAgICBpZiAoIHByb3BzLnMzTG9nVXJpS2V5QXJuICkge1xuICAgICAgY29uZmlnLmpvYkNvbmZpZy5Db25maWd1cmF0aW9uT3ZlcnJpZGVzLk1vbml0b3JpbmdDb25maWd1cmF0aW9uLlMzTW9uaXRvcmluZ0NvbmZpZ3VyYXRpb24hLkVuY3J5cHRpb25LZXlBcm4gPSBwcm9wcy5zM0xvZ1VyaUtleUFybjtcbiAgICB9XG5cblxuICAgIGlmIChwcm9wcy5jbG91ZFdhdGNoTG9nR3JvdXBOYW1lKSB7XG4gICAgICB0aGlzLmNyZWF0ZUNsb3VkV2F0Y2hMb2dzTG9nR3JvdXAocHJvcHMuY2xvdWRXYXRjaExvZ0dyb3VwTmFtZSwgcHJvcHMuY2xvdWRXYXRjaEVuY3J5cHRpb25LZXlBcm4pO1xuICAgICAgY29uZmlnLmpvYkNvbmZpZy5Db25maWd1cmF0aW9uT3ZlcnJpZGVzLk1vbml0b3JpbmdDb25maWd1cmF0aW9uLkNsb3VkV2F0Y2hMb2dnaW5nQ29uZmlndXJhdGlvbiA9IHtcbiAgICAgICAgRW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgRW5jcnlwdGlvbktleUFybjogcHJvcHMuY2xvdWRXYXRjaEVuY3J5cHRpb25LZXlBcm4sXG4gICAgICAgIExvZ0dyb3VwTmFtZTogcHJvcHMuY2xvdWRXYXRjaExvZ0dyb3VwTmFtZSA/PyBwcm9wcy5uYW1lLFxuICAgICAgICBMb2dTdHJlYW1OYW1lUHJlZml4OiBwcm9wcy5jbG91ZFdhdGNoTG9nR3JvdXBTdHJlYW1QcmVmaXgsXG4gICAgICB9O1xuICAgIH1cblxuICAgIGNvbmZpZy5qb2JDb25maWcuVGFncyA9IHByb3BzLnRhZ3M7XG5cbiAgICByZXR1cm4gY29uZmlnO1xuXG4gIH1cbn1cblxuIl19