import { Duration } from 'aws-cdk-lib';
import { Schedule } from 'aws-cdk-lib/aws-events';
import { IRole } from 'aws-cdk-lib/aws-iam';
import { ILogGroup } from 'aws-cdk-lib/aws-logs';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { FailProps, StateMachine } from 'aws-cdk-lib/aws-stepfunctions';
import { CallAwsServiceProps } from 'aws-cdk-lib/aws-stepfunctions-tasks';
import { Construct } from 'constructs';
import { SparkJobProps } from './spark-job-props';
import { TrackedConstruct } from '../../../utils';
/**
 * A base construct to run Spark Jobs.
 *
 * Creates an AWS Step Functions State Machine that orchestrates the Spark Job.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-job
 *
 * Available implementations:
 * * {@link SparkEmrServerlessJob} for Emr Serverless implementation
 * * {@link SparkEmrEksJob} for EMR On EKS implementation
 *
 */
export declare abstract class SparkJob extends TrackedConstruct {
    /**
     * Step Functions StateMachine created to orchestrate the Spark Job
     */
    stateMachine?: StateMachine;
    /**
     * The CloudWatch Log Group used by the State Machine
     */
    stateMachineLogGroup?: ILogGroup;
    /**
     * S3 log bucket for the Spark job logs
     */
    protected s3LogBucket?: IBucket;
    /**
     * CloudWatch Logs Group for the Spark job logs
     */
    protected emrJobLogGroup?: ILogGroup;
    /**
     * The removal policy when deleting the CDK resource.
     * Resources like Amazon cloudwatch log or Amazon S3 bucket
     * If DESTROY is selected, the context value '@data-solutions-framework-on-aws/removeDataOnDestroy'
     * in the 'cdk.json' or 'cdk.context.json' must be set to true
     * @default - The resources are not deleted (`RemovalPolicy.RETAIN`).
     */
    private removalPolicy;
    /**
     * Constructs a new instance of the SparkJob class.
     * @param scope the Scope of the CDK Construct.
     * @param id the ID of the CDK Construct.
     * @param props the SparkJobProps properties.
     */
    constructor(scope: Construct, id: string, trackingTag: string, props: SparkJobProps);
    /**
     * Parameters for Step Functions task that runs the Spark job
     * @returns CallAwsServiceProps
     */
    protected abstract returnJobStartTaskProps(): CallAwsServiceProps;
    /**
     * Parameters for Step Functions task that monitors the Spark job
     * @returns CallAwsServiceProps
     */
    protected abstract returnJobMonitorTaskProps(): CallAwsServiceProps;
    /**
     * Parameters for Step Functions task that fails the Spark job
     * @returns FailProps
     */
    protected abstract returnJobFailTaskProps(): FailProps;
    /**
     * Returns the status of the Spark job that succeeded based on the GetJobRun API response
     * @returns string
     */
    protected abstract returnJobStatusSucceed(): string;
    /**
     * Returns the status of the Spark job that failed based on the GetJobRun API response
     * @returns string
     */
    protected abstract returnJobStatusFailed(): string;
    /**
     * Returns the status of the Spark job that is cancelled based on the GetJobRun API response
     */
    protected abstract returnJobStatusCancelled(): string;
    /**
     * Grants the execution role to the Step Functions state machine
     * @param role
     */
    protected abstract grantExecutionRole(role: IRole): void;
    /**
     * Creates a State Machine that orchestrates the Spark Job. This is a default implementation that can be overridden by the extending class.
     * @param jobTimeout Timeout for the state machine. @defautl 30 minutes
     * @param schedule Schedule to run the state machine. @default no schedule
     * @returns StateMachine
     */
    protected createStateMachine(jobTimeout?: Duration, schedule?: Schedule): StateMachine;
    /**
     * Creates or import an S3 bucket to store the logs of the Spark job.
     * The bucket is created with SSE encryption (KMS managed or provided by user).
     * @param s3LogUri S3 path to store the logs of the Spark job. Example: s3://<bucket-name>/
     * @param encryptionKeyArn KMS Key ARN for encryption. @default - Master KMS key of the account.
     * @returns string S3 path to store the logs.
     */
    protected createS3LogBucket(s3LogUri?: string, encryptionKeyArn?: string): string;
    /**
     * Creates an encrypted CloudWatch Logs group to store the Spark job logs.
     * @param name CloudWatch Logs group name of cloudwatch log group to store the Spark job logs
     * @param encryptionKeyArn KMS Key ARN for encryption. @default - Server-side encryption managed by CloudWatch Logs.
     * @returns LogGroup CloudWatch Logs group.
     */
    protected createCloudWatchLogsLogGroup(name: string, encryptionKeyArn?: string): ILogGroup;
}
