"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SparkJob = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("aws-cdk-lib/aws-stepfunctions-tasks");
const utils_1 = require("../../../utils");
/**
 * A base construct to run Spark Jobs.
 *
 * Creates an AWS Step Functions State Machine that orchestrates the Spark Job.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-job
 *
 * Available implementations:
 * * {@link SparkEmrServerlessJob} for Emr Serverless implementation
 * * {@link SparkEmrEksJob} for EMR On EKS implementation
 *
 */
class SparkJob extends utils_1.TrackedConstruct {
    /**
     * Constructs a new instance of the SparkJob class.
     * @param scope the Scope of the CDK Construct.
     * @param id the ID of the CDK Construct.
     * @param props the SparkJobProps properties.
     */
    constructor(scope, id, trackingTag, props) {
        const trackedConstructProps = {
            trackingTag: trackingTag,
        };
        super(scope, id, trackedConstructProps);
        this.removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
    }
    /**
     * Creates a State Machine that orchestrates the Spark Job. This is a default implementation that can be overridden by the extending class.
     * @param jobTimeout Timeout for the state machine. @defautl 30 minutes
     * @param schedule Schedule to run the state machine. @default no schedule
     * @returns StateMachine
     */
    createStateMachine(jobTimeout, schedule) {
        if (!this.stateMachine) {
            const emrStartJobTask = new aws_stepfunctions_tasks_1.CallAwsService(this, 'EmrStartJobTask', this.returnJobStartTaskProps());
            const emrMonitorJobTask = new aws_stepfunctions_tasks_1.CallAwsService(this, 'EmrMonitorJobTask', this.returnJobMonitorTaskProps());
            const wait = new aws_stepfunctions_1.Wait(this, 'Wait', {
                time: aws_stepfunctions_1.WaitTime.duration(aws_cdk_lib_1.Duration.seconds(60)),
            });
            const jobFailed = new aws_stepfunctions_1.Fail(this, 'JobFailed', this.returnJobFailTaskProps());
            const jobSucceeded = new aws_stepfunctions_1.Succeed(this, 'JobSucceeded');
            const emrPipelineChain = emrStartJobTask.next(wait).next(emrMonitorJobTask).next(new aws_stepfunctions_1.Choice(this, 'JobSucceededOrFailed')
                .when(aws_stepfunctions_1.Condition.stringEquals('$.JobRunState.State', this.returnJobStatusSucceed()), jobSucceeded)
                .when(aws_stepfunctions_1.Condition.stringEquals('$.JobRunState.State', this.returnJobStatusFailed()), jobFailed)
                .when(aws_stepfunctions_1.Condition.stringEquals('$.JobRunState.State', this.returnJobStatusCancelled()), jobFailed)
                .otherwise(wait));
            // Enable CloudWatch Logs for the state machine
            this.stateMachineLogGroup = new aws_logs_1.LogGroup(this, 'LogGroup', {
                removalPolicy: this.removalPolicy,
                logGroupName: `/aws/vendedlogs/states/${aws_cdk_lib_1.Names.uniqueId(this)}`,
            });
            // StepFunctions state machine
            this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'EmrPipeline', {
                definitionBody: aws_stepfunctions_1.DefinitionBody.fromChainable(emrPipelineChain),
                tracingEnabled: true,
                timeout: jobTimeout ?? aws_cdk_lib_1.Duration.minutes(30),
                logs: {
                    destination: this.stateMachineLogGroup,
                    level: aws_stepfunctions_1.LogLevel.ALL,
                },
            });
            this.grantExecutionRole(this.stateMachine.role);
            if (schedule) {
                new aws_events_1.Rule(this, 'SparkJobPipelineTrigger', {
                    schedule: schedule,
                    targets: [new aws_events_targets_1.SfnStateMachine(this.stateMachine)],
                });
            }
        }
        return this.stateMachine;
    }
    /**
     * Creates or import an S3 bucket to store the logs of the Spark job.
     * The bucket is created with SSE encryption (KMS managed or provided by user).
     * @param s3LogUri S3 path to store the logs of the Spark job. Example: s3://<bucket-name>/
     * @param encryptionKeyArn KMS Key ARN for encryption. @default - Master KMS key of the account.
     * @returns string S3 path to store the logs.
     */
    createS3LogBucket(s3LogUri, encryptionKeyArn) {
        if (!this.s3LogBucket) {
            this.s3LogBucket = s3LogUri ? aws_s3_1.Bucket.fromBucketName(this, 'SparkLogsBucket', s3LogUri.match(/s3:\/\/([^\/]+)/)[1]) : new aws_s3_1.Bucket(this, 'SparkLogsBucket', {
                blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
                enforceSSL: true,
                removalPolicy: this.removalPolicy,
                autoDeleteObjects: this.removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY,
                encryptionKey: encryptionKeyArn ? aws_kms_1.Key.fromKeyArn(this, 'SparkLogsBucketEncryptionKey', encryptionKeyArn) : undefined,
                encryption: encryptionKeyArn ? aws_s3_1.BucketEncryption.KMS : aws_s3_1.BucketEncryption.KMS_MANAGED,
            });
        }
        return s3LogUri ? s3LogUri : `s3://${this.s3LogBucket.bucketName}/`;
    }
    /**
     * Creates an encrypted CloudWatch Logs group to store the Spark job logs.
     * @param name CloudWatch Logs group name of cloudwatch log group to store the Spark job logs
     * @param encryptionKeyArn KMS Key ARN for encryption. @default - Server-side encryption managed by CloudWatch Logs.
     * @returns LogGroup CloudWatch Logs group.
     */
    createCloudWatchLogsLogGroup(name, encryptionKeyArn) {
        if (!this.emrJobLogGroup) {
            this.emrJobLogGroup = new aws_logs_1.LogGroup(this, 'SparkLogsCloudWatchLogGroup', {
                logGroupName: name,
                encryptionKey: encryptionKeyArn ? aws_kms_1.Key.fromKeyArn(this, 'SparkLogsCloudWatchEncryptionKey', encryptionKeyArn) : undefined,
                removalPolicy: this.removalPolicy,
            });
        }
        return this.emrJobLogGroup;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SparkJob[_a] = { fqn: "aws-dsf.processing.SparkJob", version: "1.0.0-rc.3" };
exports.SparkJob = SparkJob;
//# sourceMappingURL=data:application/json;base64,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