"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.createNamespace = exports.toolingManagedNodegroupSetup = exports.awsNodeRoleSetup = exports.ebsCsiDriverSetup = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const eks_controllers_version_1 = require("./eks-controllers-version");
const IamPolicyEbsCsiDriver = require("./resources/k8s/controllers-iam-policies/iam-policy-ebs-csi-driver.json");
const utils_1 = require("../../../../utils");
/**
 * @internal
 * Configure the EBS CSI driver on an Amazon EKS cluster
 * @param {Construct} scope the CDK scope to create resources in
 * @param {ICluster} cluster the EKS cluster to install the CSI driver in
 * @param {KubernetesVersion} eksClusterK8sVersion the Kubernetes version of the EKS cluster
 * @return {IRole} the IAM role used by the CSI driver
 */
function ebsCsiDriverSetup(scope, cluster, eksClusterK8sVersion) {
    const ebsCsiDriverIrsa = cluster.addServiceAccount('EbsCsiDriverSa', {
        name: 'ebs-csi-controller-sa',
        namespace: 'kube-system',
    });
    const ebsCsiDriverPolicyDocument = aws_iam_1.PolicyDocument.fromJson(IamPolicyEbsCsiDriver);
    const ebsCsiDriverPolicy = new aws_iam_1.Policy(scope, 'EbsCsiDriverPolicy', { document: ebsCsiDriverPolicyDocument });
    ebsCsiDriverPolicy.attachToRole(ebsCsiDriverIrsa.role);
    const ebsCSIDriver = new aws_eks_1.CfnAddon(scope, 'EbsCsiDriver', {
        addonName: 'aws-ebs-csi-driver',
        clusterName: cluster.clusterName,
        serviceAccountRoleArn: ebsCsiDriverIrsa.role.roleArn,
        addonVersion: eks_controllers_version_1.EBS_CSI_DRIVER_ADDON_VERSION.get(eksClusterK8sVersion),
        resolveConflicts: 'OVERWRITE',
    });
    ebsCSIDriver.node.addDependency(ebsCsiDriverIrsa);
    // Deploy the Helm Chart for the Certificate Manager. Required for EMR Studio ALB.
    cluster.addHelmChart('CertManager', {
        createNamespace: true,
        namespace: 'cert-manager',
        chart: 'cert-manager',
        repository: 'https://charts.jetstack.io',
        version: eks_controllers_version_1.CERTMANAGER_HELM_CHART_VERSION.get(eksClusterK8sVersion),
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        values: {
            startupapicheck: {
                timeout: '5m',
            },
            installCRDs: true,
        },
    });
    return ebsCsiDriverIrsa.role;
}
exports.ebsCsiDriverSetup = ebsCsiDriverSetup;
/**
 * @internal
 * Configure the IAM role used by the aws-node pod following AWS best practice not to use the EC2 instance role
 * @param {Construct} scope the CDK scope to create resources in
 * @param {ICluster} cluster the EKS cluster to configure the aws-node pod in
 * @return {IRole} the IAM role used by the aws-node pod
 */
function awsNodeRoleSetup(scope, cluster) {
    const awsNodeRole = new aws_iam_1.Role(scope, 'AwsNodeRole', {
        assumedBy: new aws_iam_1.FederatedPrincipal(cluster.openIdConnectProvider.openIdConnectProviderArn, { ...[] }, 'sts:AssumeRoleWithWebIdentity'),
        description: `awsNodeRole-${cluster.clusterName}`,
        managedPolicies: [aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy')],
    });
    // update the aws-node service account with IAM role created for it
    new aws_eks_1.KubernetesManifest(scope, 'AwsNodeSaUpdateManifest', {
        cluster: cluster,
        manifest: [
            {
                apiVersion: 'v1',
                kind: 'ServiceAccount',
                metadata: {
                    name: 'aws-node',
                    namespace: 'kube-system',
                    annotations: {
                        'eks.amazonaws.com/role-arn': awsNodeRole.roleArn,
                    },
                },
            },
        ],
        overwrite: true,
    });
    return awsNodeRole;
}
exports.awsNodeRoleSetup = awsNodeRoleSetup;
/**
 * @internal
 * Method to setup a managed nodegroup to bootstrap all cluster vital componenets like
 * core dns, karpenter, ebs csi driver.
 * @param {Construct} scope the CDK scope to create the nodegroup in
 * @param {Cluster} cluster the EKS cluster to create the nodegroup in
 * @param {IRole} nodeRole the IAM role to use for the nodegroup
 */
function toolingManagedNodegroupSetup(scope, cluster, nodeRole) {
    const toolingLaunchTemplate = new aws_ec2_1.CfnLaunchTemplate(scope, 'toolinglaunchtemplate', {
        launchTemplateName: 'ToolingNodegroup',
        launchTemplateData: {
            metadataOptions: {
                httpEndpoint: 'enabled',
                httpProtocolIpv6: 'disabled',
                httpPutResponseHopLimit: 2,
                httpTokens: 'required',
            },
        },
    });
    let toolingManagedNodegroupOptions = {
        nodegroupName: 'tooling',
        instanceTypes: [new aws_ec2_1.InstanceType('t3.medium')],
        amiType: aws_eks_1.NodegroupAmiType.BOTTLEROCKET_X86_64,
        minSize: 2,
        maxSize: 2,
        labels: { role: 'tooling' },
        launchTemplateSpec: {
            id: toolingLaunchTemplate.ref,
            version: toolingLaunchTemplate.attrLatestVersionNumber,
        },
        nodeRole: nodeRole,
    };
    cluster.addNodegroupCapacity('toolingMNG', toolingManagedNodegroupOptions);
}
exports.toolingManagedNodegroupSetup = toolingManagedNodegroupSetup;
/**
 * @internal
 * Create a namespace with a predefined baseline
 *  * Create namespace
 *  * Define a Network Policy
 * @param {ICluster} cluster the EKS cluster to create the namespace in
 * @param {string} namespace the namespace to create
 * @return {KubernetesManifest} the Kubernetes manifest for the namespace
 */
function createNamespace(cluster, namespace) {
    const regex = /^[a-z0-9]([-a-z0-9]*[a-z0-9])?$/;
    const reg = RegExp(regex);
    if (!reg.exec(namespace) || namespace.length > 63) {
        throw new Error(`Namespace provided violates the constraints of Namespace naming ${namespace}`);
    }
    //Create namespace with pod security admission to with pod security standard to baseline
    //To learn more look at https://kubernetes.io/docs/concepts/security/pod-security-standards/
    let ns = cluster.addManifest(`${namespace}-Namespace`, {
        apiVersion: 'v1',
        kind: 'Namespace',
        metadata: {
            name: namespace,
            labels: {
                'pod-security.kubernetes.io/enforce': 'baseline',
                'pod-security.kubernetes.io/enforce-version': 'v1.28',
            },
        },
    });
    //Create network policy for namespace
    let manifestNetworkPolicy = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/network-policy-pod2pod-internet.yml`);
    manifestNetworkPolicy = manifestNetworkPolicy.replace(/(\{{NAMESPACE}})/g, namespace);
    let manifestNetworkPolicyManifestYAML = manifestNetworkPolicy.split('---').map((e) => utils_1.Utils.loadYaml(e));
    const manifestApplyNetworkPolicy = cluster.addManifest(`${namespace}-network-policy`, ...manifestNetworkPolicyManifestYAML);
    manifestApplyNetworkPolicy.node.addDependency(ns);
    //Create resource quota and limit range for namespace
    let manifestResourceManagement = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/resource-management.yaml`);
    manifestResourceManagement = manifestResourceManagement.replace(/(\{{NAMESPACE}})/g, namespace);
    let manifestResourceManagementYAML = manifestResourceManagement.split('---').map((e) => utils_1.Utils.loadYaml(e));
    const manifestApplResourceManagement = cluster.addManifest(`${namespace}-resource-management`, ...manifestResourceManagementYAML);
    manifestApplResourceManagement.node.addDependency(ns);
    return ns;
}
exports.createNamespace = createNamespace;
//# sourceMappingURL=data:application/json;base64,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