"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.karpenterSetup = exports.karpenterManifestSetup = exports.setDefaultKarpenterProvisioners = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_sqs_1 = require("aws-cdk-lib/aws-sqs");
const utils_1 = require("../../../../utils");
/**
 * @internal
 * Method to add the default Karpenter provisioners for Spark workloads
 * @param {Cluster} cluster the EMR on EKS cluster to configure the Karpenter provisioners in
 * @param {KarpenterVersion} karpenterVersion the Karpenter version to use for the provisioners
 * @param {IRole} nodeRole the IAM role to use for the provisioners
 */
function setDefaultKarpenterProvisioners(cluster, karpenterVersion, nodeRole) {
    const subnets = cluster.eksCluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    subnets.forEach((subnet, index) => {
        let criticalManifestYAML = karpenterManifestSetup(cluster.eksCluster, `${__dirname}/resources/k8s/karpenter-provisioner-config/${karpenterVersion}/critical-provisioner.yml`, subnet, nodeRole);
        cluster.addKarpenterProvisioner(`karpenterCriticalManifest-${index}`, criticalManifestYAML);
        let sharedDriverManifestYAML = karpenterManifestSetup(cluster.eksCluster, `${__dirname}/resources/k8s/karpenter-provisioner-config/${karpenterVersion}/shared-driver-provisioner.yml`, subnet, nodeRole);
        cluster.addKarpenterProvisioner(`karpenterSharedDriverManifest-${index}`, sharedDriverManifestYAML);
        let sharedExecutorManifestYAML = karpenterManifestSetup(cluster.eksCluster, `${__dirname}/resources/k8s/karpenter-provisioner-config/${karpenterVersion}/shared-executor-provisioner.yml`, subnet, nodeRole);
        cluster.addKarpenterProvisioner(`karpenterSharedExecutorManifest-${index}`, sharedExecutorManifestYAML);
        let notebookDriverManifestYAML = karpenterManifestSetup(cluster.eksCluster, `${__dirname}/resources/k8s/karpenter-provisioner-config/${karpenterVersion}/notebook-driver-provisioner.yml`, subnet, nodeRole);
        cluster.addKarpenterProvisioner(`karpenterNotebookDriverManifest-${index}`, notebookDriverManifestYAML);
        let notebookExecutorManifestYAML = karpenterManifestSetup(cluster.eksCluster, `${__dirname}/resources/k8s/karpenter-provisioner-config/${karpenterVersion}/notebook-executor-provisioner.yml`, subnet, nodeRole);
        cluster.addKarpenterProvisioner(`karpenterNotebookExecutorManifest-${index}`, notebookExecutorManifestYAML);
    });
}
exports.setDefaultKarpenterProvisioners = setDefaultKarpenterProvisioners;
/**
   * @internal
   * Method to generate the Karpenter manifests from templates and targeted to the specific EKS cluster
   * @param {ICluster} cluster the name of the EKS cluster to target the manifests to
   * @param {string} path the path to the manifest template
   * @param {ISubnet} subnet the subnet to target the manifests to
   * @param {IRole} nodeRole the IAM role to use for the manifests
   * @return {any} the Kubernetes manifest for Karpenter provisioned
   */
function karpenterManifestSetup(cluster, path, subnet, nodeRole) {
    let manifest = utils_1.Utils.readYamlDocument(path);
    manifest = manifest.replace('{{subnet-id}}', subnet.subnetId);
    manifest = manifest.replace(/(\{{az}})/g, subnet.availabilityZone);
    manifest = manifest.replace('{{cluster-name}}', cluster.clusterName);
    manifest = manifest.replace(/(\{{ROLENAME}})/g, nodeRole.roleName);
    let manfifestYAML = manifest.split('---').map((e) => utils_1.Utils.loadYaml(e));
    return manfifestYAML;
}
exports.karpenterManifestSetup = karpenterManifestSetup;
/**
   * @internal
   * Install all the required configurations of Karpenter SQS and Event rules to handle spot and unhealthy instance termination
   * Create a security group to be used by nodes created with karpenter
   * Tags the subnets and VPC to be used by karpenter
   * create a tooling provisioner that will deploy in each of the AZs, one per AZ
   * @param {Cluster} cluster the EKS cluster to configure the Karpenter provisioners in
   * @param {string} clusterName the name of the EKS cluster to target the manifests to
   * @param {CfnInstanceProfile} instanceProfile the IAM instance profile to use for the Karpenter nodes
   * @param {IRole} nodeRole the IAM role to use for the Karpenter nodes
   * @param {KarpenterVersion} karpenterVersion the Karpenter version to use for the provisioners
   * @return {[HelmChart, IRole, IQueue, Array<IRule>]} the Helm chart to install, the IAM Role for service account, the SQS queue and the EventBridge rules for Karpenter
   */
function karpenterSetup(cluster, clusterName, scope, instanceProfile, nodeRole, karpenterRemovalPolicy, karpenterVersion) {
    const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, karpenterRemovalPolicy);
    const karpenterInterruptionQueue = new aws_sqs_1.Queue(scope, 'KarpenterInterruptionQueue', {
        retentionPeriod: aws_cdk_lib_1.Duration.seconds(300),
        enforceSSL: true,
        removalPolicy,
    });
    karpenterInterruptionQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['sqs:SendMessage'],
        principals: [new aws_iam_1.ServicePrincipal('sqs.amazonaws.com'), new aws_iam_1.ServicePrincipal('events.amazonaws.com')],
    }));
    const scheduledChangeRule = new aws_events_1.Rule(scope, 'ScheduledChangeRule', {
        eventPattern: {
            source: ['aws.heatlh'],
            detail: ['AWS Health Event'],
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)],
    });
    const stateChangeRule = new aws_events_1.Rule(scope, 'InstanceStateChangeRule', {
        eventPattern: {
            source: ['aws.ec2'],
            detail: ['EC2 Instance State-change Notification'],
        },
        targets: [new aws_events_targets_1.SqsQueue(karpenterInterruptionQueue)],
    });
    const karpenterControllerPolicyStatementSSM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['ssm:GetParameter', 'pricing:GetProducts'],
        resources: ['*'],
    });
    const karpenterControllerPolicyStatementEC2 = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: [
            'ec2:DescribeAvailabilityZones',
            'ec2:DescribeImages',
            'ec2:DescribeInstances',
            'ec2:DescribeInstanceTypeOfferings',
            'ec2:DescribeInstanceTypes',
            'ec2:DescribeLaunchTemplates',
            'ec2:DescribeSecurityGroups',
            'ec2:DescribeSpotPriceHistory',
            'ec2:DescribeSubnets',
        ],
        resources: ['*'],
        conditions: {
            StringEquals: {
                'aws:RequestedRegion': aws_cdk_lib_1.Stack.of(scope).region,
            },
        },
    });
    const allowScopedEC2InstanceActions = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        resources: [
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}::image/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}::snapshot/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:spot-instances-request/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:security-group/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:subnet/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:launch-template/*`,
        ],
        actions: ['ec2:RunInstances', 'ec2:CreateFleet'],
    });
    const allowScopedEC2LaunchTemplateActions = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        resources: [`arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:launch-template/*`],
        actions: ['ec2:CreateLaunchTemplate'],
        conditions: {
            StringEquals: {
                [`aws:RequestTag/kubernetes.io/cluster/${clusterName}`]: 'owned',
            },
            StringLike: {
                'aws:RequestTag/karpenter.sh/provisioner-name': '*',
            },
        },
    });
    const allowScopedEC2InstanceActionsWithTags = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        resources: [
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:fleet/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:instance/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:volume/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:network-interface/*`,
        ],
        actions: ['ec2:RunInstances', 'ec2:CreateFleet'],
        conditions: {
            StringEquals: {
                [`aws:RequestTag/kubernetes.io/cluster/${clusterName}`]: 'owned',
            },
            StringLike: {
                'aws:RequestTag/karpenter.sh/provisioner-name': '*',
            },
        },
    });
    const allowScopedResourceCreationTagging = new aws_iam_1.PolicyStatement({
        sid: 'AllowScopedResourceCreationTagging',
        effect: aws_iam_1.Effect.ALLOW,
        resources: [
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:fleet/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:instance/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:volume/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:network-interface/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:launch-template/*`,
        ],
        actions: ['ec2:CreateTags'],
        conditions: {
            StringEquals: {
                [`aws:RequestTag/kubernetes.io/cluster/${clusterName}`]: 'owned',
                'ec2:CreateAction': ['RunInstances', 'CreateFleet', 'CreateLaunchTemplate'],
            },
            StringLike: {
                'aws:RequestTag/karpenter.sh/provisioner-name': '*',
            },
        },
    });
    const allowMachineMigrationTagging = new aws_iam_1.PolicyStatement({
        sid: 'AllowMachineMigrationTagging',
        effect: aws_iam_1.Effect.ALLOW,
        resources: [`arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:instance/*`],
        actions: ['ec2:CreateTags'],
        conditions: {
            'StringEquals': {
                [`aws:ResourceTag/kubernetes.io/cluster/${clusterName}`]: 'owned',
                'aws:RequestTag/karpenter.sh/managed-by': `${clusterName}`,
            },
            'StringLike': {
                'aws:RequestTag/karpenter.sh/provisioner-name': '*',
            },
            'ForAllValues:StringEquals': {
                'aws:TagKeys': ['karpenter.sh/provisioner-name', 'karpenter.sh/managed-by'],
            },
        },
    });
    const allowScopedDeletion = new aws_iam_1.PolicyStatement({
        sid: 'AllowScopedDeletion',
        effect: aws_iam_1.Effect.ALLOW,
        resources: [
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:instance/*`,
            `arn:aws:ec2:${aws_cdk_lib_1.Stack.of(scope).region}:*:launch-template/*`,
        ],
        actions: ['ec2:TerminateInstances', 'ec2:DeleteLaunchTemplate'],
        conditions: {
            StringEquals: {
                [`aws:ResourceTag/kubernetes.io/cluster/${clusterName}`]: 'owned',
            },
            StringLike: {
                'aws:ResourceTag/karpenter.sh/provisioner-name': '*',
            },
        },
    });
    const karpenterControllerPolicyStatementIAM = new aws_iam_1.PolicyStatement({
        effect: aws_iam_1.Effect.ALLOW,
        actions: ['iam:PassRole'],
        resources: [`${nodeRole.roleArn}`],
        conditions: {
            StringEquals: {
                'iam:PassedToService': 'ec2.amazonaws.com',
            },
        },
    });
    const allowInterruptionQueueActions = new aws_iam_1.PolicyStatement({
        sid: 'AllowInterruptionQueueActions',
        effect: aws_iam_1.Effect.ALLOW,
        resources: [karpenterInterruptionQueue.queueArn],
        actions: ['sqs:DeleteMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl', 'sqs:ReceiveMessage'],
    });
    const allowAPIServerEndpointDiscovery = new aws_iam_1.PolicyStatement({
        sid: 'AllowAPIServerEndpointDiscovery',
        effect: aws_iam_1.Effect.ALLOW,
        resources: [`arn:aws:eks:${aws_cdk_lib_1.Stack.of(scope).region}:${aws_cdk_lib_1.Stack.of(scope).account}:cluster/${clusterName}`],
        actions: ['eks:DescribeCluster'],
    });
    const allowInstanceProfileReadActions = new aws_iam_1.PolicyStatement({
        sid: 'AllowInstanceProfileReadActions',
        effect: aws_iam_1.Effect.ALLOW,
        resources: ['*'],
        actions: ['iam:GetInstanceProfile'],
    });
    const karpenterNS = cluster.addManifest('karpenterNS', {
        apiVersion: 'v1',
        kind: 'Namespace',
        metadata: { name: 'karpenter' },
    });
    const karpenterAccount = cluster.addServiceAccount('karpenterServiceAccount', {
        name: 'karpenter',
        namespace: 'karpenter',
    });
    karpenterAccount.node.addDependency(karpenterNS);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementSSM);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementEC2);
    karpenterAccount.addToPrincipalPolicy(karpenterControllerPolicyStatementIAM);
    karpenterAccount.addToPrincipalPolicy(allowScopedEC2InstanceActions);
    karpenterAccount.addToPrincipalPolicy(allowScopedEC2InstanceActionsWithTags);
    karpenterAccount.addToPrincipalPolicy(allowScopedEC2LaunchTemplateActions);
    karpenterAccount.addToPrincipalPolicy(allowMachineMigrationTagging);
    karpenterAccount.addToPrincipalPolicy(allowScopedResourceCreationTagging);
    karpenterAccount.addToPrincipalPolicy(allowScopedDeletion);
    karpenterAccount.addToPrincipalPolicy(allowInterruptionQueueActions);
    karpenterAccount.addToPrincipalPolicy(allowAPIServerEndpointDiscovery);
    karpenterAccount.addToPrincipalPolicy(allowInstanceProfileReadActions);
    //Deploy Karpenter Chart
    const karpenterChart = cluster.addHelmChart('KarpenterHelmChart', {
        chart: 'karpenter',
        release: 'karpenter',
        repository: 'oci://public.ecr.aws/karpenter/karpenter',
        namespace: 'karpenter',
        version: karpenterVersion,
        timeout: aws_cdk_lib_1.Duration.minutes(14),
        wait: true,
        values: {
            serviceAccount: {
                name: 'karpenter',
                create: false,
                annotations: {
                    'eks.amazonaws.com/role-arn': karpenterAccount.role.roleArn,
                },
            },
            settings: {
                aws: {
                    defaultInstanceProfile: instanceProfile.instanceProfileName,
                    clusterName: clusterName,
                    clusterEndpoint: cluster.clusterEndpoint,
                    interruptionQueueName: karpenterInterruptionQueue.queueName,
                },
            },
        },
    });
    karpenterChart.node.addDependency(karpenterAccount);
    const karpenterInstancesSg = new aws_ec2_1.SecurityGroup(scope, 'KarpenterSg', {
        vpc: cluster.vpc,
        allowAllOutbound: true,
        description: 'security group for a karpenter instances',
        disableInlineRules: true,
    });
    aws_cdk_lib_1.Tags.of(karpenterInstancesSg).add('karpenter.sh/discovery', `${clusterName}`);
    cluster.clusterSecurityGroup.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(karpenterInstancesSg, aws_ec2_1.Port.allTraffic());
    karpenterInstancesSg.addIngressRule(cluster.clusterSecurityGroup, aws_ec2_1.Port.allTraffic());
    aws_cdk_lib_1.Tags.of(cluster.vpc).add('karpenter.sh/discovery', clusterName);
    cluster.vpc.privateSubnets.forEach((subnet) => {
        aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', clusterName);
    });
    cluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('karpenter.sh/discovery', clusterName));
    const privateSubnets = cluster.vpc.selectSubnets({
        onePerAz: true,
        subnetType: aws_ec2_1.SubnetType.PRIVATE_WITH_EGRESS,
    }).subnets;
    let manifest = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/karpenter-provisioner-config/${karpenterVersion}/tooling-provisioner.yml`);
    manifest = manifest.replace(/(\{{cluster-name}})/g, clusterName);
    manifest = manifest.replace(/(\{{ROLENAME}})/g, nodeRole.roleName);
    const subnetIdHolder = ['subnet-1', 'subnet-2'];
    privateSubnets.forEach((subnet, index) => {
        let subnetHolder = `{{${subnetIdHolder[index]}}}`;
        let re = new RegExp(subnetHolder, 'g');
        manifest = manifest.replace(re, subnet.subnetId);
    });
    let manfifestYAML = manifest.split('---').map((e) => utils_1.Utils.loadYaml(e));
    const manifestApply = cluster.addManifest('provisioner-tooling', ...manfifestYAML);
    manifestApply.node.addDependency(karpenterChart);
    return [karpenterChart, karpenterAccount.role, karpenterInterruptionQueue, karpenterInstancesSg, [scheduledChangeRule, stateChangeRule]];
}
exports.karpenterSetup = karpenterSetup;
//# sourceMappingURL=data:application/json;base64,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