import { RemovalPolicy } from 'aws-cdk-lib';
import { ISecurityGroup } from 'aws-cdk-lib/aws-ec2';
import { Cluster, KubernetesVersion } from 'aws-cdk-lib/aws-eks';
import { CfnVirtualCluster } from 'aws-cdk-lib/aws-emrcontainers';
import { IRule } from 'aws-cdk-lib/aws-events';
import { IManagedPolicy, IRole, Role } from 'aws-cdk-lib/aws-iam';
import { IBucket } from 'aws-cdk-lib/aws-s3';
import { IQueue } from 'aws-cdk-lib/aws-sqs';
import { Construct } from 'constructs';
import { EmrVirtualClusterProps } from './emr-virtual-cluster-props';
import { SparkEmrContainersRuntimeProps } from './spark-emr-containers-runtime-props';
import { TrackedConstruct } from '../../../../utils';
/**
 * A construct to create an EKS cluster, configure it and enable it with EMR on EKS
 * @see https://awslabs.github.io/aws-data-solutions-framework/docs/constructs/library/spark-emr-containers-runtime
*/
export declare class SparkEmrContainersRuntime extends TrackedConstruct {
    static readonly DEFAULT_EMR_EKS_VERSION: import("../../emr-releases").EmrRuntimeVersion;
    static readonly DEFAULT_EKS_VERSION: KubernetesVersion;
    static readonly DEFAULT_CLUSTER_NAME = "data-platform";
    static readonly DEFAULT_VPC_CIDR = "10.0.0.0/16";
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope: Construct, props: SparkEmrContainersRuntimeProps): SparkEmrContainersRuntime;
    /**
     * The EKS cluster created by the construct if it is not provided
     */
    readonly eksCluster: Cluster;
    readonly csiDriverIrsa?: IRole;
    /**
     * IAM Role used by IRSA for the aws-node daemonset
     */
    readonly awsNodeRole?: IRole;
    /**
     * IAM role used by the tooling managed nodegroup hosting core Kubernetes controllers
     * like EBS CSI driver, core dns
     */
    readonly ec2InstanceNodeGroupRole: IRole;
    /**
     * SQS queue used by Karpenter to receive critical events from AWS services which may affect your nodes.
     */
    readonly karpenterQueue?: IQueue;
    /**
     * The security group used by the EC2NodeClass of the default nodes
     */
    readonly karpenterSecurityGroup?: ISecurityGroup;
    /**
     * Rules used by Karpenter to track node health, rules are defined in the cloudformation below
     * https://raw.githubusercontent.com/aws/karpenter/"${KARPENTER_VERSION}"/website/content/en/preview/getting-started/getting-started-with-karpenter/cloudformation.yaml
     */
    readonly karpenterEventRules?: Array<IRule>;
    /**
     * The configuration override for the spark application to use with the default nodes dedicated for notebooks
     */
    readonly notebookDefaultConfig?: string;
    /**
     * The configuration override for the spark application to use with the default nodes for criticale jobs
     */
    readonly criticalDefaultConfig?: string;
    /**
     * The configuration override for the spark application to use with the default nodes for none criticale jobs
     */
    readonly sharedDefaultConfig?: string;
    /**
     * The bucket holding podtemplates referenced in the configuration override for the job
     */
    readonly assetBucket?: IBucket;
    /**
     * The s3 location holding the driver pod tempalte for critical nodes
     */
    readonly podTemplateS3LocationCriticalDriver?: string;
    /**
     * The s3 location holding the executor pod tempalte for critical nodes
     */
    readonly podTemplateS3LocationCriticalExecutor?: string;
    /**
     * The s3 location holding the driver pod tempalte for shared nodes
     */
    readonly podTemplateS3LocationDriverShared?: string;
    /**
     * The s3 location holding the executor pod tempalte for shared nodes
     */
    readonly podTemplateS3LocationExecutorShared?: string;
    /**
     * The s3 location holding the driver pod tempalte for interactive sessions
     */
    readonly podTemplateS3LocationNotebookDriver?: string;
    /**
     * The s3 location holding the executor pod tempalte for interactive sessions
     */
    readonly podTemplateS3LocationNotebookExecutor?: string;
    /**
     * The IAM Role created for the EBS CSI controller
     */
    readonly csiDriverIrsaRole?: IRole;
    /**
     * The IAM role created for the Karpenter controller
     */
    readonly karpenterIrsaRole?: IRole;
    private readonly emrServiceRole?;
    private readonly assetUploadBucketRole?;
    private readonly karpenterChart?;
    private readonly defaultNodes?;
    private readonly createEmrOnEksServiceLinkedRole?;
    private readonly logKmsKey?;
    private readonly eksSecretKmsKey?;
    private readonly podTemplateLocation;
    private readonly podTemplatePolicy;
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    private constructor();
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterProps} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope: Construct, options: EmrVirtualClusterProps): CfnVirtualCluster;
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} eksNamespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope: Construct, id: string, policy: IManagedPolicy, eksNamespace: string, name: string): Role;
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id: string, filePath: string, removalPolicy: RemovalPolicy): void;
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id: string, manifest: any): any;
}
