"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SparkEmrContainersRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const fs_1 = require("fs");
const path_1 = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_eks_1 = require("aws-cdk-lib/aws-eks");
const aws_emrcontainers_1 = require("aws-cdk-lib/aws-emrcontainers");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const aws_s3_deployment_1 = require("aws-cdk-lib/aws-s3-deployment");
const SimpleBase = require("simple-base");
const eks_cluster_helpers_1 = require("./eks-cluster-helpers");
const eks_karpenter_helpers_1 = require("./eks-karpenter-helpers");
const CriticalDefaultConfig = require("./resources/k8s/emr-eks-config/critical.json");
const NotebookDefaultConfig = require("./resources/k8s/emr-eks-config/notebook-pod-template-ready.json");
const SharedDefaultConfig = require("./resources/k8s/emr-eks-config/shared.json");
const utils_1 = require("../../../../utils");
const emr_releases_1 = require("../../emr-releases");
const karpenter_releases_1 = require("../../karpenter-releases");
/**
 * A construct to create an EKS cluster, configure it and enable it with EMR on EKS
 * @see https://awslabs.github.io/aws-data-solutions-framework/docs/constructs/library/spark-emr-containers-runtime
*/
class SparkEmrContainersRuntime extends utils_1.TrackedConstruct {
    /**
     * Get an existing EmrEksCluster based on the cluster name property or create a new one
     * only one EKS cluster can exist per stack
     * @param {Construct} scope the CDK scope used to search or create the cluster
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps} if created
     */
    static getOrCreate(scope, props) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const id = utils_1.Utils.toPascalCase(props.eksClusterName || SparkEmrContainersRuntime.DEFAULT_CLUSTER_NAME);
        let emrEksCluster = stack.node.tryFindChild(id) ??
            new SparkEmrContainersRuntime(stack, id, props);
        return emrEksCluster;
    }
    /**
     * Constructs a new instance of the EmrEksCluster construct.
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {EmrEksClusterProps} props the EmrEksClusterProps [properties]{@link EmrEksClusterProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: SparkEmrContainersRuntime.name,
        };
        super(scope, id, trackedConstructProps);
        let removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
        // Create a role to be used as instance profile for nodegroups
        this.ec2InstanceNodeGroupRole = props.ec2InstanceRole || new aws_iam_1.Role(this, 'Ec2InstanceNodeGroupRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
        });
        //attach policies to the role to be used by the nodegroups
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
        this.ec2InstanceNodeGroupRole.addManagedPolicy(aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        //Create instance profile to be used by Managed nodegroup and karpenter
        const clusterInstanceProfile = new aws_iam_1.CfnInstanceProfile(scope, 'KarpenterInstanceProfile', {
            roles: [this.ec2InstanceNodeGroupRole.roleName],
            // instanceProfileName: `adsfNodeInstanceProfile-${clusterName ?? 'default'}`,
            path: '/',
        });
        const karpenterVersion = props.karpenterVersion ?? karpenter_releases_1.DEFAULT_KARPENTER_VERSION;
        let eksCluster;
        // create an Amazon EKS CLuster with default parameters if not provided in the properties
        if (props.eksCluster == undefined) {
            this.logKmsKey = new aws_kms_1.Key(scope, 'LogKmsKey', {
                enableKeyRotation: true,
                description: 'log-vpc-key',
                removalPolicy: removalPolicy,
            });
            this.eksSecretKmsKey = new aws_kms_1.Key(scope, 'EksSecretKmsKey', {
                enableKeyRotation: true,
                description: 'eks-secrets-key',
                removalPolicy: removalPolicy,
            });
            const clusterName = props.eksClusterName ?? SparkEmrContainersRuntime.DEFAULT_CLUSTER_NAME;
            //Define EKS cluster logging
            const eksClusterLogging = [
                aws_eks_1.ClusterLoggingTypes.API,
                aws_eks_1.ClusterLoggingTypes.AUTHENTICATOR,
                aws_eks_1.ClusterLoggingTypes.SCHEDULER,
                aws_eks_1.ClusterLoggingTypes.CONTROLLER_MANAGER,
                aws_eks_1.ClusterLoggingTypes.AUDIT,
            ];
            //Set the flag for creating the EMR on EKS Service Linked Role
            this.createEmrOnEksServiceLinkedRole = props.createEmrOnEksServiceLinkedRole ?? true;
            //Set flag for default karpenter provisioners for Spark jobs
            this.defaultNodes = props.defaultNodes ?? true;
            const vpcCidr = props.vpcCidr ? props.vpcCidr : SparkEmrContainersRuntime.DEFAULT_VPC_CIDR;
            let eksVpc = props.eksVpc ? props.eksVpc : (0, utils_1.vpcBootstrap)(scope, vpcCidr, this.logKmsKey, removalPolicy, clusterName, undefined).vpc;
            eksCluster = new aws_eks_1.Cluster(scope, 'EksCluster', {
                defaultCapacity: 0,
                clusterName: clusterName,
                version: props.kubernetesVersion ?? SparkEmrContainersRuntime.DEFAULT_EKS_VERSION,
                clusterLogging: eksClusterLogging,
                kubectlLayer: props.kubectlLambdaLayer,
                vpc: eksVpc,
                endpointAccess: aws_eks_1.EndpointAccess.PUBLIC_AND_PRIVATE,
                secretsEncryptionKey: this.eksSecretKmsKey,
                albController: {
                    version: aws_eks_1.AlbControllerVersion.V2_5_1,
                    policy: JSON.parse((0, fs_1.readFileSync)((0, path_1.join)(__dirname, 'resources/k8s/controllers-iam-policies/alb/iam-policy-alb-v2.5.json'), 'utf8')),
                },
                placeClusterHandlerInVpc: true,
            });
            // Add the provided Amazon IAM Role as Amazon EKS Admin
            if (props.eksAdminRole === undefined) {
                throw new Error('An IAM role must be passed to create an EKS cluster');
            }
            else {
                eksCluster.awsAuth.addMastersRole(props.eksAdminRole, 'AdminRole');
            }
            // Configure the EBS CSI controler
            this.csiDriverIrsaRole = (0, eks_cluster_helpers_1.ebsCsiDriverSetup)(this, eksCluster, props.kubernetesVersion ?? SparkEmrContainersRuntime.DEFAULT_EKS_VERSION);
            // Configure the AWS Node Role
            this.awsNodeRole = (0, eks_cluster_helpers_1.awsNodeRoleSetup)(this, eksCluster);
            // Configure the tooling nodegroup for hosting tooling components
            (0, eks_cluster_helpers_1.toolingManagedNodegroupSetup)(this, eksCluster, this.ec2InstanceNodeGroupRole);
            //Deploy karpenter
            [this.karpenterChart, this.karpenterIrsaRole, this.karpenterQueue, this.karpenterSecurityGroup, this.karpenterEventRules] = (0, eks_karpenter_helpers_1.karpenterSetup)(eksCluster, clusterName, scope, clusterInstanceProfile, this.ec2InstanceNodeGroupRole, removalPolicy, karpenterVersion);
        }
        else {
            //Initialize with the provided EKS Cluster
            eksCluster = props.eksCluster;
        }
        this.eksCluster = eksCluster;
        // Create an Amazon S3 Bucket for podTemplate assets
        this.assetBucket = new aws_s3_1.Bucket(this, 'AssetBucket', {
            encryption: aws_s3_1.BucketEncryption.KMS_MANAGED,
            enforceSSL: true,
            removalPolicy: removalPolicy,
        });
        // Configure the podTemplate location
        this.podTemplateLocation = {
            bucketName: this.assetBucket.bucketName,
            objectKey: `${this.eksCluster.clusterName}/pod-template`,
        };
        let s3DeploymentLambdaPolicyStatement = [];
        s3DeploymentLambdaPolicyStatement.push(new aws_iam_1.PolicyStatement({
            actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
            resources: [`arn:aws:logs:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:*`],
            effect: aws_iam_1.Effect.ALLOW,
        }));
        //Policy to allow lambda access to cloudwatch logs
        const lambdaExecutionRolePolicy = new aws_iam_1.ManagedPolicy(this, 'S3BucketDeploymentPolicy', {
            statements: s3DeploymentLambdaPolicyStatement,
            description: 'Policy used by S3 deployment cdk construct',
        });
        //Create an execution role for the lambda and attach to it a policy formed from user input
        this.assetUploadBucketRole = new aws_iam_1.Role(this, 'S3BucketDeploymentRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('lambda.amazonaws.com'),
            description: 'Role used by S3 deployment cdk construct',
            managedPolicies: [lambdaExecutionRolePolicy],
        });
        const stack = aws_cdk_lib_1.Stack.of(scope);
        this.podTemplatePolicy = new aws_iam_1.PolicyDocument({
            statements: [
                new aws_iam_1.PolicyStatement({
                    actions: [
                        's3:getObject',
                    ],
                    resources: [
                        stack.formatArn({
                            region: '',
                            account: '',
                            service: 's3',
                            resource: this.podTemplateLocation.bucketName,
                            resourceName: `${this.podTemplateLocation.objectKey}/*`,
                        }),
                    ],
                }),
            ],
        });
        if (this.defaultNodes) {
            (0, eks_karpenter_helpers_1.setDefaultKarpenterProvisioners)(this, karpenterVersion, this.ec2InstanceNodeGroupRole);
            // Upload the default podTemplate to the Amazon S3 asset bucket
            this.uploadPodTemplate('defaultPodTemplates', (0, path_1.join)(__dirname, 'resources/k8s/pod-template'), removalPolicy);
            // Replace the pod template location for driver and executor with the correct Amazon S3 path in the notebook default config
            NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] =
                this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
            this.podTemplateS3LocationNotebookDriver = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-driver.yaml`);
            NotebookDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] =
                this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
            this.podTemplateS3LocationNotebookExecutor = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/notebook-executor.yaml`);
            this.notebookDefaultConfig = JSON.parse(JSON.stringify(NotebookDefaultConfig));
            // Replace the pod template location for driver and executor with the correct Amazon S3 path in the critical default config
            CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] =
                this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
            this.podTemplateS3LocationCriticalDriver = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-driver.yaml`);
            CriticalDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] =
                this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
            this.podTemplateS3LocationCriticalExecutor = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/critical-executor.yaml`);
            this.criticalDefaultConfig = JSON.stringify(CriticalDefaultConfig);
            // Replace the pod template location for driver and executor with the correct Amazon S3 path in the shared default config
            SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.driver.podTemplateFile'] =
                this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
            this.podTemplateS3LocationDriverShared = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-driver.yaml`);
            SharedDefaultConfig.applicationConfiguration[0].properties['spark.kubernetes.executor.podTemplateFile'] =
                this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
            this.podTemplateS3LocationExecutorShared = this.assetBucket.s3UrlForObject(`${this.podTemplateLocation.objectKey}/shared-executor.yaml`);
            this.sharedDefaultConfig = JSON.stringify(SharedDefaultConfig);
        }
        // Tags the Amazon VPC and Subnets of the Amazon EKS Cluster
        aws_cdk_lib_1.Tags.of(eksCluster.vpc).add('for-use-with-amazon-emr-managed-policies', 'true');
        eksCluster.vpc.privateSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        eksCluster.vpc.publicSubnets.forEach((subnet) => aws_cdk_lib_1.Tags.of(subnet).add('for-use-with-amazon-emr-managed-policies', 'true'));
        // Create Amazon IAM ServiceLinkedRole for Amazon EMR and add to kubernetes configmap
        // required to add a dependency on the Amazon EMR virtual cluster
        if (this.createEmrOnEksServiceLinkedRole) {
            this.emrServiceRole = new aws_iam_1.CfnServiceLinkedRole(this, 'EmrServiceRole', {
                awsServiceName: 'emr-containers.amazonaws.com',
            });
        }
        eksCluster.awsAuth.addRoleMapping(aws_iam_1.Role.fromRoleArn(this, 'ServiceRoleForAmazonEMRContainers', `arn:aws:iam::${aws_cdk_lib_1.Stack.of(this).account}:role/AWSServiceRoleForAmazonEMRContainers`), {
            username: 'emr-containers',
            groups: [''],
        });
    }
    /**
     * Add a new Amazon EMR Virtual Cluster linked to Amazon EKS Cluster.
     * @param {Construct} scope of the stack where virtual cluster is deployed
     * @param {EmrVirtualClusterProps} options the EmrVirtualClusterProps [properties]{@link EmrVirtualClusterProps}
     */
    addEmrVirtualCluster(scope, options) {
        const eksNamespace = options.eksNamespace ?? 'default';
        let ns = undefined;
        if (options.createNamespace) {
            ns = (0, eks_cluster_helpers_1.createNamespace)(this.eksCluster, options.eksNamespace);
        }
        // deep clone the Role Binding template object and replace the namespace
        let manifest = utils_1.Utils.readYamlDocument(`${__dirname}/resources/k8s/rbac/emr-containers-rbac.yaml`);
        manifest = manifest.replace(/(\{{NAMESPACE}})/g, eksNamespace);
        let manfifestYAML = manifest.split('---').map((e) => utils_1.Utils.loadYaml(e));
        const manifestApply = this.eksCluster.addManifest(`emr-containers-rbac-${eksNamespace}`, ...manfifestYAML);
        if (ns) {
            manifestApply.node.addDependency(ns);
        }
        const virtualCluster = new aws_emrcontainers_1.CfnVirtualCluster(scope, `${options.name}VirtualCluster`, {
            name: options.name,
            containerProvider: {
                id: this.eksCluster.clusterName,
                type: 'EKS',
                info: { eksInfo: { namespace: options.eksNamespace ?? 'default' } },
            },
        });
        virtualCluster.node.addDependency(manifestApply);
        if (this.emrServiceRole) {
            manifestApply.node.addDependency(this.emrServiceRole);
            virtualCluster.node.addDependency(this.emrServiceRole);
        }
        if (ns) {
            virtualCluster.node.addDependency(ns);
        }
        return virtualCluster;
    }
    /**
     * Create and configure a new Amazon IAM Role usable as an execution role.
     * This method makes the created role assumed by the Amazon EKS cluster Open ID Connect provider.
     * @param {Construct} scope of the IAM role
     * @param {string} id of the CDK resource to be created, it should be unique across the stack
     * @param {IManagedPolicy} policy the execution policy to attach to the role
     * @param {string} eksNamespace The namespace from which the role is going to be used. MUST be the same as the namespace of the Virtual Cluster from which the job is submitted
     * @param {string} name Name to use for the role, required and is used to scope the iam role
     */
    createExecutionRole(scope, id, policy, eksNamespace, name) {
        let irsaConditionkey = new aws_cdk_lib_1.CfnJson(scope, `${id}IrsaConditionkey'`, {
            value: {
                [`${this.eksCluster.openIdConnectProvider.openIdConnectProviderIssuer}:sub`]: 'system:serviceaccount:' + eksNamespace + ':emr-containers-sa-*-*-' + aws_cdk_lib_1.Aws.ACCOUNT_ID.toString() + '-' + SimpleBase.base36.encode(name),
            },
        });
        // Create an execution role assumable by EKS OIDC provider
        return new aws_iam_1.Role(scope, `${id}ExecutionRole`, {
            assumedBy: new aws_iam_1.FederatedPrincipal(this.eksCluster.openIdConnectProvider.openIdConnectProviderArn, {
                StringLike: irsaConditionkey,
            }, 'sts:AssumeRoleWithWebIdentity'),
            roleName: name,
            managedPolicies: [policy],
            inlinePolicies: this.podTemplatePolicy ? { podTemplateAccess: this.podTemplatePolicy } : undefined,
        });
    }
    /**
     * Upload podTemplates to the Amazon S3 location used by the cluster.
     * @param {string} id the unique ID of the CDK resource
     * @param {string} filePath The local path of the yaml podTemplate files to upload
     */
    uploadPodTemplate(id, filePath, removalPolicy) {
        new aws_s3_deployment_1.BucketDeployment(this, `${id}AssetDeployment`, {
            destinationBucket: this.assetBucket,
            destinationKeyPrefix: this.podTemplateLocation.objectKey,
            sources: [aws_s3_deployment_1.Source.asset(filePath)],
            role: this.assetUploadBucketRole,
            retainOnDelete: removalPolicy === aws_cdk_lib_1.RemovalPolicy.RETAIN ? true : false,
        });
    }
    /**
     * Apply the provided manifest and add the CDK dependency on EKS cluster
     * @param {string} id the unique ID of the CDK resource
     * @param {any} manifest The manifest to apply.
     * You can use the Utils class that offers method to read yaml file and load it as a manifest
     */
    addKarpenterProvisioner(id, manifest) {
        let manifestApply = this.eksCluster.addManifest(id, ...manifest);
        if (this.karpenterChart) {
            manifestApply.node.addDependency(this.karpenterChart);
        }
        return manifestApply;
    }
}
_a = JSII_RTTI_SYMBOL_1;
SparkEmrContainersRuntime[_a] = { fqn: "aws-dsf.processing.SparkEmrContainersRuntime", version: "1.0.0-rc.3" };
SparkEmrContainersRuntime.DEFAULT_EMR_EKS_VERSION = emr_releases_1.EMR_DEFAULT_VERSION;
SparkEmrContainersRuntime.DEFAULT_EKS_VERSION = aws_eks_1.KubernetesVersion.V1_27;
SparkEmrContainersRuntime.DEFAULT_CLUSTER_NAME = 'data-platform';
SparkEmrContainersRuntime.DEFAULT_VPC_CIDR = '10.0.0.0/16';
exports.SparkEmrContainersRuntime = SparkEmrContainersRuntime;
//# sourceMappingURL=data:application/json;base64,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