import { ISecurityGroup, IVpcEndpoint } from 'aws-cdk-lib/aws-ec2';
import { CfnApplication } from 'aws-cdk-lib/aws-emrserverless';
import { PolicyDocument, IRole } from 'aws-cdk-lib/aws-iam';
import { Construct } from 'constructs';
import { SparkEmrServerlessRuntimeProps } from './spark-emr-runtime-serverless-props';
import { NetworkConfiguration, TrackedConstruct } from '../../../../utils';
/**
 * A construct to create a Spark EMR Serverless Application, along with methods to create IAM roles having the least privilege.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-emr-serverless-runtime
*/
export declare class SparkEmrServerlessRuntime extends TrackedConstruct {
    /**
       * A static method which will create an execution IAM role that can be assumed by EMR Serverless
       * The method returns the role it creates. If no `executionRolePolicyDocument` or `iamPolicyName`
       * The method will return a role with only a trust policy to EMR Servereless service principal.
       * You can use this role then to grant access to any resources you control.
       *
       * @param scope the scope in which to create the role
       * @param id passed to the IAM Role construct object
       * @param executionRolePolicyDocument the inline policy document to attach to the role. These are IAM policies needed by the job.
       * This parameter is mutually execlusive with iamPolicyName.
       * @param iamPolicyName the IAM policy name to attach to the role, this is mutually execlusive with executionRolePolicyDocument
       */
    static createExecutionRole(scope: Construct, id: string, executionRolePolicyDocument?: PolicyDocument, iamPolicyName?: string): IRole;
    /**
       * A static method which will grant an IAM Role the right to start and monitor a job.
       * The method will also attach an iam:PassRole permission limited to the IAM Job Execution roles passed
       *
       * @param startJobRole the role that will call the start job api and which needs to have the iam:PassRole permission
       * @param executionRoleArn the role used by EMR Serverless to access resources during the job execution
       * @param applicationArns the EMR Serverless aplication ARN,
       * this is used by the method to limit the EMR Serverless applications the role can submit job to.
       */
    static grantStartJobExecution(startJobRole: IRole, executionRoleArn: string[], applicationArns: string[]): void;
    /**
     * The EMR Serverless application
     */
    readonly application: CfnApplication;
    /**
     * If no VPC is provided, one is created by default along with a security group attached to the EMR Serverless Application
     * This attribute is used to expose the security group,
     * if you provide your own security group through the {@link SparkEmrServerlessRuntimeProps} the attribute will be `undefined`
     */
    readonly emrApplicationSecurityGroup?: ISecurityGroup;
    /**
     * If no VPC is provided, one is created by default
     * This attribute is used to expose the Gateway Vpc Endpoint for Amazon S3
     * The attribute will be undefined if you provided the `networkConfiguration` through the {@link SparkEmrServerlessRuntimeProps}
     */
    readonly s3GatewayVpcEndpoint?: IVpcEndpoint;
    /**
     * The EMR Serverless application network configuration including VPC, S3 interface endpoint and flow logs.
     */
    readonly networkConfiguration?: NetworkConfiguration;
    /**
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param props {@link SparkEmrServerlessRuntimeProps}
     */
    constructor(scope: Construct, id: string, props: SparkEmrServerlessRuntimeProps);
    /**
      * A method which will grant an IAM Role the right to start and monitor a job.
      * The method will also attach an iam:PassRole permission to limited to the IAM Job Execution roles passed.
      * The excution role will be able to submit job to the EMR Serverless application created by the construct.
      *
      * @param startJobRole the role that will call the start job api and which need to have the iam:PassRole permission
      * @param executionRoleArn the role use by EMR Serverless to access resources during the job execution
      */
    grantStartExecution(startJobRole: IRole, executionRoleArn: string): void;
}
