"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SparkEmrServerlessRuntime = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_ec2_1 = require("aws-cdk-lib/aws-ec2");
const aws_emrserverless_1 = require("aws-cdk-lib/aws-emrserverless");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const semver = require("semver");
const utils_1 = require("../../../../utils");
const emr_releases_1 = require("../../emr-releases");
/**
 * A construct to create a Spark EMR Serverless Application, along with methods to create IAM roles having the least privilege.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/spark-emr-serverless-runtime
*/
class SparkEmrServerlessRuntime extends utils_1.TrackedConstruct {
    /**
       * A static method which will create an execution IAM role that can be assumed by EMR Serverless
       * The method returns the role it creates. If no `executionRolePolicyDocument` or `iamPolicyName`
       * The method will return a role with only a trust policy to EMR Servereless service principal.
       * You can use this role then to grant access to any resources you control.
       *
       * @param scope the scope in which to create the role
       * @param id passed to the IAM Role construct object
       * @param executionRolePolicyDocument the inline policy document to attach to the role. These are IAM policies needed by the job.
       * This parameter is mutually execlusive with iamPolicyName.
       * @param iamPolicyName the IAM policy name to attach to the role, this is mutually execlusive with executionRolePolicyDocument
       */
    static createExecutionRole(scope, id, executionRolePolicyDocument, iamPolicyName) {
        if (executionRolePolicyDocument && iamPolicyName) {
            throw new Error('You must provide either executionRolePolicyDocument or iamPolicyName');
        }
        if (executionRolePolicyDocument) {
            //create an IAM role with emr-serverless as service pricinpal and return it
            return new aws_iam_1.Role(scope, id, {
                assumedBy: new aws_iam_1.ServicePrincipal('emr-serverless.amazonaws.com'),
                inlinePolicies: { executionRolePolicyDocument },
            });
        }
        if (iamPolicyName) {
            return new aws_iam_1.Role(scope, id, {
                assumedBy: new aws_iam_1.ServicePrincipal('emr-serverless.amazonaws.com'),
                managedPolicies: [aws_iam_1.ManagedPolicy.fromManagedPolicyName(scope, id, iamPolicyName)],
            });
        }
        return new aws_iam_1.Role(scope, id, {
            assumedBy: new aws_iam_1.ServicePrincipal('emr-serverless.amazonaws.com'),
        });
    }
    /**
       * A static method which will grant an IAM Role the right to start and monitor a job.
       * The method will also attach an iam:PassRole permission limited to the IAM Job Execution roles passed
       *
       * @param startJobRole the role that will call the start job api and which needs to have the iam:PassRole permission
       * @param executionRoleArn the role used by EMR Serverless to access resources during the job execution
       * @param applicationArns the EMR Serverless aplication ARN,
       * this is used by the method to limit the EMR Serverless applications the role can submit job to.
       */
    static grantStartJobExecution(startJobRole, executionRoleArn, applicationArns) {
        //method to validate if IAM Role ARN is valid or if a token do not fail
        //We need to test for CDK token in case the ARN is resolved at deployment time
        //This is done to fail early
        const regex_arn = /^arn:aws:iam::[0-9]{12}:role\/[a-zA-Z0-9-_]+$/;
        const regex_token = /^Token\[([0-9]+)\]$/;
        executionRoleArn.forEach(arn => {
            if (!regex_arn.test(arn) && regex_token.test(arn)) {
                throw new Error(`Invalid IAM Role ARN ${arn}`);
            }
        });
        startJobRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['iam:PassRole'],
            resources: executionRoleArn,
            conditions: {
                StringLike: {
                    'iam:PassedToService': 'emr-serverless.amazonaws.com',
                },
            },
        }));
        startJobRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'emr-serverless:StartApplication',
                'emr-serverless:StopApplication',
                'emr-serverless:StopJobRun',
                'emr-serverless:DescribeApplication',
                'emr-serverless:GetJobRun',
            ],
            resources: applicationArns,
        }));
        startJobRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'emr-serverless:StartJobRun',
            ],
            resources: applicationArns,
        }));
        startJobRole.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'emr-serverless:TagResource',
            ],
            resources: applicationArns,
        }));
    }
    /**
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param props {@link SparkEmrServerlessRuntimeProps}
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: SparkEmrServerlessRuntime.name,
        };
        super(scope, id, trackedConstructProps);
        const emrReleaseLabel = props.releaseLabel ? props.releaseLabel : emr_releases_1.EMR_DEFAULT_VERSION;
        const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props.removalPolicy);
        const releaseLabelSemver = emrReleaseLabel.split('-')[1];
        if (semver.lt(releaseLabelSemver, '6.9.0')) {
            throw new Error(`EMR Serverless supports release EMR 6.9 and above, provided release is ${emrReleaseLabel.toString()}`);
        }
        let emrNetworkConfiguration = undefined;
        if (!props.networkConfiguration) {
            const logKmsKey = new aws_kms_1.Key(scope, 'logKmsKey', {
                enableKeyRotation: true,
                alias: `flowlog-vpc-key-for-emr-application-${props.name}`,
                removalPolicy: removalPolicy,
                keyUsage: aws_kms_1.KeyUsage.ENCRYPT_DECRYPT,
                description: `Key used by the VPC created for EMR serverless application ${props.name}`,
            });
            this.networkConfiguration = (0, utils_1.vpcBootstrap)(scope, '10.0.0.0/16', logKmsKey, props.removalPolicy, undefined, props.name);
            let privateSubnetIds = [];
            this.networkConfiguration.vpc.privateSubnets.forEach(function (subnet) {
                privateSubnetIds.push(subnet.subnetId);
            });
            this.emrApplicationSecurityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', {
                vpc: this.networkConfiguration.vpc,
                description: `Security group used by emr serverless application ${props.name}`,
            });
            emrNetworkConfiguration = {
                securityGroupIds: [this.emrApplicationSecurityGroup.securityGroupId],
                subnetIds: privateSubnetIds,
            };
            this.s3GatewayVpcEndpoint = this.networkConfiguration.s3GatewayVpcEndpoint;
        }
        this.application = new aws_emrserverless_1.CfnApplication(scope, `spark-serverless-application-${props.name}`, {
            ...props,
            networkConfiguration: props.networkConfiguration ?? emrNetworkConfiguration,
            releaseLabel: emrReleaseLabel,
            type: 'Spark',
        });
    }
    /**
      * A method which will grant an IAM Role the right to start and monitor a job.
      * The method will also attach an iam:PassRole permission to limited to the IAM Job Execution roles passed.
      * The excution role will be able to submit job to the EMR Serverless application created by the construct.
      *
      * @param startJobRole the role that will call the start job api and which need to have the iam:PassRole permission
      * @param executionRoleArn the role use by EMR Serverless to access resources during the job execution
      */
    grantStartExecution(startJobRole, executionRoleArn) {
        SparkEmrServerlessRuntime.grantStartJobExecution(startJobRole, [executionRoleArn], [this.application.attrArn]);
    }
}
_a = JSII_RTTI_SYMBOL_1;
SparkEmrServerlessRuntime[_a] = { fqn: "aws-dsf.processing.SparkEmrServerlessRuntime", version: "1.0.0-rc.3" };
exports.SparkEmrServerlessRuntime = SparkEmrServerlessRuntime;
//# sourceMappingURL=data:application/json;base64,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