"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AnalyticsBucket = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const utils_1 = require("../../utils");
/**
* Amazon S3 Bucket configured with best-practices and defaults for analytics.
* The default bucket name is `analytics-<AWS_ACCOUNT_ID>-<AWS_REGION>-<UNIQUE_ID>`
* @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/analytics-bucket
*
* @example
* import { Key } from 'aws-cdk-lib/aws-kms';
*
* // Set context value for global data removal policy (or set in cdk.json).
* this.node.setContext('@data-solutions-framework-on-aws/removeDataOnDestroy', true);
*
* const encryptionKey = new Key(this, 'DataKey', {
*  removalPolicy: cdk.RemovalPolicy.DESTROY,
*  enableKeyRotation: true,
* });
*
* new dsf.storage.AnalyticsBucket(this, 'MyAnalyticsBucket', {
*  encryptionKey,
*  removalPolicy: cdk.RemovalPolicy.DESTROY,
* });
*/
class AnalyticsBucket extends aws_s3_1.Bucket {
    constructor(scope, id, props) {
        const bucketName = props.bucketName || utils_1.BucketUtils.generateUniqueBucketName(scope, id, 'analytics');
        const removalPolicy = utils_1.Context.revertRemovalPolicy(scope, props?.removalPolicy);
        const autoDeleteObjects = removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY;
        super(scope, id, {
            ...props,
            autoDeleteObjects,
            bucketName: bucketName,
            blockPublicAccess: props?.blockPublicAccess || aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            encryption: aws_s3_1.BucketEncryption.KMS,
            lifecycleRules: props?.lifecycleRules?.concat(AnalyticsBucket.LIFECYCLE_RULE) || AnalyticsBucket.LIFECYCLE_RULE,
            removalPolicy,
            serverAccessLogsPrefix: props?.serverAccessLogsPrefix || bucketName,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
AnalyticsBucket[_a] = { fqn: "aws-dsf.storage.AnalyticsBucket", version: "1.0.0-rc.3" };
AnalyticsBucket.LIFECYCLE_RULE = [{ abortIncompleteMultipartUploadAfter: aws_cdk_lib_1.Duration.days(1) }];
exports.AnalyticsBucket = AnalyticsBucket;
//# sourceMappingURL=data:application/json;base64,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