import { IKey } from 'aws-cdk-lib/aws-kms';
import { Construct } from 'constructs';
import { AccessLogsBucket } from './access-logs-bucket';
import { AnalyticsBucket } from './analytics-bucket';
import { DataLakeStorageProps } from './data-lake-storage-props';
import { TrackedConstruct } from '../../utils';
/**
 * Creates the storage layer for a data lake, composed of 3 {@link AnalyticsBucket} for Bronze, Silver, and Gold data.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/data-lake-storage
 *
 * @example
 * // Set the context value for global data removal policy
 * this.node.setContext('@data-solutions-framework-on-aws/removeDataOnDestroy', true);
 *
 * new dsf.storage.DataLakeStorage(this, 'MyDataLakeStorage', {
 *  bronzeBucketName: 'my-bronze',
 *  bronzeBucketInfrequentAccessDelay: 90,
 *  bronzeBucketArchiveDelay: 180,
 *  silverBucketName: 'my-silver',
 *  silverBucketInfrequentAccessDelay: 180,
 *  silverBucketArchiveDelay: 360,
 *  goldBucketName: 'my-gold',
 *  goldBucketInfrequentAccessDelay: 180,
 *  goldBucketArchiveDelay: 360,
 *  removalPolicy: cdk.RemovalPolicy.DESTROY,
 * });
 */
export declare class DataLakeStorage extends TrackedConstruct {
    /**
     * The S3 Bucket for Bronze layer.
     */
    readonly bronzeBucket: AnalyticsBucket;
    /**
     * The S3 Bucket for Silver layer.
     */
    readonly silverBucket: AnalyticsBucket;
    /**
     * The S3 Bucket for Gold layer.
     */
    readonly goldBucket: AnalyticsBucket;
    /**
     * The KMS Key used to encrypt all DataLakeStorage S3 buckets.
     */
    readonly dataLakeKey: IKey;
    /**
     * The S3 Bucket for access logs.
     */
    readonly accessLogsBucket: AccessLogsBucket;
    /**
     * Construct a new instance of DataLakeStorage
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataLakeStorageProps} props the DataLakeStorageProps properties
     */
    constructor(scope: Construct, id: string, props?: DataLakeStorageProps);
}
