"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataLakeStorage = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_kms_1 = require("aws-cdk-lib/aws-kms");
const aws_s3_1 = require("aws-cdk-lib/aws-s3");
const access_logs_bucket_1 = require("./access-logs-bucket");
const analytics_bucket_1 = require("./analytics-bucket");
const utils_1 = require("../../utils");
/**
 * Creates the storage layer for a data lake, composed of 3 {@link AnalyticsBucket} for Bronze, Silver, and Gold data.
 * @see https://awslabs.github.io/data-solutions-framework-on-aws/docs/constructs/library/data-lake-storage
 *
 * @example
 * // Set the context value for global data removal policy
 * this.node.setContext('@data-solutions-framework-on-aws/removeDataOnDestroy', true);
 *
 * new dsf.storage.DataLakeStorage(this, 'MyDataLakeStorage', {
 *  bronzeBucketName: 'my-bronze',
 *  bronzeBucketInfrequentAccessDelay: 90,
 *  bronzeBucketArchiveDelay: 180,
 *  silverBucketName: 'my-silver',
 *  silverBucketInfrequentAccessDelay: 180,
 *  silverBucketArchiveDelay: 360,
 *  goldBucketName: 'my-gold',
 *  goldBucketInfrequentAccessDelay: 180,
 *  goldBucketArchiveDelay: 360,
 *  removalPolicy: cdk.RemovalPolicy.DESTROY,
 * });
 */
class DataLakeStorage extends utils_1.TrackedConstruct {
    /**
     * Construct a new instance of DataLakeStorage
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {DataLakeStorageProps} props the DataLakeStorageProps properties
     */
    constructor(scope, id, props) {
        const trackedConstructProps = {
            trackingTag: DataLakeStorage.name,
        };
        super(scope, id, trackedConstructProps);
        this.accessLogsBucket = new access_logs_bucket_1.AccessLogsBucket(this, 'AccessLogsBucket', {
            removalPolicy: props?.removalPolicy,
        });
        const removalPolicy = utils_1.Context.revertRemovalPolicy(this, props?.removalPolicy);
        // Create the key if it not provided in the parameters
        this.dataLakeKey = props?.dataLakeKey || new aws_kms_1.Key(this, 'DataKey', {
            removalPolicy,
            enableKeyRotation: true,
        });
        // Prepare Amazon S3 Lifecycle Rules for bronze data
        const bronzeTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(props?.bronzeBucketInfrequentAccessDelay || 90),
            },
            {
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(props?.bronzeBucketArchiveDelay || 180),
            },
        ];
        // Create the bronze data bucket with the bronze transitions
        this.bronzeBucket = new analytics_bucket_1.AnalyticsBucket(this, 'BronzeBucket', {
            encryptionKey: this.dataLakeKey,
            bucketName: props?.bronzeBucketName || utils_1.BucketUtils.generateUniqueBucketName(this, 'BronzeBucket', 'bronze'),
            lifecycleRules: [
                {
                    transitions: bronzeTransitions,
                },
            ],
            removalPolicy,
            serverAccessLogsBucket: this.accessLogsBucket,
        });
        // Prepare Amazon S3 Lifecycle Rules for silver data
        const silverTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(props?.silverBucketInfrequentAccessDelay || 180),
            },
        ];
        if (props?.silverBucketArchiveDelay) {
            silverTransitions.push({
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(props?.silverBucketArchiveDelay),
            });
        }
        // Create the silver data bucket
        this.silverBucket = new analytics_bucket_1.AnalyticsBucket(this, 'SilverBucket', {
            encryptionKey: this.dataLakeKey,
            bucketName: props?.silverBucketName || utils_1.BucketUtils.generateUniqueBucketName(this, 'SilverBucket', 'silver'),
            lifecycleRules: [
                {
                    transitions: silverTransitions,
                },
            ],
            removalPolicy,
            serverAccessLogsBucket: this.accessLogsBucket,
        });
        // Prepare Amazon S3 Lifecycle Rules for silver data
        const goldTransitions = [
            {
                storageClass: aws_s3_1.StorageClass.INFREQUENT_ACCESS,
                transitionAfter: aws_cdk_lib_1.Duration.days(props?.goldBucketInfrequentAccessDelay || 180),
            },
        ];
        if (props?.goldBucketArchiveDelay) {
            goldTransitions.push({
                storageClass: aws_s3_1.StorageClass.GLACIER,
                transitionAfter: aws_cdk_lib_1.Duration.days(props?.goldBucketArchiveDelay),
            });
        }
        // Create the gold data bucket
        this.goldBucket = new analytics_bucket_1.AnalyticsBucket(this, 'GoldBucket', {
            encryptionKey: this.dataLakeKey,
            bucketName: props?.goldBucketName || utils_1.BucketUtils.generateUniqueBucketName(this, 'GoldBucket', 'gold'),
            lifecycleRules: [
                {
                    transitions: goldTransitions,
                },
            ],
            removalPolicy,
            serverAccessLogsBucket: this.accessLogsBucket,
        });
    }
}
_a = JSII_RTTI_SYMBOL_1;
DataLakeStorage[_a] = { fqn: "aws-dsf.storage.DataLakeStorage", version: "1.0.0-rc.3" };
exports.DataLakeStorage = DataLakeStorage;
//# sourceMappingURL=data:application/json;base64,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