"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BucketUtils = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
const crypto_1 = require("crypto");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Utils for working with Amazon S3 buckets.
 */
class BucketUtils {
    /**
     * Generate a unique Amazon S3 bucket name based on the provided name, CDK construct ID and CDK construct scope.
     * The bucket name is suffixed the AWS account ID, the AWS region and a unique 8 characters hash.
     * The maximum length for name is 26 characters.
     * @param name the name of the bucket
     * @param id the CDK ID of the construct
     * @param scope the current scope where the construct is created (generally `this`)
     * @returns the unique Name for the bucket
     */
    static generateUniqueBucketName(scope, id, name) {
        if (name.length > 26) {
            throw new Error('Bucket name is too long, maximum length for bucketName is 26');
        }
        return name + '-' + aws_cdk_lib_1.Aws.ACCOUNT_ID + '-' + aws_cdk_lib_1.Aws.REGION + '-' + BucketUtils.generateHash(scope, id);
    }
    /**
     * Generate an 8 characters hash of the CDK scope using its path.
     * @param scope the CDK construct scope
     * @returns the hash
     */
    static generateHash(scope, id) {
        const node = scope.node;
        const components = node.scopes.slice(1).map(c => c.node.id).join('-').concat(id);
        const secret = 'Data Solutions Framework on AWS';
        const hash = (0, crypto_1.createHmac)('sha256', secret)
            .update(components)
            .digest('hex')
            .slice(0, 8);
        return hash;
    }
}
_a = JSII_RTTI_SYMBOL_1;
BucketUtils[_a] = { fqn: "aws-dsf.utils.BucketUtils", version: "1.0.0-rc.3" };
exports.BucketUtils = BucketUtils;
//# sourceMappingURL=data:application/json;base64,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