"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextOptions = exports.Context = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * @internal
 * Utils class to work with the CDK context and options
 */
class Context {
    /**
     * Method to revert a DESTROY removal policy to RETAIN if the global removal policy parameter
     * in the CDK context is not set to true.
     * Also create a warning to warn the user if the retention policy has been reverted to RETAIN.
     * @param retentionPolicy The retention policy provided to the construct
     * @return the new retention policy based on the global retention parameter set in the CDK context
     */
    static revertRemovalPolicy(scope, removalPolicy) {
        const globalRemovalPolicy = scope.node.tryGetContext(ContextOptions.REMOVE_DATA_ON_DESTROY) || false;
        if (removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY && !globalRemovalPolicy) {
            aws_cdk_lib_1.Annotations.of(scope).addWarning(`WARNING: removalPolicy was reverted back to 'RemovalPolicy.RETAIN'.
        If you wish to set 'removalPolicy' to 'DESTROY' you must also
        set the global removal policy flag context variable in the 'cdk.json'
        or 'cdk.context.json': '@data-solutions-framework-on-aws/removeDataOnDestroy: true'`);
        }
        return removalPolicy == aws_cdk_lib_1.RemovalPolicy.DESTROY && globalRemovalPolicy ? aws_cdk_lib_1.RemovalPolicy.DESTROY : aws_cdk_lib_1.RemovalPolicy.RETAIN;
    }
}
exports.Context = Context;
/**
 * @internal
 * Options used in the CDK context
 */
var ContextOptions;
(function (ContextOptions) {
    ContextOptions["DISABLE_CONSTRUCTS_DEPLOYMENT_TRACKING"] = "@data-solutions-framework-on-aws/disableConstructsDeploymentTracking";
    ContextOptions["REMOVE_DATA_ON_DESTROY"] = "@data-solutions-framework-on-aws/removeDataOnDestroy";
})(ContextOptions = exports.ContextOptions || (exports.ContextOptions = {}));
//# sourceMappingURL=data:application/json;base64,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