"use strict";
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: MIT-0
Object.defineProperty(exports, "__esModule", { value: true });
exports.TrackedConstruct = void 0;
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const context_1 = require("./context");
const constants_1 = require("../../constants");
/**
 * @internal
 * A type of CDK Construct that is tracked via a unique code in CloudFormation Stack description.
 * It is used to measure the number of deployments.
 */
class TrackedConstruct extends constructs_1.Construct {
    /**
     * Constructs a new instance of the TrackedConstruct
     * @param {Construct} scope the Scope of the CDK Construct
     * @param {string} id the ID of the CDK Construct
     * @param {TrackedConstructProps} props the TrackedConstruct [properties] {TrackedConstructProps}
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!scope.node.tryGetContext(context_1.ContextOptions.DISABLE_CONSTRUCTS_DEPLOYMENT_TRACKING)) {
            const stack = aws_cdk_lib_1.Stack.of(this);
            const currentDescription = stack.templateOptions.description || '';
            stack.templateOptions.description = this.updateDescription(currentDescription, props);
        }
        aws_cdk_lib_1.Tags.of(scope).add(TrackedConstruct.DSF_OWNED_TAG, 'true');
    }
    updateDescription(currentDescription, props) {
        const fullDescription = TrackedConstruct.trackingRegExp.exec(currentDescription);
        const version = this.retrieveVersion();
        const tag = props.trackingTag.split(TrackedConstruct.TRACKING_TAG_SEPARATOR).join('_'); // make sure there's no separator in the tag name
        if (fullDescription == null) {
            return `${currentDescription} (${TrackedConstruct.DSF_TRACKING_CODE}) (version:${version}) (tag:${tag})`;
        }
        else {
            const description = fullDescription[1];
            const existingTags = fullDescription[5];
            let newTags;
            if (existingTags) {
                const tags = existingTags.split(TrackedConstruct.TRACKING_TAG_SEPARATOR);
                if (tags.includes(tag)) {
                    newTags = existingTags;
                }
                else {
                    newTags = existingTags + TrackedConstruct.TRACKING_TAG_SEPARATOR + tag;
                }
            }
            else {
                newTags = tag;
            }
            return `${description} (${TrackedConstruct.DSF_TRACKING_CODE}) (version:${version}) (tag:${newTags})`;
        }
    }
    /**
     * Retrieve DSF package.json version
     */
    retrieveVersion() {
        // We cannot import package.json as a module, because it's not at rootDir, so using direct JS require
        const file = '../../../package.json';
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const json = require(file);
        return json.version;
    }
}
TrackedConstruct.DSF_TRACKING_CODE = 'uksb-1tupboc21';
TrackedConstruct.DSF_OWNED_TAG = `${constants_1.DSF_AWS_TAG}:owned`;
/**
 * Format is "Description (uksb_12345abcde) (version:1.2.3) (tag:construct1,construct2)"
 */
TrackedConstruct.trackingRegExp = new RegExp('(.*) \\(' + TrackedConstruct.DSF_TRACKING_CODE + '\\)( \\(version:([^)]*)\\))?( \\(tag:([^)]*)\\))?');
TrackedConstruct.TRACKING_TAG_SEPARATOR = ',';
exports.TrackedConstruct = TrackedConstruct;
//# sourceMappingURL=data:application/json;base64,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